package code;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.concurrent.TimeoutException;

/**
 * This class is used to coordinate OutgoingConnectionWorker
 * with UpdateLog new update events and the OutgoingConnection new remove/add
 * subscription set requests.
 * 
 * It is used to fix the following issue in our previous design:
 * OutgoingConnectionWorker is a single-thread for three tasks:
 *   1. deal with addSubscription/removeSubscription requests
 *   2. send catchup streams
 *   3. sending next new invalidation by calling UpdateLog.getNext()
 *
 *   Currently, the possible blocking it has is:
 *    block at UpdateLog.getNext() due to no new updates which have a timeout
 *  
 *   During the blocking, the new coming addSub/removeSub requests can't be
 *   applied until it returns due to timeout if no new updates ever issued.
 *
 *
 *   Another issue with the worker is about Accumulating imprecise invalidation:
 *   currently, the accumulating is done in InvalIterator.
 *    Then it is possible that while InvalIterator is accumulating impreciseInv
 *   for a certain accumulatingTimeout period, there're new addSub/removeSub
 *   coming. Then what we will see is that the processing of addSub/removeSub
 *   has a huge delay.
 *
 *
 *  Solution:
 *
 *   add OutgoingConnectionEventMailbox to coordinate the worker and updateLog
 *   and outgoingConnection.
 *
 *   OutgoingConnectionEventMailbox:
 *
 *   1. block:
 *      the worker only wait on OutgoingConnectionEventMailbox.getNextEvent() if
 *      there's no pending requests or new updates.
 *
 *   2. notify:
 *      OutgoingConnection will notify mailbox when there's new request
 *      UpdateLog will notify mailbox when there's new updates.
 *
 *   move accumulating to Worker.
 *
 *   tbd: for now it is unbounded
 *   
 *   The add/remove subscription request has higher priority than the next
 *   invalidation to send in the main stream. Because if the receiver doesn't
 *   need any precise invalidation for a specific interest set or need any precise
 *   invalidation, we should reflect this request to invalidation stream as soon as
 *   possible so that we don't send unnecessary information.
 *   
 *   mailbox.getNext() will return either an add/remove subscription request or an invalidation
 *   to send.
 *   
 * @author zjiandan
 *
 */
public class OutgoingConnectionMailBox{
//for now unbounded
  
  LinkedList<SubscriptionRequest> requestQueue;
  Iterator<SingleWriterInval> ii;

  private boolean timeToDie;

  public
  OutgoingConnectionMailBox(Iterator<SingleWriterInval> ii){
    requestQueue = new LinkedList<SubscriptionRequest>();
    this.ii = ii;
    timeToDie = false;
  }

  /**
   * get the next add/remove subscription requests if any from requestQueue queue
   *     if not get the next invalidation from the invaliterator
   *     if both are empty, the thread is blocked until new items available or 
   *     waited for "timeout" ms. 
   *     
   * We need a timeout, because the caller might be the one that
   * need to accumulate imprecise invalidations and flush the impreciseInvalidations
   * periodically. If there's no timeout here, then the caller will be blocked
   * forever if no new events come up. Therefore the accumulated imprecise invalidation
   * will not be able to be flushed at the end of the period.
   * 
   * if timeout = 0 then it works the same as wait();
   * 
   * called by OutgoingConnectionWorker
   * 
   * return add/remove requests or nextInvalidate to send
   */
  synchronized public Object getNext(long timeout)
  throws TimeoutException{
    
    assert timeout >= 0;
    boolean noTimeOut = false;
    if(timeout == 0){
      noTimeOut = true;
    }
    long now = System.currentTimeMillis();
    long estimatedEndTime = now + timeout;
    
    Object gi = null;
    if(requestQueue.isEmpty()){
      gi = ii.next();
    }

    while (requestQueue.isEmpty() && (gi == null) && !timeToDie){
      try{
	if(!noTimeOut){
	  now = System.currentTimeMillis();
	  timeout = estimatedEndTime - now;
	
	  //note: it's weird that the Object.wait(long timeout) doesn't 
	  //      throw TimeoutException.
	  //      It seems that everyone who wants to use the timeout
	  //      has to write similar code to check whether it is 
	  //      woke up because of timeout or because of new updates and
	  //      then create his own TimeoutException so that the caller could
	  //      do some meaningful things to take care of timeout.
	  if(timeout <= 0){
	    throw new TimeoutException("No new invalidations or subscription requests.");
	  }
	  assert timeout > 0;
	  wait(timeout);
	}else{
	  wait();
	}
      }catch(InterruptedException e){
        //ignore
      }

      if(this.timeToDie){
          return null;
      }
      if(requestQueue.isEmpty()){
        gi =ii.next();//will grab UpdateLog lock
      }
    }
    //princem: I added the following lines
    if(this.timeToDie){
      return null;
    }

    if(!requestQueue.isEmpty()){
      return requestQueue.remove();
    }else{
      assert gi != null;
      return gi;
    }

  }

  synchronized public void timeToDie(){
    this.timeToDie = true;
    notifyAll();
  }

  //called by OutgoingConnectionPool
  synchronized public void addNewRequest(SubscriptionRequest newRequest){
     requestQueue.add(newRequest);
     notifyAll();
  }

  //called by UpdateLog
  //Note: don't put it under any lock.--> deadlock
  synchronized public void newUpdate(){
     notifyAll();
  }
  
  //used only for test 
  synchronized public int getRequestSize(){
	  return requestQueue.size();
  }

  
  public Iterator getIterator(){
    return ii;
  }

  synchronized public void clear(){
    requestQueue.clear();   
  }

}
