package code;
/* OutgoingConnectionPool
 *
 *   Container for OutgoingConnections of invalidation subscriptions.
 *   
 *
 * (C) Copyright -- See the file COPYRIGHT for additional details
 */

import java.util.Hashtable;
import java.util.Enumeration;

public class OutgoingConnectionPool
{
 
  final static boolean breakDownTimeForCallbacks = false;

  private Hashtable<NetAddr, OutgoingConnection> pool; // netAddr --> OutgoingConnection
  private Core core;
  private Controller controller;
  private boolean shutdown = false;
  
 /** 
 *  Constructor 
 **/ 
  public OutgoingConnectionPool(Core core, Controller controller){
    pool = new Hashtable<NetAddr, OutgoingConnection>();
    this.core = core;
    this.controller = controller;
  }

 /** 
 *  return the matching OutgoingConnection in pool 
 *   
 *  side effect: if (1) createOneIfNon is set to be true and  
 *                  (2) no existing OutgoingConnection for this netAddr, 
 *               then   create one and return it. 
 *               otherwise return null if createOneIfNon is false; 
 *  
 *  tbd: remove either receiverId or (receiverDNS, portInval)--redundant 
 *  
 **/ 
  synchronized private OutgoingConnection getOutgoingConnection(NodeId receiverId,
								String receiverDNS,
								int portInval, 
								boolean createOneIfNon){
    if(shutdown){
      return null;
    }
    NetAddr netAddr = new NetAddr(receiverDNS, portInval);
    OutgoingConnection oc = (OutgoingConnection)pool.get(netAddr);
    if(((oc == null)&&(createOneIfNon))||oc.getTimeToDie()){
      oc = null;
      oc = new OutgoingConnection(core, controller, 
                                  receiverId, receiverDNS, portInval);
      pool.put(netAddr, oc);
      
      
    } 
    return oc;
  }

 /** 
 *  remove from pool any matching OutgoingConnection 
 *  
 *  tbd: remove either receiverId or (receiverDNS, portInval)--redundant 
 *  
 **/ 
  synchronized private void removeOutgoingConnection(NodeId receiverId,
						     String receiverDNS,
						     int portInval){
    if(shutdown){
      return;
    }
    NetAddr netAddr = new NetAddr(receiverDNS, portInval);
    pool.remove(netAddr);
  }
  
 /** 
 *   add a subscriptionset to a connection 
 *        
 *      For simplicity: here we just use one connection per (sender, receiver); 
 *      tbd: for multiple connections can take the simple policy such as: 
 *          always the most advanced stream i.e. the one with largest prevVV. 
 *          if the all prevVV is much less than cvv, then create a new stream? 
 *    Note: multi-stream might not worthwhile, because of overhead and the bottleneck 
 *    might be accessing the log, and there's redundant OUT imprecise invalidations. 
 **/ 
  public void addToConnection(NodeId receiverId,
			      String receiverDNS,
			      int portInval,
			      SubscriptionSet ss,
			      AcceptVV startVV,
                              boolean sendBodiesWithCP,
			      boolean catchupWithCP){
    
    long start, end, totalTime;
    start = System.currentTimeMillis();
    boolean createOneIfNon = true;
    OutgoingConnection oc = getOutgoingConnection(receiverId, receiverDNS, portInval, createOneIfNon);
    if(breakDownTimeForCallbacks ){
      end = System.currentTimeMillis();
      
      totalTime = (end-start);
      
      Env.dprintln(breakDownTimeForCallbacks ,
		   "OutogingConnectionPool::getOutgoingConnection(" + receiverId + " " 
		   + ss.toString() + " takes "+ totalTime + "ms");
    }
    if(oc != null){
      start = System.currentTimeMillis();
      oc.addSubscriptionSet(ss, startVV, sendBodiesWithCP, catchupWithCP);
      if(breakDownTimeForCallbacks ){
	end = System.currentTimeMillis();
	
	totalTime = (end-start);
	
	Env.dprintln(breakDownTimeForCallbacks ,
		     "OutogingConnectionPool::getOutgoingConnection(" + receiverId + " " 
		     + ss.toString() + " takes "+ totalTime + "ms");
      }
    }else{
      assert shutdown;
    }
  }

 /** 
 *  remove a subscriptionset from a connection 
 **/ 
  public void removeFromConnection(NodeId receiverId,
				   String receiverDNS,
				   int portInval,
				   SubscriptionSet ss){
    
    boolean createOneIfNon = false;
    OutgoingConnection oc = getOutgoingConnection(receiverId, receiverDNS, 
						  portInval, createOneIfNon);
   
    if (oc != null){
      oc.removeSubscriptionSet(ss);
    }
    
  }

 /** 
 *  remove a connection from the pool 
 **/ 
  public void removeConnection(NodeId receiverId,
			       String receiverDNS,
			       int portInval){
    
    
    boolean createOneIfNon = false;
    OutgoingConnection oc = getOutgoingConnection(receiverId, receiverDNS, 
						  portInval, createOneIfNon);
    
    
    if (oc != null){
      oc.close();
      removeOutgoingConnection(receiverId, receiverDNS, portInval);
    }
  }

  public int size(){
    return pool.size();
  }
 /** 
 *  shutdown all connections  
 **/ 
  synchronized void shutdownAll(){
    for (Enumeration<NetAddr> e = pool.keys(); e.hasMoreElements();){
      NetAddr key = (NetAddr)e.nextElement();
      OutgoingConnection oc = (OutgoingConnection)(pool.get(key));
      assert oc != null;
      oc.close();
      oc = null;
      pool.remove(key);
      
    }
  
    shutdown = true;
  }
}

//---------------------------------------------------------------------------
/* $Log: OutgoingConnectionPool.java,v $
/* Revision 1.8  2007/10/06 18:44:06  zjiandan
/* *** empty log message ***
/*
/* Revision 1.7  2007/09/12 01:55:54  zjiandan
/* gc dead OCs
/*
/* Revision 1.6  2007/06/25 05:21:29  zjiandan
/* Cleanup OutgoingConnection and add unit tests
/*
/* Revision 1.5  2007/06/04 21:40:59  zjiandan
/* expose stream catchup type CP|LOG option to rmiClient.subscribeInval().
/*
/* Revision 1.4  2007/05/30 20:30:19  dahlin
/* Added checkpoint to SubscribeBWUnit. Changed outgoingconnection checkpoint 
/* send to not send body by default. Told barrier to close sockets when done with them.
/*
/* Revision 1.3  2007/04/02 21:11:38  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.2  2006/08/16 21:26:18  dahlin
/* PicShareUnit test works on linux, too!
/*
/* Revision 1.1  2006/04/20 03:52:53  zjiandan
/* Callbacks merged with runTime.
/* */
//---------------------------------------------------------------------------
