package code;
/* OutgoingInvalConnectionPool
 *
 *   Container for OutgoingInvalConnections of invalidation subscriptions.
 *   
 *
 * (C) Copyright -- See the file COPYRIGHT for additional details
 */


import java.util.HashMap;
import java.util.Iterator;

public class OutgoingInvalConnectionPool
{
 
  final static boolean breakDownTimeForCallbacks = false;

  protected HashMap<NetAddr, OutgoingInvalConnection> pool; // netAddr --> OutgoingInvalConnection
  protected Core core;
  protected Controller controller;
  protected boolean shutdown = false;
  
 /** 
 *  Constructor 
 **/ 
  public OutgoingInvalConnectionPool(Core core, Controller controller){
    pool = new HashMap<NetAddr, OutgoingInvalConnection>();
    this.core = core;
    this.controller = controller;
  }

 /** 
 *  return the matching OutgoingInvalConnection in pool 
 *  return null if not exists. 
 **/ 
  synchronized private OutgoingInvalConnection 
  getOutgoingInvalConnection(String receiverDNS, int portInval){
    
    NetAddr netAddr = new NetAddr(receiverDNS, portInval);
    OutgoingInvalConnection oc = (OutgoingInvalConnection)pool.get(netAddr);
    return oc;
  }
  
 /** 
 *   add a subscriptionset to a connection 
 *        
 *      For simplicity: here we just use one connection per (sender, receiver); 
 *      tbd: for multiple connections can take the simple policy such as: 
 *          always the most advanced stream i.e. the one with largest prevVV. 
 *          if the all prevVV is much less than cvv, then create a new stream? 
 *    Note: multi-stream might not worthwhile, because of overhead and the bottleneck 
 *    might be accessing the log, and there's redundant OUT imprecise invalidations. 
 *  return the matching OutgoingInvalConnection in pool 
 *  create one if not exists yet. 
 *  
 **/ 
  synchronized public AcceptVV addToConnection(NodeId receiverId,
                                           String receiverDNS,
                                           int portInval,
                                           SubscriptionSet newSS,
                                           AcceptVV newStartVV,
                                           boolean sendBodiesWithCP,
                                           boolean catchupWithCP, 
                                           boolean includeAll){
    AcceptVV returnVV = null;
    if(shutdown){
      return returnVV;
    }
    SubscriptionRequest newRequest = 
      new SubscriptionRequest(catchupWithCP?SubscriptionRequest.CP:SubscriptionRequest.LOG,
                              newSS, 
                              newStartVV,
                              sendBodiesWithCP/*, 
                              includeAll*/);
    NetAddr netAddr = new NetAddr(receiverDNS, portInval);
    OutgoingInvalConnection oc = (OutgoingInvalConnection)pool.get(netAddr);
    if(oc == null|| oc.getWorker().getTimeToDie()){
      OutgoingConnectionMailBox box = core.makeOutgoingConnectionMailBox(core.getCurrentVV());
      //box.addNewRequest(newRequest); -- handled in the initialization
      OutgoingInvalConnectionWorker worker = null;
      worker = new OutgoingInvalConnectionWorker(core, controller, 
						 receiverId, receiverDNS, portInval,
						 newRequest,
						 box);
      returnVV = worker.getStreamCVV();
      oc = new OutgoingInvalConnection(box, worker);
      pool.put(netAddr, oc);
      new Thread(worker).start();
    }else{
      assert(!oc.getWorker().getTimeToDie());
      oc.getMailBox().addNewRequest(newRequest);
    }
    return returnVV;
  }
  
  
  /** 
   *   add a subscriptionset to a connection 
   *        
   *      For simplicity: here we just use one connection per (sender, receiver); 
   *      tbd: for multiple connections can take the simple policy such as: 
   *          always the most advanced stream i.e. the one with largest prevVV. 
   *          if the all prevVV is much less than cvv, then create a new stream? 
   *    Note: multi-stream might not worthwhile, because of overhead and the bottleneck 
   *    might be accessing the log, and there's redundant OUT imprecise invalidations. 
   *  return the matching OutgoingInvalConnection in pool 
   *  create one if not exists yet. 
   *  
   **/ 
    synchronized public AcceptVV addToConnection(NodeId receiverId,
                                             String receiverDNS,
                                             int portInval,
                                             SubscriptionSet newSS,
                                             AcceptVV newStartVV,
                                             boolean sendBodiesWithCP,
                                             boolean catchupWithCP){
      return this.addToConnection(receiverId, receiverDNS, portInval, newSS, newStartVV, sendBodiesWithCP, catchupWithCP, false);
    }

 /** 
 *  remove from pool any matching OutgoingInvalConnection 
 *  
 *  tbd: remove either receiverId or (receiverDNS, portInval)--redundant 
 *  
 **/ 
  synchronized public void removeConnection(NodeId receiverId,
				            String receiverDNS,
				            int portInval){
    if(shutdown){
      return;
    }
    NetAddr netAddr = new NetAddr(receiverDNS, portInval);
   
    OutgoingInvalConnection oc = pool.remove(netAddr);
    System.out.println("Removing connection");

    System.out.println("receiverId : " + receiverId);
    System.out.println("receiverDNS : " + receiverDNS);
    System.out.println("receiverId : " + portInval);


    if(oc != null){
      System.out.println("Removing connection!!");
        oc.close();
    }
  }
  
  
  
 /** 
 *  remove a subscriptionset from a connection 
 **/ 
  public synchronized void removeFromConnection(NodeId receiverId,
                                                String receiverDNS,
                                                int portInval,
                                                SubscriptionSet ss){
    if(shutdown){
      return;
    }
    OutgoingInvalConnection oc = getOutgoingInvalConnection(receiverDNS, 
						            portInval);
   
    if (oc != null){
      if(oc.getWorker().getTimeToDie()){
        new Thread(oc.getWorker()).start();
      }
      oc.getMailBox().addNewRequest(new SubscriptionRequest(SubscriptionRequest.REMOVE, 
                                                            ss, 
                                                            null, 
                                                            false/*, 
                                                            false*/));
    }
    
  }


  public int size(){
    return pool.size();
  }

 /** 
 *  shutdown all connections  
 **/ 
  synchronized void shutdownAll(){
    for (Iterator iter = pool.values().iterator(); iter.hasNext();){
      OutgoingInvalConnection oc = (OutgoingInvalConnection)iter.next();
      assert oc != null;
      oc.close();
      core.removeOutgoingConnectionMailBox(oc.getMailBox());
      oc = null;
    }
    pool.clear();
    shutdown = true;
  }
}


//---------------------------------------------------------------------------
/* $Log$ */
//---------------------------------------------------------------------------
