package code;
/* OutgoingInvalConnectionWorker
 *
 *   A dynamic outgoing connection for subscribing invalidation at the sender side.
 *   
 *   The difference between this OutgoingInvalConnectionWorker and OutgoingConnection:
 *   
 *       - For cp catchup, it does not need to defer serving the request if 
 *         the core.cvv > connection.prevVV,
 *         Instead, it sends the CP for the newSubSet from newStartVV to core.cvv
 *         and the CP for the stream.subset from prevVV to core.cvv.
 *         
 *         This seems neat because it solves the racing problem while won't knock
 *         out the original attached subscriptionset and make progress.
 *         
 *       - Accumulate the imprecise invalidation at the OutgoingInvalConnectionWorker 
 *         rather inside the invalIterator.
 *
 * (C) Copyright -- See the file COPYRIGHT for additional details
 */

import java.net.Socket;
import java.util.concurrent.TimeoutException;
import java.io.IOException;
import java.io.OutputStream;
import java.io.FileOutputStream;

import code.security.SangminConfig;
import code.security.SecureInv;

public class OutgoingInvalConnectionWorker implements Runnable, SocketMagicConstant{

  public static final boolean ACCUMULATED_PRECISE_CONVERT2_IMPRECISE = true;

  private OutgoingConnectionMailBox box;
  private boolean timeToDie;//true if close() called



  protected Core core;
  protected Controller controller;
  protected NodeId receiverId;
  protected String receiverDNS;
  protected int portInval;
  protected SubscriptionSet streamSS = null;
  protected CounterVV streamCVV = null;

  protected SubscriptionRequest initialRequest = null;
  private FileOutputStream fos = null;
  protected Socket sock = null;

  private long accumulateTimeOut;

  private long accumulateEndTime = Long.MAX_VALUE;//only when accumulatedInv becomes non null
  //do we set the timeout.
  private GeneralInv accumulatedInv = null;
  private long timeoutForInv = 0;//no timeout; i.e. timeout = infinity

  //file name to store output objects -- for unit tests and dbgs only
  //name format: OutgoingInvalConnection_SenderId_receiverId.out"
  public final static String outFilePrefix = "OutgoingInvalConnection_";




  protected final static String dbgTag = "DBG OutgoingInvalConnectionWorker:: ";
  protected static boolean dbgProgress = false;
  protected static boolean measureTime = false;
  protected static boolean printStream = false;
  protected static boolean dbgWithFileOutputStream = false;

  public final static boolean USE_HEARTBEATS = false;
  public final static boolean USE_BW_CONSTRAINTS = true;

  protected final boolean autoRemoveIfEmptySubset = false;
  // by default automatically remove empty connection

 /** 
 *  Constructor 
 **/ 
  public OutgoingInvalConnectionWorker(Core core, 
      Controller controller,
      NodeId receiverId,
      String receiverDNS,
      int portInval,
      SubscriptionRequest newRequest,
      OutgoingConnectionMailBox box){
    if(dbgProgress){
      System.out.println("OutgoingInvalConnectionWorker(rid= " + receiverId 
          + ") is created"); 
    }
    this.core = core;
    this.controller = controller;
    this.receiverId = receiverId;
    this.receiverDNS = receiverDNS;
    this.portInval = portInval;
    this.box = box;

    this.timeToDie = false;
    this.streamSS = SubscriptionSet.makeEmptySet();

    //streamSS = SubscriptionSet.makeSubscriptionSet("/never");
    /*
    if(streamSS.isEmpty()){
      // can't make the initial ss empty(ii.getNext() will complain), 
      // use a dumb subscriptoinset "/never", which will never overlap any interest sets.
      streamSS = SubscriptionSet.makeSubscriptionSet("/never");
    }
     */
    this.streamCVV = new CounterVV(core.getCurrentVV());
    this.accumulateTimeOut = core.getMaxAccumulateTime();
    this.initialRequest = newRequest;
    if(dbgProgress){
      System.out.println(" OutgoingInvalConnection.initialRequest: " + initialRequest);
    }
    Env.performanceWarning("OutgoingInvalConnectionWorker:: might delay for " + 
        accumulateTimeOut +
        " ms when it accumulates outside invalidations" + 
    "and there's no new inside invalidations to send.");
    
  }



 /** 
 *  Decide whether use Checkpoint to catch up the stream 
 *        
 *  simple policy: use Checkpoint only if there's something newer than startVV is omitted 
 *                 return true, if the omivv is larger than the startvv 
 *                 return false, otherwise 
 **/ 
  private boolean useCP(SubscriptionSet ss, VV startVV){   
    AcceptVV currentOmitVV = core.getInMemOmitVV();
    if (currentOmitVV.includesAnyPartOf(startVV)){//log has missing information
      return true;
    } else {
      return false;
    }
  }


 /** 
 * set timeToDie so that the thread would stop at some point 
 **/ 
  public void die(){
    if(dbgProgress){
      System.out.println("OutgoingInvalConnectionWorker.die() is called"); 
    }
    timeToDie = true;
    box.timeToDie();
  }


 /** 
 * get timeToDie 
 **/ 
  public boolean getTimeToDie(){
    return timeToDie;
  }


  protected void addToStreamSS(SubscriptionSet newSS){
    streamSS = streamSS.getCompleteUnion(newSS);
    if(dbgProgress){
      Env.dprintln(dbgProgress, 
          "streamSS.addSub(" + newSS);
    }
    controller.informOutgoingSubscribeInvalInitiated(receiverId,
        newSS);
  }

  private void removeFromStreamSS(SubscriptionSet removeSS){
    try{
      streamSS = streamSS.remove(removeSS, true);
    }catch(IllegalRemoveSubscriptionSetException e){

      assert false : (" can't remove subscriptionSet " + removeSS 
          + " from "+ streamSS + ":" + e 
          + " Are you trying to remove /a/b from /a/*?"
          + " Try to remove /a/b from /a/b:/a/c:/a/d ...");
    }

    controller.informOutgoingSubscribeInvalTerminated(receiverId,
        removeSS);
  }

//  //-------------------------------------------------------------------------------------------
//  // send the checkpoint + CP_END
//  //-------------------------------------------------------------------------------------------
//  protected void sendCP(TaggedOutputStream tos, 
//      SubscriptionSet subset, 
//      AcceptVV startvv, 
//      boolean withBody, 
//      boolean includeAll)
//  throws IOException{
//
//    assert false;
//    this.sendCP(tos, subset, startvv, withBody, false);
//
//  }
  
  //-------------------------------------------------------------------------------------------
  // send the checkpoint + CP_END
  //-------------------------------------------------------------------------------------------
  protected void sendCP(TaggedOutputStream tos, 
      SubscriptionSet subset, 
      AcceptVV startvv, 
      boolean withBody/*, 
      boolean includeAll*/)
  throws IOException{

    long start = 0, end = 0;
    if(measureTime){
      start = System.currentTimeMillis();
    }

    //note: 
    // the checkpoint apply needs to be careful,
    // the streamSS in the sender side might not match with the attachedSS
    // at the receiver side. 
    // we shouldn't kickoff the entire streamSS if at the end some of 
    // object under streamSS doesn't catchup. Instead we need to attach
    // these objects that are attachable, i.e. don't treat the cp ss
    // as one attach unit. Then the availability might be very bad.
    if(dbgProgress){
      System.out.println("OutgoingInvalConnection.sendCP:( " + subset + ", " + startvv 
          + " streamSS= " + subset //streamSS.getIntersection(subset) 
          + " streamCVV= " + streamCVV + " ) starts.");
          
    }
    AcceptVV cpEndVV = core.sendCheckpoint(tos, subset, startvv, withBody, 
        subset,
//        streamSS.getIntersection(subset), 
        streamCVV.cloneAcceptVV()/*, 
        includeAll*/);

    streamCVV.advanceTimestamps(cpEndVV); 
    if(dbgProgress){
      System.out.println("OutgoingInvalConnection.sendCP:( " + subset + ", " + startvv 
          + ") finished."); 
    }

    if(measureTime){
      end = System.currentTimeMillis();
      LatencyWatcher.put("Core.sendCP", (end-start));
    }

  }



  //-------------------------------------------------------------------------------------------
  // send an invalidate stream generated by an CatchupInvalIterator
  //
  //-------------------------------------------------------------------------------------------
  private void sendInvals(TaggedOutputStream tos, InMemLogIterator ii, AcceptVV endVV,
      SubscriptionSet catchupSS)
  throws IOException{

    long start = 0, end;

    if(dbgProgress){
      System.out.println("OutgoingInvalConnection.sendInvals() -- for catch up stream is called"); 
    }
    if(measureTime){
      start = System.currentTimeMillis();
    }

    assert ii != null;
    assert core.getCurrentVV().includes(endVV);
    assert endVV.includes(ii.getCurrentVV());
    Object nextObj = null;

    while(!ii.getCurrentVV().includes(endVV)){
      if(timeToDie){
        throw new IOException(" OutgoingInvalConnectionWorker shutdown with possible incomplete catchupStream.");
      }

      if(measureTime){
        end = System.currentTimeMillis();
        LatencyWatcher.put("sendInvals-preGetNext", (end-start));
        start = System.currentTimeMillis();
      }

      nextObj = ii.next();
      if(measureTime){
        end = System.currentTimeMillis();
        LatencyWatcher.put("sendInvals-GetNext", (end-start));
        start = System.currentTimeMillis();
      }


      if(nextObj instanceof GeneralInv){
        GeneralInv gi = (GeneralInv)nextObj;


        if (gi.getEndVV().isAnyPartGreaterThan(endVV)){//a component has catch up  
          assert false;//impossible because endVV is some node.cvv at some point
        }
        if(measureTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("postGetNext2-1", (end-start));
          start = System.currentTimeMillis();
        }
        assert gi != null;
        if(gi.getInvalTarget().intersects(catchupSS)){
          writeInvToStream(tos, gi);
          if(dbgProgress || printStream){
            long time = System.currentTimeMillis();
            System.out.println("OutgoingInvalConnection-"+ receiverId + ".writeObject(gi) for catchup (" 
                + streamSS + "): " 
                + nextObj + " at " + time); 
          }
        }
        
        if(measureTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("postGetNext2-2", (end-start));
          start = System.currentTimeMillis();
        }
      }else{
        assert ((nextObj == null)&&(ii.getCurrentVV().includes(endVV)));
        break;
      }

    }

  } 

  private static boolean warnedHB = false;
  private static boolean printedTBD = false;
  //--------------------------------------------------------------------------
  // Make connection with remote node.
  // Side effect -- sets fos and sock, which are global
  // so that we can clean them up later.
  //--------------------------------------------------------------------------
  private TaggedOutputStream makeConnection() throws IOException{
    TaggedOutputStream tos = null;

    if(dbgProgress){
      Env.dprintln(dbgProgress, "OutgoingInvalConnection::makeConnection-entered");
    }
    if(dbgWithFileOutputStream) {
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingInvalConnection::makeConnection-else to " + 
            outFilePrefix + core.getMyNodeId() 
            + "_" + receiverId + ".out");
      }
      try{
        fos = new FileOutputStream(outFilePrefix + core.getMyNodeId() 
            + "_" + receiverId + ".out");
        tos = new TaggedOutputStream(fos);
      }catch(Exception e){
        e.printStackTrace();
        assert false;
        return null;
      }

    }else{
      OutputStream os = null;
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingInvalConnection::makeConnection-else to " + 
            receiverDNS + ":" + portInval);
      }
      if(USE_HEARTBEATS){
        sock = new HeartbeatSocket(receiverDNS, portInval);
        if(!printedTBD){
          Env.tbd(" cleanly shutdown HeartbeatOutputStream");
          printedTBD = true;
        }
      }else{
        if(!warnedHB){
          Env.warn("OutgoingInvalConnectionWorker -- heartbeats turned off");
          warnedHB = true;
        }
        sock = new Socket(receiverDNS, portInval);
      }
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingInvalConnection::makeConnection-sock created");
      }
      sock.setTcpNoDelay(true);
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingInvalConnection::makeConnection-sock nodelay");
      }
      if(USE_BW_CONSTRAINTS){
        os = new ConstrainedOutputStream(sock.getOutputStream(),
            Config.getBandwidth(core.getMyNodeId()));
      }else{
        Env.warn("OutgoingInvalConnectionWorker -- BW constraints turned off");
        os = sock.getOutputStream();
      }
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingInvalConnection::makeConnection-got os");
      }
      tos = new TaggedOutputStream(os);
      if(dbgProgress){
        Env.dprintln(dbgProgress, "OutgoingInvalConnection::makeConnection-got tos");
      }
    }
    return tos;
  }


  protected void handshakeHelper(TaggedOutputStream tos) throws IOException{
    tos.writeObject(new Long(INVAL_SOCKET_MAGIC));
    if(dbgProgress || printStream){
      System.out.println("OutgoingInvalConnection-" + receiverId + ".writeObject(INVAL_SOCKET_MAGIC): " 
          + new Long(INVAL_SOCKET_MAGIC)); 
    }

    tos.writeObject(core.getMyNodeId());
    if(dbgProgress || printStream){
      System.out.println("OutgoingInvalConnection-" + receiverId + ".writeObject(myNodeId): " 
          + core.getMyNodeId()); 
    }

    /*
    if(initialRequest.getType() != SubscriptionRequest.CP){
        Env.warn(" initial request is not a checkpoint catchup, but we send a catchup checkpoint anyhow.");
        initialRequest = new SubscriptionRequest(SubscriptionRequest.CP,
                initialRequest.getSubscriptionSet(), 
                initialRequest.getStartVV(),
                initialRequest.getCPBodies());
        
    }
    assert initialRequest.getType() == SubscriptionRequest.CP;
    */
//    tos.writeObject(initialRequest.getStartVV().cloneMinVV(core.getCurrentVV()));//synchronize prevVV with receiver
    
    tos.writeObject(initialRequest.getStartVV().cloneMinVVOmit(this.streamCVV));//synchronize prevVV with receiver
    if(dbgProgress || printStream){
      System.out.println("OutgoingInvalConnection-" + receiverId + ".writeObject(initial endVV): " 
                         + initialRequest.getStartVV().cloneMinVVOmit(streamCVV)); 
    }
    
    tos.writeObject(initialRequest.getSubscriptionSet());//synchronize prevVV with receiver
    
  }
  
  //--------------------------------------------------------------------------
  // Send the handshake to start the connection.
  //--------------------------------------------------------------------------
  protected void handshake(TaggedOutputStream tos) throws IOException{

   
    handshakeHelper(tos);
    
    this.serveNewRequest(tos, initialRequest);
    //tos.writeObject(streamCVV.cloneAcceptVV());//synchronize prevVV with receiver
    //if(dbgProgress || printStream){
    //  System.out.println("OutgoingInvalConnection-" + receiverId + ".writeObject(initial endVV): " 
    //      + streamCVV); 
    //}
    
    
    //
    // fall back with the original protocol:
    // initialize with sender.cvv and emptySubscriptionSet
    //
    // 
    //tos.writeObject(streamSS);//synchronize streamSS with receiver
    //if(dbgProgress || printStream){
    //  System.out.println("OutgoingInvalConnection-" + receiverId + ".writeObject(initial streamSS): " 
    //                     + streamSS); 
    //}

  }


 /** 
 *  For a LOG request, send the events from startVV needed to catch up 
 *  Return true if "gap" has been filled by sending a catchup stream 
 *  (or true if no gap existed so no catchup stream was needed.) 
 *  
 *  Return false (and add a CP request in our place) if we were unable to fill  
 *  the gap. 
 **/ 
  private boolean sendCatchupStream(SubscriptionRequest nextRequest, 
      TaggedOutputStream tos) 
  throws IOException{
    if(useCP(nextRequest.getSubscriptionSet(),
        nextRequest.getStartVV())){
      return false;//sendCP instead
    }


    long start = 0, end;
    if(measureTime){
      start = System.currentTimeMillis();
    }
    //
    // notify receiver that catchup stream is starting
    //
    CatchupStreamStartMsg cssm = new CatchupStreamStartMsg(nextRequest.getSubscriptionSet(), 
        nextRequest.getStartVV());
    tos.writeObject(cssm);

    if(dbgProgress || printStream){
      long time = System.currentTimeMillis();
      System.out.println("OutgoingInvalConnectionWorker-" + receiverId + ".writeObject(CatchupStreamStartMsg): " 
          + cssm + " done sending at " + time); 
    }

    if(measureTime){
      end = System.currentTimeMillis();
      LatencyWatcher.put("sendCatchupStream-sendstartMsg", (end-start));
      start = System.currentTimeMillis();
    }

    //
    // Prepare and send the catchup invalIterator (if needed)
    //
    //System.out.println("------check log catchup startVV:" +nextRequest.getStartVV() + " with streamCVV:"+streamCVV);
    if(!nextRequest.getStartVV().includes(streamCVV)){
      if(dbgProgress){
        Env.dprintln(dbgProgress, 
            "going to catch up the stream for "+ nextRequest.getSubscriptionSet() 
            + "from "+ nextRequest.getStartVV() + "to" + streamCVV);
      }

      InMemLogIterator catchupIterator = null;
      try{
        // 
        // make sure that the catch up stream never sends any invalidate that
        // is not coverred by prevVV, otherwise it might kick off all the existing
        // subscriptionSet of the incomming connection.
        // Be careful about the mismatch component in startVV and prevVV
        //
        CounterVV catchupStart = new CounterVV(nextRequest.getStartVV());
        catchupStart.setToMinVV(streamCVV);
        if(measureTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("sendCatchupStream-preInval1", (end-start));
          start = System.currentTimeMillis();
        }
        if(dbgProgress){
          Env.dprintln(dbgProgress, 
              "going to catch up the stream for "+ nextRequest.getSubscriptionSet() 
              + "from "+ nextRequest.getStartVV() + "to" + catchupStart);
        }
        catchupIterator = core.makeInMemLogIterator(catchupStart.cloneAcceptVV());//used for catchup
        if(measureTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("sendCatchupStream-preInval2", (end-start));
          start = System.currentTimeMillis();
        }

        sendInvals(tos, catchupIterator, streamCVV.cloneAcceptVV(), nextRequest.getSubscriptionSet());
        if(measureTime){
          end = System.currentTimeMillis();
          LatencyWatcher.put("sendCatchupStream-sendInv", (end-start));
          start = System.currentTimeMillis();
        }

      }finally{
        if(catchupIterator != null){
          this.core.removeInMemLogIterator(catchupIterator);
          catchupIterator = null;
        }
      }



      if(measureTime){
        end = System.currentTimeMillis();
        LatencyWatcher.put("sendCatchupStream-postSendInv", (end-start));
        start = System.currentTimeMillis();
      }
    }


    //
    // Whether we succeeded or failed, send CatchupStreamEndMsg
    // Notify the receiver that the catch up stream has been stopped
    //
    CatchupStreamEndMsg csem = new CatchupStreamEndMsg();
    tos.writeObject(csem);
    if(dbgProgress || printStream){
      long time = System.currentTimeMillis();
      System.out.println("OutgoingInvalConnection-" + receiverId + ".writeObject(CatchupStreamEndMsg): " 
          + csem + " done sending at " + time + " streamCVV" + streamCVV); 
    }

    if(measureTime){
      end = System.currentTimeMillis();
      LatencyWatcher.put("sendCatchupStream-sendEndMsg", (end-start));
      start = System.currentTimeMillis();
    }
    return true;//catchedup by log
  }              


 /** 
 *  remove a request from pendingQue 
 *  and process the request by sending catchupstream && add the subscriptionset 
 *                          or removing the subscriptionset 
 *  
 *  Side effect - update pendingCP with deferred checkpoints catchup requests. 
 **/ 
  protected void serveNewRequest(TaggedOutputStream tos, SubscriptionRequest nextRequest)
  throws IOException{

    if(dbgProgress){
      Env.dprintln(dbgProgress, "OutgoingInvalConnectionWorker processing " + nextRequest 
          + " @ streamSS=" + streamSS + " streamCVV=" + streamCVV);
    }

    //
    // CASE 1: New request is for CP
    //
    if(nextRequest.getType() == SubscriptionRequest.CP){


      sendCP(tos, 
          nextRequest.getSubscriptionSet(), 
//          nextRequest.getStartVV(),
          nextRequest.getStartVV().cloneMinVV(core.getCurrentVV()),
          nextRequest.getCPBodies()/*, 
          nextRequest.isIncludeAll()*/);

      this.addToStreamSS(nextRequest.getSubscriptionSet());

      //
      // CASE 2: New request is for log
      //
    } else if (nextRequest.getType() == SubscriptionRequest.LOG){

      boolean success = sendCatchupStream(nextRequest, tos);

      if(!success){
        if(dbgProgress){
          Env.dprintln(dbgProgress, 
              "sendCatchupStream for " + nextRequest.getSubscriptionSet()
              + streamCVV + " by CP instead of LOG due to gap");
        }
        sendCP(tos, 
            nextRequest.getSubscriptionSet(), 
//            nextRequest.getStartVV(),
            nextRequest.getStartVV().cloneMinVV(core.getCurrentVV()),
            nextRequest.getCPBodies()/*, 
            nextRequest.isIncludeAll()*/);

      }

      this.addToStreamSS(nextRequest.getSubscriptionSet());

      //
      // Case 3: New request to REMOVE
      //
    } else{
      assert nextRequest.getType() == SubscriptionRequest.REMOVE;

      this.removeFromStreamSS(nextRequest.getSubscriptionSet());
    }
    if(dbgProgress){
      System.out.println(" after serveNewRequest:" + nextRequest + " streamSS=" + streamSS 
          + " streamCVV=" + streamCVV);
    }
  }


 /** 
 *   main fuction 
 *  send a stream constructing of: 
 *  [INVAL_SOCKET_MAGIC, senderId, intial prevVV, subscriptionset [new], 
 *   {(CatchupStreamStartMsg - (GerneralInv)* - CatchupStreamEndMsg)| 
 *   (GeneralInv)*| 
 *   (CPStartMsg-cvv-LPVVRecord*- (PerObjStateForSend|PerRangeStateForSend|BodyMsg)*-RAS_SHIP_DONE)}* 
 *   TerminateMsg] 
 **/ 
  private static boolean printTBDs = false;
  public void run(){

    TaggedOutputStream tos = null;

    if(!printTBDs){
      Env.tbd("OutgoingInvalConnectionWorker can be optimized by adding a set of buffers for \n"+
          "multiple catchup stream threads and a main network sender thread.\n" +
          "Each catchup stream threads can parallelly generate the objects for\n" +
          "a catchupstream in its own buffer and the main network sender thread\n"  +
          "picks one stream buffer at a time and sends it all then pick another one...\n"+
          "And Give the preferrence to the ready and short ones." +
          "This strategies will quickly drain the pending add/remove requests, at least"+
      "the remove request can be done quickly and don't have to wait for any thread");
      printTBDs=true;
    }


    try{
      if(dbgProgress){
        Env.dprintln(dbgProgress, dbgTag + " run-starts");
      }

      tos = makeConnection();

      if(dbgProgress){
        Env.dprintln(dbgProgress, dbgTag + " run-connected");
      }

      handshake(tos);

      if(dbgProgress){
        Env.dprintln(dbgProgress, dbgTag + " run-handshake");
      }
      controller.informOutgoingInvalStreamInitiated(receiverId,
          streamCVV, 
          true);
      assert streamSS.equals(initialRequest.getSubscriptionSet()): "streamSS= " + streamSS + " request SS= " 
								    + initialRequest.getSubscriptionSet();
      //assert streamCVV.includes(core.getCurrentVV()); maybe not true because new updates come afer cp
      controller.informOutgoingSubscribeInvalInitiated(receiverId,
          streamSS);
      

      long now, timeDiff;
      Object nextEvent = null;
      //
      // main flow: 
      //            check if there's new requests
      //               if yes, send CP or catch up invalidation stream or remove subset
      //            
      //            send new invalidate(the invalidateIterator's interestset might have changed)
      while(!timeToDie){


       


        timeoutForInv = 0;//no timeout for next event, i.e. block until new event comes

        now = System.currentTimeMillis();
        timeDiff = (accumulateEndTime - now);


        if(timeDiff <= 0){//timeout for accumulating
          if(dbgProgress){
            System.out.println("timeout for accumulating");
            assert accumulatedInv != null;
          }
          this.flushImpreciseInv(tos);
        }else{
          if(accumulatedInv != null){//need to set timeout to make sure that
            //we will send the accumulatedInv eventually even if there's no more new updates
            //i.e. the getNext won't block forever.
            assert accumulateEndTime != Long.MAX_VALUE;
            timeoutForInv = timeDiff; //update the new timeout needed
          }
        }
        try{
          //if(dbgProgress){
          //  System.out.println(" box.getNext(" + timeoutForInv + ")");
          //}
          nextEvent = box.getNext(timeoutForInv);
        }catch(TimeoutException e){
          if(dbgProgress){
            System.out.println(" box.getNext(" + timeoutForInv + ") timeoutException." );
            assert accumulatedInv != null;
          }
          this.flushImpreciseInv(tos);
          continue;
        }


        //
        // process nextEvent( add/remove subscriptoin | next invalidation to send)
        //
        //assert nextEvent != null;

        // time to Die --- OutgoingInvalConnectionWorker.die() had the mailbox to return with null
        if(nextEvent == null){
            assert timeToDie;
            continue;
        }

        if(nextEvent instanceof GeneralInv){
          this.processNextInval((GeneralInv)nextEvent, tos);
        }else if(nextEvent instanceof SubscriptionRequest){
          if(accumulatedInv != null){//flush out the imprecise invalidation if any
            this.flushImpreciseInv(tos);
          }
          SubscriptionRequest request = (SubscriptionRequest) nextEvent;
          this.serveNewRequest(tos, request);
        }

	 if(autoRemoveIfEmptySubset  && streamSS.isEmpty()){
          if(dbgProgress){  
            Env.dprintln(dbgProgress, dbgTag 
                + " run-close and clean up because empty subscriptionset");
          }        
          this.die();
          if(dbgProgress){ 
            Env.dprintln(dbgProgress, 
                dbgTag + "OutgoingInvalConnectionWorker close due to empty subscriptionset");
          }
          break;
        }


      } // time to die = true
      if(dbgProgress){
        Env.dprintln(dbgProgress, dbgTag + " run-terminate no work todo");
      }
    }catch(IOException e){
      if(!timeToDie){
        e.printStackTrace();
        Env.inform("OutgoingInvalConnectionWorker unable to send to node " +
            receiverDNS + ":" + portInval + ". Exception: "+ e.toString());

        Env.dprintln(true, "OutgoingInvalConnectionWorker unable to send to node " +
            receiverDNS + ":" + portInval + ". Exception: "+ e.toString());
        timeToDie = true;
      }else{
        e.printStackTrace();
        System.err.println(" Ignore any exceptions during shutdown: " + e);
      }
      return;

    }catch(Exception e2){ // Unexpected exception
      if(!timeToDie){
        System.out.println("Unexpected exception in OutgoignInvalConnectionWorker " 
            + e2.toString());
        e2.printStackTrace();
        Env.remoteAssert(false,
            "Something broken in OutgoingInvalConnectionWorker");
        timeToDie = true;
      }else{
        e2.printStackTrace();
        System.err.println(" Ignore any exceptions during shutdown: " + e2.toString());
      }

      return;

    }
    finally{

      controller.informOutgoingInvalStreamTerminated(receiverId,
          streamSS, 
          streamCVV, 
          true);


      timeToDie = true;
      closeAndCleanupStreamAndSocket(tos, sock, fos);

    }
    assert timeToDie;
    return; 
  }



  // following fields are for read throughput measurement
  protected int count = 0;
  protected long startTime = 0;
  
  protected void writeInvToStream(TaggedOutputStream tos, GeneralInv gi)
  throws IOException{

    tos.writeObject(gi);
    streamCVV.advanceTimestamps(gi.getEndVV());
    
    if(SangminConfig.readThroughPutMeasure){
      if(count++ == 0){
        startTime = System.currentTimeMillis();
      }
      if(count == SangminConfig.readThroughputExpRate){
        count = 0;
        long timeToSleep = startTime + 1000 - System.currentTimeMillis();
        if(timeToSleep > 0){
          try{
            Thread.sleep(timeToSleep);
          }catch(Exception e){          
          }
        }
      }
    }
  }
  
  /**
   * process next Invalidate
   * if it intersects streamSS, send it immediately. Note flush the previously
   *     accumulated imprecise invalidation before send the new invalidate
   * else accumulate it in the accumulatedInv.
   */
  private void processNextInval(GeneralInv gi, TaggedOutputStream tos)
  throws IOException{
	if(dbgProgress){
		//System.out.println("processNextInval:" + gi);
	}
    if(gi.getInvalTarget().intersects(streamSS)){
      if(accumulatedInv != null){//flush out the imprecise invalidation
        this.flushImpreciseInv(tos);
      }
      writeInvToStream(tos, gi);
    }else{//accumulate
      this.accumulateImpreciseInv(gi);
      streamCVV.advanceTimestamps(gi.getEndVV());
    }

  }

  private void flushImpreciseInv(TaggedOutputStream tos)
  throws IOException{
    assert accumulatedInv != null;
    assert (ACCUMULATED_PRECISE_CONVERT2_IMPRECISE
            &&!(accumulatedInv instanceof PreciseInv))
      || (!(ACCUMULATED_PRECISE_CONVERT2_IMPRECISE));

    writeInvToStream(tos, accumulatedInv);
    accumulatedInv = null;
    accumulateEndTime = Long.MAX_VALUE;
  }

  private void accumulateImpreciseInv(GeneralInv gi){
    if ( accumulatedInv == null ){ //the accumulate starts

      if((ACCUMULATED_PRECISE_CONVERT2_IMPRECISE)&&(gi instanceof PreciseInv)){
        String objId = ((PreciseInv)gi).getObjId().toString();
        HierInvalTarget hit = HierInvalTarget.makeHierInvalTarget(objId);
        accumulatedInv = new ImpreciseInv(hit, gi.getStartVV(), gi.getEndVV());
      }else{
        accumulatedInv = (GeneralInv) gi.clone();
      }
      assert accumulateEndTime == Long.MAX_VALUE;
      accumulateEndTime = System.currentTimeMillis() + accumulateTimeOut;

    }else{//continue accumulating
      if (dbgProgress){
        Env.dprintln(dbgProgress, "InvalIterator::handleNextEvent() accumu:" + accumulatedInv);
        Env.dprintln(dbgProgress,"    nextEvent:" + gi);
        Env.dprintln(dbgProgress,"    ss:" + streamSS);
      }  
      accumulatedInv =(ImpreciseInv)accumulatedInv.newUnion(gi, 
          streamSS);

    }

    assert accumulateEndTime != Long.MAX_VALUE;
  }

 /** 
 *  Cleanly close the socket and stream 
 **/ 
  private void closeAndCleanupStreamAndSocket(TaggedOutputStream tos,
      Socket sock,
      FileOutputStream fos){
    assert timeToDie == true;
    if(dbgProgress){
      Env.inform(dbgTag 
          + "<<<close and cleanup>>>"                
          + " target: " + receiverId.toString()
          + " dns: " + receiverDNS
          + " port: " + portInval
          + ")");
    }

    System.out.println("closeAndCleanupStreamandSocket called");



    if(dbgProgress){
      Env.sprinterrln("Closing socket: OutgoingInvalConnection");
    }
    try{
      if(tos != null){
        tos.writeObject(new TerminateMsg());
        if(dbgProgress || printStream){
          System.out.println("OutgoingInvalConnection-" + receiverId + ".writeObject(terminagemsg)"); 
        }
          System.out.println("OutgoingInvalConnection-" + receiverId + ".writeObject(terminagemsg)"); 
        tos.close();
      }
    }catch(IOException e){
      // Do nothing when closing
    }
    try{
      if(sock != null){
        sock.close();
      }
      if(fos != null){
        fos.close();
      }
    }catch(IOException e){
      // Do nothing when closing
    }

    core.removeOutgoingConnectionMailBox(box);

    System.out.println("closeAndCleanupStreamandSocket called 1");
    return;
  }

 /** 
 *  main() -- self test code 
 **/ 
  public static void 
  main(String[] argv) {

  }

 /** 
 *  Unit Test helper functions 
 *   
 *  set the outputstream to a tmp file 
 **/ 
  public static void 
  setDbgWithFileOutputStream(boolean withFile){
    dbgWithFileOutputStream = withFile;
  }



public AcceptVV getStreamCVV(){
  return new AcceptVV(streamCVV);
}




}

//---------------------------------------------------------------------------
/* $Log$ */
//---------------------------------------------------------------------------
