//---------------------------------------------------------------------------
/* DemandReadWorkerThread.java
 * 
 * gets pending demand reads from the queue and reissues command
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------


import java.io.*;

public class DemandReadWorkerThread extends Thread{
    
  private RMIClient rmiClient; 
  private P2Runtime runtime;
  private Status status;
  private NodeId myNodeId;
  private static final boolean verbose = P2Runtime.verboseDR;
  private static final boolean dbg = false;

  public DemandReadWorkerThread(RMIClient rmiClient, 
                                P2Runtime runtime, 
                                Status status, 
                                NodeId myNodeId){
    this.rmiClient = rmiClient;
    this.runtime = runtime;
    this.status = status;
    this.myNodeId =  myNodeId;
  }

  //-------------------------------------------------------------------------
  //  gets the next pending subscription from the queue and adds it again
  //-------------------------------------------------------------------------
  public void 
  run(){
    
    long maxRetries = P2Config.getMaxDemandReadRetries(myNodeId);
    PendingDemandReadQueue drQueue = status.getPendingDemandReadQueue();
    long timeoutValue = P2Config.getDemandReadTimeout(myNodeId);   
    while(true){

      if(dbg){
	Env.dprintln(dbg, "DemandReadWorkerThread: wiating to get next from queue");
      }

      // this will automatically update the timeout and add
      // a clone to the pending queue.
      // if it reaches maxRetries, it will not be added to the queue
      DemandRead dr = drQueue.getNextAndUpdateTimeout(maxRetries, timeoutValue);
      

      if(dbg){
	Env.dprintln(dbg||verbose, "DemandReadWrokerThread: got dr " + dr.toString());
      }

      if(dr.getNumTries() > maxRetries){      
        if(dbg||verbose){
	  Env.dprintln(dbg||verbose,"DemandReadWorkerThread: already reached maxRetries");
	}
        drQueue.remove(dr);
        runtime.informDemandReadFailedMaxRetries(dr);
        continue;
      } else {
	if(dbg||verbose){
	  Env.dprintln(dbg||verbose, "DemandReadWorkerThread issuing command for " + dr);
	}
        long res = issueCommand(dr);
        
        //
        // For result = 0, the callee has indicated that it does not
        // have a copy of the data. Stop retrying and tell the
        // controller to try plan B. 
        //
        if(res == 0){
	  if(dbg||verbose){
	    Env.dprintln(dbg||verbose, "DemandReadWorkerThread: DR Miss " + dr);
	  }
          drQueue.remove(dr);
	  runtime.informDemandReadFailedMiss(dr);
	  continue;
        } 
        //
        // For result < 0, we had an RMI failure. Just retry the 
        // request at the already-specified timeout time.
        // (No action needed here.) Note that if the RMI
        // failure takes longer than the timeout time, the
        // retry may already have happened. That's OK.
        //
        else if(res < 0){
          // No action needed. Timeout event already pending.
        } 
        // 
        // For result == 0, wait for body to arrive or timeout.
        //
        else {
          // No action needed. Timeout event already pending.
        }

      }
    }        
  }


  //
  // 
  // Return >0 on success (destination plans to send us at least
  // 1 byte of data
  //
  // 0 = fail b/c destination does not have desired data
  // -1 = fail b/c not able to contact destination (retry)
  //
  private long issueCommand(DemandRead dr){
    try {

      long expect = rmiClient.issueDemandRead(dr.getObjId(),
                                              dr.getOffset(), 
                                              dr.getLength(),
                                              dr.getRequestId(),
                                              dr.getSenderNode(),
                                              dr.getReceiverNode(),
                                              Config.getDNS(dr.getReceiverNode()),
                                              Config.getPortBody(dr.getReceiverNode()),
					      dr.getAcceptStamp());
      return expect;
    }   
    //
    // For exceptions that reflect communications errors, treat
    // them as timeouts. Retry.
    //
    catch(ObjNotFoundException d){
      return 0;
    }
    catch(ReadOfInvalidRangeException e){
      //
      // The node does not have the data we want
      //
      return 0;
    }
    catch(RMIApplicationException f){
      if(verbose){
        System.out.println("DBG: DemandReadWorkerThread exception: " + f.toString());
        f.printStackTrace();
      }
      return -1; // Retry request on IO error
    }
    catch(IOException e){
      return -1;  // Retry request on IO error
    }
    catch(RMINetworkException g){
      return -1; // Retry request on IO error
    }
    catch(Exception e){
      // Unexpected exception!
      System.out.println("Unexpected exception DemandReadWorkerThread" + e.toString());
      e.printStackTrace();
      assert(false);
      return -1;
    }

    
  }

    
}

//---------------------------------------------------------------------------
/* $Log: DemandReadWorkerThread.java,v $
/* Revision 1.15  2007/04/02 22:10:00  nalini
/* junit fixes
/*
/* Revision 1.14  2007/04/02 21:06:48  zjiandan
/* fix Env.dprintln problems.
/*
/* Revision 1.13  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.11  2007/03/06 21:26:03  nalini
/* added verbose flags so as to print out rmi exceptions
/*
/* Revision 1.10  2007/03/06 18:25:51  zjiandan
/* Add optimization for CatchupInvalIterator, fixed SubscriptionSet, HierInvalTarget
/* and P2Runtime problems. Add aways split when receiving subtree ISStatus in Checkpoint.
/*
/* Revision 1.9  2007/02/01 06:12:09  zjiandan
/* Add acceptStamp to demandRead so that the sender only sends the data
/* that's at least as new as the acceptStamp.
/*
/* Revision 1.8  2006/10/13 22:18:46  dahlin
/* *** empty log message ***
/*
/* Revision 1.7  2006/09/24 20:06:31  nalini
/* trying to make overlog and practi work
/*
*/
//---------------------------------------------------------------------------
