//---------------------------------------------------------------------------
/*  MailBox
 * 
 *  generic mailbox which waits until the specified message is received.
 *  An additional MsgMatcher can be specified if a customized
 *  matching algorithm is required
 *
 * All msgs are string arrays so that we can use the same interface for
 * overlog and java policies
 * 
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.util.*;

public class MailBox {

  private MsgMatcher matcher;   // class to match msgs
  private LinkedList receivedMsgs;  // msg that have been received by not claimed yet
  private Hashtable waitingList;   // msgs being waited for

  private long highestIssuedKey = 0;

 
  //---------------------------------------------------------------------------
  // constructors
  //---------------------------------------------------------------------------
  
  public MailBox(){
    this(new PrefixMatcher());
  }
   
  public MailBox(MsgMatcher matcher) {
    this.receivedMsgs = new LinkedList();
    this.waitingList = new Hashtable();
    this.matcher = matcher;
  }

  //---------------------------------------------------------------------------
  // waitForMsg -- wait until a message matching <msg> is received
  //---------------------------------------------------------------------------

  public synchronized String[] waitForMsg(String[] msg){
      String[] res = checkIfAlreadyReceived(msg);	 
      if(res != null) {
        return res;
      }    

      Long index = new Long(highestIssuedKey++);
      WaitingListEntry wle = new WaitingListEntry(msg);
      waitingList.put(index, wle);

      while(((WaitingListEntry)waitingList.get(index)).getReceivedMsg() == null) {
        try{
          wait();
        }catch(InterruptedException e){
        }
      }
      
      res = ((WaitingListEntry)waitingList.get(index)).getReceivedMsg();
      assert(res != null);
      
      waitingList.remove(index);
      
      return res;
   }


  //---------------------------------------------------------------------------
  //  checkIfAlreadyReceived - 
  // return the first msg which matches, and removes it from
  // the received list
  // (i.e. if we receive a msg before waitFor is called, it will only 
  //  only unblock the first matching waitForMsg)
  //---------------------------------------------------------------------------

   private String[] checkIfAlreadyReceived(String[] msg){
     Iterator i = receivedMsgs.iterator();
     
     while(i.hasNext()) {
       String[] receivedMsg = (String[]) i.next();
       if(matcher.match(msg, receivedMsg)){
            i.remove();
            return receivedMsg;
          }
     }
     return null;
   }   

 //---------------------------------------------------------------------------
 // method by which the mailbox receives msgs
 //  -- marks all entries in the waiting list that match this msg
 //---------------------------------------------------------------------------


  public synchronized void receiveMsg(String[] rMsg){

    Iterator i = waitingList.values().iterator();

    // note: a waiting list entry will be given the most recent msg received
    while(i.hasNext()){
      WaitingListEntry wlEntry = (WaitingListEntry)i.next();
      if(matcher.match(wlEntry.getRequiredMsg(), rMsg)){
           wlEntry.setReceivedMsg(rMsg);
         }
    }
    
    notifyAll();
  }      

  public class WaitingListEntry {
    private String[] requiredMsg;
    private String[] receivedMsg;

    public WaitingListEntry(String[] m) {
      requiredMsg = m;
    }

    public void setReceivedMsg(String[] m){
      receivedMsg = m;
    }

    public String[] getReceivedMsg() {
      return receivedMsg;
    }
    
    public String[] getRequiredMsg() {
      return requiredMsg;
    }

  }
}

//---------------------------------------------------------------------------
/* $Log: MailBox.java,v $
/* Revision 1.2  2007/12/18 17:30:47  nalini
/* updated safetymodule
/*
/* Revision 1.1  2007/11/28 08:11:34  nalini
/* safety policy module and example checked in
/*
*/
//---------------------------------------------------------------------------
