//---------------------------------------------------------------------------
/* NodeIdMap.java
 *
 * Converts from Practi Node Ids  to Overlog Ids and vice versa
 *
 * A map file holds mapping from Practi node Ids to 
 *  overlog dataflow mapIds.
 *
 *  Format:
 *  <nodeid>  <hostname>  <port>
 *   
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.util.*;
import java.io.*;

public class NodeIdMap {
  static Hashtable olg2Practi;
  static Hashtable practi2Olg;
  private static  boolean initialized = false;
  
  static boolean dbg = false;

  public static void readFile(String mapFile) {
    olg2Practi = new Hashtable();
    practi2Olg = new Hashtable();
    parseFile(mapFile);
    initialized = true;
  }

  public static void createEmptyMap() {
    olg2Practi = new Hashtable();
    practi2Olg = new Hashtable();
    initialized = true;
  } 

  private static void parseFile(String mapFile){
    if(dbg){
      Env.dprintln(dbg, "NodeIdMap: Trying to read " + mapFile);
    }
    String line = null;
  
    try {
      BufferedReader in = new BufferedReader(new FileReader(mapFile));
      while((line = in.readLine()) != null){
        StringTokenizer st = new StringTokenizer(line);
        NodeId nodeId = new NodeId (new Integer(st.nextToken()).intValue());
        String hostname = st.nextToken();
        long port = (new Long(st.nextToken())).longValue();
        OverlogId olgId = new OverlogId(hostname, port);

        olg2Practi.put(olgId, nodeId);
        practi2Olg.put(nodeId, olgId);
      }
    }catch(Exception e){
      //unexpected error
      e.printStackTrace();
      assert(false);
    }  
  }

  public static void add(NodeId nodeId, OverlogId olgId){
    if(!initialized){
      createEmptyMap();
    }
      olg2Practi.put(olgId, nodeId);
      practi2Olg.put(nodeId, olgId);
  }

  public static void writeToFile(String fileName) {
    if (initialized) {
      try {
        BufferedWriter out = new BufferedWriter(new FileWriter(fileName));
        for (Enumeration e = practi2Olg.keys(); e.hasMoreElements();){
          NodeId nodeId = (NodeId) e.nextElement();
          OverlogId olgId = (OverlogId) practi2Olg.get(nodeId);
          
          out.write(nodeId + "\t" + olgId.getHostname() + "\t" + olgId.getPort() + "\n");
        }
        out.close();
      } catch(Exception e) {
        //unexpected error
        e.printStackTrace();
      }
    }
  }

  public static NodeId getNodeId(OverlogId olgId){
    NodeId ret = null;
    if(initialized) {
      ret =  (NodeId) olg2Practi.get(olgId);
    }
    if(ret == null){
      assert(false):"NodeIdMap has not been initialized or could not find entry for overlogId:" + olgId;
    }
    return ret;
  }

  public static OverlogId getOverlogId(NodeId nodeId) {
    OverlogId ret = null;

    if(initialized) {
      ret =  (OverlogId) practi2Olg.get(nodeId);
    }

    if(ret == null) {
      assert(false):"NodeIdMap has not been initialized or could not find entry for NodeId";
    }
    return ret;
  }

  public static String getString(){
    if(initialized){
      return practi2Olg.toString();
    }
    return null;
  }


}    
//---------------------------------------------------------------------------
/* $Log: NodeIdMap.java,v $
/* Revision 1.4  2007/04/02 21:06:48  zjiandan
/* fix Env.dprintln problems.
/*
/* Revision 1.3  2007/01/15 20:27:15  nalini
/* debug statement added
/*
/* Revision 1.2  2006/11/02 20:27:38  nalini
/* updated NodeIdMap so that it will assert false instead of returning null if a mapping is not found
/*
/* Revision 1.1  2006/09/19 22:22:58  nalini
/* p2 and Practi integration
/*
*/
//---------------------------------------------------------------------------

 
