//---------------------------------------------------------------------------
/* ObjReaderWriter
 * 
 * Has methods to read lines from an object and to 
 * append to the object
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.io.*;

public class ObjReaderWriter {
  private LocalInterface li;
  private ObjId objId;
  private int currLoc = 0;
  private int blockSize = 80;


  private boolean reachedEOF = false;
  private String bufferedStr = "";
  private boolean dbg = false;

  //-------------------------------------------------------------------------
  // Constructor
  //-------------------------------------------------------------------------  
  public ObjReaderWriter(LocalInterface li, ObjId objId) {
    this.li = li;
    this.objId = objId;

  }


  //-------------------------------------------------------------------------
  // reads a line from the obj and returns it as a string.
  //-------------------------------------------------------------------------  
 
  public String readLine() throws
    EOFException,
    IOException{
    if(dbg) {
      Env.dprintln(dbg, "ObjReaderWriter:readLn called " + this.toString());
    }
    try{
      // look for \n in the string
      while(bufferedStr.indexOf("\n") < 0) {
        if(reachedEOF){
          if(dbg){
            Env.dprintln(dbg, "ObjReaderWriter: reachedEOF throwing EOF");
          }
          throw new EOFException("Reached EOF");
        }
        byte[] b = readNextBlock();
        bufferedStr += new String(b);      
      }
      
      int index = bufferedStr.indexOf('\n');
      String returnStr = bufferedStr.substring(0, index);
      bufferedStr = bufferedStr.substring(index + 1);
      
      if(dbg) {
        Env.dprintln(dbg, "ObjReaderWriter: readLn returns" + this.toString());
      }

      return returnStr;
    } catch(ObjNotFoundException onfe) {
      if(dbg){
        Env.dprintln(dbg, "ObjReaderWriter:ObjNotfound throwing EOF exception");
      }
      reachedEOF = true;
      currLoc = 0; 
      throw new EOFException("ObjNotFound");
    
    } catch(EOFException e){
      reachedEOF=true;
      throw e;
    }

  }


  //-------------------------------------------------------------------------
  // reads the next block from the object
  //-------------------------------------------------------------------------  

  private byte[] readNextBlock() throws 
    EOFException,
    ObjNotFoundException,
    IOException{
    
    if(dbg) {
      Env.dprintln(dbg, "ObjReaderWriter:readNextBlock called ");
    }

    try{
      // we assume that the tuples are always consistent
      BodyMsg bMsg = this.li.read(objId,
                                  currLoc,
                                  blockSize,
                                  false,
                                  false);
      ImmutableBytes ib = bMsg.getBody();
  
      currLoc += ib.getLength();

      if(dbg) {
        Env.dprintln(dbg, "ObjReaderWriter:readNextBlock returns " + ib);
      }
      return ib.getCopyBytes();  // we return a copy of the bytes

    }catch(ReadOfInvalidRangeException roire) {
      //this should not happen
      assert(false):"ReadOfInvalidRangeException";
    }catch(ReadOfHoleException rohe) {
      //this should not happen
      assert(false):"ReadOfHoleException";
    }

    return null;
  }
     
  //-------------------------------------------------------------------------
  // finds the EOF location of the obj
  //-------------------------------------------------------------------------  
  private int findEOF() throws IOException{
    if(reachedEOF) {
      return currLoc;
    }
    
  // now we need to find the EOF loc
    int tLoc = currLoc;
    try{
      while(true) {
        try{
          BodyMsg bMsg = this.li.read(objId,
                                      tLoc,
                                      blockSize,
                                      false,
                                      false);
          ImmutableBytes ib = bMsg.getBody();
          tLoc += ib.getLength();
          
        }catch(ReadOfInvalidRangeException roire) {
          //this should not happen
          assert(false);
        }catch(ReadOfHoleException rohe) {
          //this should not happen
          assert(false);
        }

      }
    }catch(EOFException eof) {
      return tLoc;
    } catch(ObjNotFoundException onfe) {
      return 0;
    }
  }

  //-------------------------------------------------------------------------
  // writes a string to the end of an object
  // note: appends a new line character at the end of the str
  // synchronized so that only one thread can write.
  //-------------------------------------------------------------------------  
  public synchronized void appendLn(String str) throws IOException {
    String strToWrite = str + '\n';
    
    ImmutableBytes ib = new ImmutableBytes(strToWrite.getBytes());
    int eof = findEOF();
    if(dbg) {
      Env.dprintln(dbg,"ObjReaderWriter appendLn: EOF found at " + eof);
    }
    li.write(objId, eof, ib.getLength(), ib, true);
    
    if(reachedEOF){
      reachedEOF=false;
    }
  }

  //-------------------------------------------------------------------------
  // toString
  //------------------------------------------------------------------------- 

  public String toString() {
    String str = "<" + objId + ":Loc=" + currLoc + ":eof=" + reachedEOF + 
                 ":buffer=" + bufferedStr + ">";
    return str;
  }
}

//---------------------------------------------------------------------------
/* $Log: ObjReaderWriter.java,v $
/* Revision 1.1  2007/07/11 20:25:53  nalini
/* persistent tuple support added
/*
*/
//---------------------------------------------------------------------------


    