//---------------------------------------------------------------------------
/* Overlog.java
 *
 * Responsible for handling the overlog.
 * Sets up  and installs the Overlog workflow.
 * Sets up  the watch tuples & handlers
 * Starts the tuple insert workers and the tuple watch workers
 *  
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.util.*;

public class Overlog{
  private static int NUM_INSERT_WORKERS = 10;
  private static int NUM_WATCH_WORKERS = 10;
  private String olgFilename;
  private P2JavaWrapper javaWrapper;
  private OverlogId myOverlogId;
  private NodeId myNodeId;
  private P2ServerInterface serverInterface;

  public static boolean useServer = false;
  
  // queues
  private BlockedQueue overlogInsertQueue;
  private BlockedQueue overlogWatchQueue;
  private Hashtable tupleHandlers;


//---------------------------------------------------------------------------
// Constructor
//---------------------------------------------------------------------------

  public Overlog(String olgFilename, OverlogId myOverlogId, NodeId myNodeId) {
    this.olgFilename = olgFilename;
    this.myOverlogId = myOverlogId;
    this.myNodeId = myNodeId;

    overlogInsertQueue = new BlockedQueue();
    overlogWatchQueue = new BlockedQueue();
    tupleHandlers = new Hashtable();

    if(useServer) {
      serverInterface = new P2ServerInterface( P2Config.getP2ServerPort(myNodeId),
                                               P2Config.getP2ClientPort(myNodeId), 
                                               olgFilename,
                                               myOverlogId.getHostname(),
                                               new Long(myOverlogId.getPort()).toString());
    } else {
      javaWrapper = new P2JavaWrapper(olgFilename, myOverlogId.getHostname(), 
                                    new Long(myOverlogId.getPort()).toString());
    }
  }

//---------------------------------------------------------------------------
// Register tuples which need to be watched
//---------------------------------------------------------------------------
  public void registerHandler(TupleHandler th) {

    if(!tupleHandlers.containsKey(th.getTupleName())){
      tupleHandlers.put(th.getTupleName(), th);
    }
  }

//---------------------------------------------------------------------------
// starts the overlog data flow in a separte thread and 
// starts the insert and watch workers
//---------------------------------------------------------------------------
 
  public void start(){

    //
    // start workers
    // 
   
    if(useServer) {
      for(int i = 0; i < NUM_INSERT_WORKERS; i++){    

        OverlogInsertWorker overlogInsertWorker = new OverlogInsertWorker(serverInterface, 
                                                                          overlogInsertQueue);
        overlogInsertWorker.start();
      }
      
      
      for(int i = 0; i < NUM_WATCH_WORKERS; i++) {
        OverlogWatchWorker overlogWatchWorker = new OverlogWatchWorker(tupleHandlers, 
                                                                       overlogWatchQueue);
        overlogWatchWorker.start();
      }
      
    } else {
      for(int i = 0; i < NUM_INSERT_WORKERS; i++){    

        OverlogInsertWorker overlogInsertWorker = new OverlogInsertWorker(javaWrapper, 
                                                                          overlogInsertQueue);
        overlogInsertWorker.start();
      }


      for(int i = 0; i < NUM_WATCH_WORKERS; i++) {
        OverlogWatchWorker overlogWatchWorker = new OverlogWatchWorker(tupleHandlers, 
                                                                       overlogWatchQueue);
        overlogWatchWorker.start();
      }
    }

    //
    // start P2JavaWrapper
    // 
    String[] tuplesToWatch;
    if(OverlogPolicy.P2VersionHasSubscribeProblem){
      tuplesToWatch = new String[] {"issueCommand"};
    }
    else {
      tuplesToWatch= new String[tupleHandlers.size()];
      tuplesToWatch = (String[])tupleHandlers.keySet().toArray(tuplesToWatch);
    }

    if(useServer) {
      serverInterface.setWatchTuples(tuplesToWatch);
      serverInterface.setWatcher(new OverlogWatcher(myOverlogId, overlogWatchQueue));
      OverlogThread t = new OverlogThread(serverInterface);
      t.start();
    } else{      
      javaWrapper.setWatchTuples(tuplesToWatch);
      javaWrapper.setWatcher(new OverlogWatcher(myOverlogId, overlogWatchQueue));
      OverlogThread t = new OverlogThread(javaWrapper);
      t.start();
    }


  }

//---------------------------------------------------------------------------
// Inserts a Tuple into the queue which will be consumed by a worker
//---------------------------------------------------------------------------
 
  public void insertTuple(Tuple tp){
    overlogInsertQueue.add(tp);
  }
}

//---------------------------------------------------------------------------
/* $Log: Overlog.java,v $
/* Revision 1.6  2007/09/12 19:08:12  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.5  2007/07/11 20:25:53  nalini
/* persistent tuple support added
/*
/* Revision 1.4  2007/06/05 21:15:41  nalini
/* changed Overlog to same process
/*
/* Revision 1.3  2007/03/17 01:24:46  nalini
/* P2Server updated
/*
/* Revision 1.2  2007/03/16 23:57:35  nalini
/* p2serverinterface added
/*
/* Revision 1.1  2006/10/16 20:55:37  nalini
/* P2 tuple handler support added
/*
*/
//---------------------------------------------------------------------------