//---------------------------------------------------------------------------
/* OverlogPolicy.java
 *
 *  Installs and starts the overlog policy
 *  Converts the inform events to Overlog tuples and inserts it into
 *  the data flow, and converts overlog tupels into runtime calls.
 *
 *   ****Note: P2 V 0.7.2 does not support "/" in strings so 
 *   we convert all "/" to "-" when inserting tuples and 
 *   all "-' to "/" when receiveing tuples
 *
 *  However P2 v0.8.2 can support "/" in strings 
 *
 *  Please set the P2VersionHasStringProblem appropriately
 *
 *  Note(2) P2 v0.8.2 has a subscribe problem.  If more than 
 *  2 tuples are watched, a "subscribed" tuple calls both handlers
 *  Workaround: we just watch the "issueCommand" tuple and convert it 
 *  to a practi action
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------
import java.util.*;

public class OverlogPolicy extends Policy {
  public final static boolean P2VersionHasStringProblem = false;
  public final static boolean P2VersionHasSubscribeProblem = true;

  private OverlogId myOverlogId;
  private Overlog overlog;
  private PersistentTupleInterface ptInterface;
  private boolean started = false;

  private boolean dbg = false;  // prints out a lot of debug output
  private boolean quietMode = false; // does not insert tuples to overlog

  // Enables debug mode -- all insert events are added to a linked list
  // rather than to overlog
  private boolean dbgMode = false;
  private LinkedList dbgInsertedTuples;



  // selectively prints output
  private static boolean verboseDR = P2Runtime.verboseDR;
  private static boolean verboseBodySub = P2Runtime.verboseBodySub;
  private static boolean verboseInvalSub = P2Runtime.verboseInvalSub;
  private static boolean verboseLocal = P2Runtime.verboseLocal;
  private static boolean verboseRecvInval = P2Runtime.verboseRecvInval;


 //---------------------------------------------------------------------------
// Constructor
//---------------------------------------------------------------------------
  
  public OverlogPolicy(String olgFilename, 
                       NodeId myNodeId,
                       String nodeMapFilename){
    
   NodeIdMap.readFile(nodeMapFilename);
   myOverlogId = NodeIdMap.getOverlogId(myNodeId);
   if(myOverlogId == null) {
     System.err.println("myOverlogId is null");
     assert(false);
   }

   overlog = new Overlog(olgFilename, myOverlogId, myNodeId);
  }

  
  //---------------------------------------------------------------------------
  // setting the runtime to interact with
  //---------------------------------------------------------------------------
  public void setRuntime(P2Runtime runtime){
    this.runtime = runtime;
    // persistent tuple interface can only be set after the runtime is set
    this.ptInterface = new PersistentTupleInterface(runtime, this);
    // we register the handlers once the runtime is set
    registerDefaultHandlers();
  }

  //---------------------------------------------------------------------------
  // Starts the policy
  //---------------------------------------------------------------------------
  public void start(){
    startOverlog();
  }


 //---------------------------------------------------------------------------
  // Shuts down the policy
  //---------------------------------------------------------------------------
  public void shutdown(){
    stopOverlog();
  }



  //---------------------------------------------------------------------------
  // interface to interact with safety module
  //---------------------------------------------------------------------------

  // receiveMsg: called by safety to send a msg to liveness policy
  public void receiveMsg(String[] msg){
    insertTuple(new Tuple(msg));
  }

  //---------------------------------------------------------------------------
  // Starts the overlog workflow
  //---------------------------------------------------------------------------
  public void startOverlog(){
    if(dbg) {
      System.out.println("overlog start...");
    }
    overlog.start();
    started = true;
  }

  //---------------------------------------------------------------------------
  // DebugMode:  does not start overlog
  // all inserts are added to the linked list.
  //---------------------------------------------------------------------------
  public void startInDebugMode(LinkedList list){
    assert(list != null);
    dbgMode = true;
    dbgInsertedTuples = list;
    started = true;
  }

//---------------------------------------------------------------------------
// stop Overlog
//---------------------------------------------------------------------------
  
  public void stopOverlog(){
    if(dbg) {
      System.out.println("overlog stop...");
    }
    // not implemented
    // TODO: need to implement a clean way to shutdown overlog and all
    // the workers
  }


//---------------------------------------------------------------------------
// Register Default Handlers for tuples to watch 
//---------------------------------------------------------------------------
  
  private void registerDefaultHandlers(){ 
    TupleHandler th = new AddBodySubscriptionHandler(runtime);
    overlog.registerHandler(th);
       
    th = new RemoveBodySubscriptionHandler(runtime);
    overlog.registerHandler(th);

    th = new RemoveOutgoingBodySubscriptionHandler(runtime);
    overlog.registerHandler(th);

    th = new AddInvalSubscriptionHandler(runtime);
    overlog.registerHandler(th);

    th = new RemoveInvalSubscriptionHandler(runtime);
    overlog.registerHandler(th);

    th = new RemoveOutgoingInvalSubscriptionHandler(runtime);
    overlog.registerHandler(th);

    th = new DemandReadHandler(runtime);
    overlog.registerHandler(th);

    th = new PushUpdateHandler(runtime);
    overlog.registerHandler(th);

    th = new UnbindHandler(runtime);
    overlog.registerHandler(th);

    th = new DebargoHandler(runtime);
    overlog.registerHandler(th);

    th = new RequestSyncHandler(runtime);
    overlog.registerHandler(th);

    th = new SyncHandler(runtime);
    overlog.registerHandler(th);

    th = new TupleReadHandler(ptInterface);
    overlog.registerHandler(th);

    th = new TupleReadAndWatchHandler(ptInterface);
    overlog.registerHandler(th);

    th = new TupleWriteHandler(ptInterface);
    overlog.registerHandler(th);

    th = new TupleStopWatchHandler(ptInterface);
    overlog.registerHandler(th);

    th = new TupleDeleteHandler(ptInterface);
    overlog.registerHandler(th);

  }

//---------------------------------------------------------------------------
// Register user-defined handlers - can be used by the local interface 
// or policy writer 
// ** Note this should be called before start() **
//---------------------------------------------------------------------------
  
  public void registerHandler(TupleHandler th){
    if(!started) {
      overlog.registerHandler(th);
    } else{
      assert(false): "OverlogPolicy: handlers must be registered before overlog started";
    }
  }

//---------------------------------------------------------------------------
// Insert tuple - Can be used by LI to insert tuples into the
// overlog file
//---------------------------------------------------------------------------
  public void insertTuple(Tuple tp) {
    if(started) {
      if(!dbgMode) {
        overlog.insertTuple(tp);
      }else {
        dbgInsertedTuples.add(tp);
      }
    }else {
      assert(quietMode): "OverlogPolicy: tuples can be inserted only after overlog has started";
    }
  }


//---------------------------------------------------------------------------
// information of practi events - convert them to tuples to be inserted in
// overlog
//---------------------------------------------------------------------------


  // Note: we may get muliple informLocalReadInvalid events for the
  // same read. If we already have issued a demand read for this
  // objId/offset, then we may want to rely on underlying retry
  // logic rather than resend request.
  public void informLocalReadInvalid(ObjId objId, 
                                     long offset, 
                                     long length,
				     AcceptStamp inval) {
    if(dbg || verboseLocal){
      Env.dprintln(dbg || verboseLocal, "OverlogPolicy: informLocalReadInvalid:"+
                 objId+":"+offset+":"+length);
    }

    String objIdStr = objId.toString();

    if(P2VersionHasStringProblem){
      objIdStr = objId.toString().replaceAll("/", "-");
    }
    String[] strA = { "informLocalReadInvalid", 
                      myOverlogId.toString(),  
                      objIdStr,
                      new Long(offset).toString(), 
                      new Long(length).toString(), 
                      new Long(inval.getLocalClock()).toString(),
                      inval.getNodeId().toString()};

    insertTuple(new Tuple(strA));
  }


  // Note: we may get muliple informLocalReadImprecise events for the
  // same read. If we already have issued connection request,
  // then we may want to rely on underlying retry logic rather 
  // than resend request.
  public void informLocalReadImprecise(ObjId objId, 
                                       long offset, 
                                       long length){
    if(dbg || verboseLocal){
      Env.dprintln(dbg || verboseLocal, "OverlogPolicy :: informLocalReadImprecise:"+
		   objId+":"+offset+":"+length);
    }

    String objIdStr = objId.toString();
    if(P2VersionHasStringProblem){
      objIdStr = objId.toString().replaceAll("/", "-");
    }

    String[] strA = { "informLocalReadImprecise",
                      myOverlogId.toString(), 
                      objIdStr,
                      new Long(offset).toString(), 
                      new Long(length).toString()};

    insertTuple(new Tuple(strA));
  }


  public void informLocalWrite(ObjId objId, 
                               long offset, 
                               long length, 
                               AcceptStamp as,
                               boolean isBound, 
                               boolean isEmbargoed){
    if(dbg || verboseLocal){
      Env.dprintln(dbg || verboseLocal, "OverlogPolicy :: informLocalWrite"+
		   objId+":"+offset+":"+length +isBound + ":" + isEmbargoed);
    }

    String objIdStr = objId.toString();
    if(P2VersionHasStringProblem){
      objIdStr = objId.toString().replaceAll("/", "-");
    }

    String[] strA = { "informLocalWrite",
                      myOverlogId.toString(), 
                      objIdStr,
                      new Long(offset).toString(), 
                      new Long(length).toString(),
                      new Long(as.getLocalClock()).toString(),
                      as.getNodeId().toString(),
                      new Boolean(isBound).toString(), 
                      new Boolean(isEmbargoed).toString()};

    insertTuple(new Tuple(strA));
  }

  public void informLocalDelete(ObjId objId){  
    if(dbg || verboseLocal){
      Env.dprintln(dbg || verboseLocal, "OverlogPolicy :: informLocalDelete "+ objId);  
    }
    
    String objIdStr = objId.toString();
    if(P2VersionHasStringProblem){
      objIdStr = objId.toString().replaceAll("/", "-");
    }

    String[] strA = { "informLocalDelete",
                      myOverlogId.toString(), 
                      objIdStr};

    insertTuple(new Tuple(strA));
  }

  public void informReceiveInval(NodeId senderNodeId,
                                 ObjId objId, 
                                 long offset, 
                                 long length,
                                 AcceptStamp as, 
                                 boolean isBound, 
                                 boolean isEmbargoed) {
    if(dbg || verboseRecvInval){
      Env.dprintln(dbg || verboseRecvInval, "OverlogPolicy :: informReceiveInval"+ 
		   senderNodeId+":"+objId+":"+offset+":"+length+":as="+as);
    }

    String objIdStr = objId.toString();
    if(P2VersionHasStringProblem){
      objIdStr = objId.toString().replaceAll("/", "-");
    }


    String[] strA = { "informReceiveInval",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      objIdStr, 
                      new Long(offset).toString(),
                      new Long(length).toString(),
                      new Long(as.getLocalClock()).toString(),
                      as.getNodeId().toString(),
                      new Boolean(isBound).toString(),
                      new Boolean(isEmbargoed).toString()};

    insertTuple(new Tuple(strA));
  }  



  public void informDemandReadSuccess(NodeId senderNodeId, 
                                      ObjId objId, 
                                      long offset, 
                                      long length,
				      AcceptStamp as) {
    if(dbg || verboseDR){
      Env.dprintln(dbg || verboseDR, "OverlogPolicy :: informDemandReadSuccess"+senderNodeId+":"
		   +objId+":"+offset+":"+length
		   + ":" + as.toString());
    }

    String objIdStr = objId.toString();
    if(P2VersionHasStringProblem){
      objIdStr = objId.toString().replaceAll("/", "-");
    }

    String[] strA = { "informDemandReadSuccess",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      objIdStr,
                      new Long(offset).toString(),
                      new Long(length).toString(),
                      new Long(as.getLocalClock()).toString(),
		      as.getNodeId().toString()};

    insertTuple(new Tuple(strA));
  }
 
  public void informDemandReadFailedMiss(NodeId senderNodeId, 
                                         ObjId objId, 
                                         long offset, 
                                         long length,
					 AcceptStamp as){
    if(dbg || verboseDR){
      Env.dprintln(dbg || verboseDR, "OverlogPolicy :: informDemandReadFailedMiss"+senderNodeId+":"
		   +objId+":"+offset+":"+length);
    }

    String objIdStr = objId.toString();
    if(P2VersionHasStringProblem){
      objIdStr = objId.toString().replaceAll("/", "-");
    }

     String[] strA = { "informDemandReadFailedMiss",
                       myOverlogId.toString(),
                       NodeIdMap.getOverlogId(senderNodeId).toString(),
                       objIdStr,
                       new Long(offset).toString(),
                       new Long(length).toString(),
                       new Long(as.getLocalClock()).toString(),
                       as.getNodeId().toString()};
     
    insertTuple(new Tuple(strA));
  }

  public void informDemandReadFailedMaxRetries(NodeId senderNodeId, 
                                               ObjId objId, 
                                               long offset, 
                                               long length,
					       AcceptStamp as){
    if(dbg || verboseDR){
      Env.dprintln(dbg || verboseDR, "OverlogPolicy :: informDemandReadFailedMaxRetries"+senderNodeId
		   +":"+objId+":"+offset+":"+length);
    }

    String objIdStr = objId.toString();
    if(P2VersionHasStringProblem){
      objIdStr = objId.toString().replaceAll("/", "-");
    }

    String[] strA = { "informDemandReadFailedMaxRetries",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      objIdStr,
                      new Long(offset).toString(),
                      new Long(length).toString(),
                      new Long(as.getLocalClock()).toString(),
		      as.getNodeId().toString()};

    insertTuple(new Tuple(strA));
  }


  public void recvSyncReply(NodeId senderNodeId, 
                            AcceptStamp as) {
    if(dbg){
      Env.dprintln(dbg, "OverlogPolicy :: recvSyncReply "+senderNodeId+":"+as);
    }
    String[] strA = { "recvSyncReply",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      new Long(as.getLocalClock()).toString(),
                      as.getNodeId().toString()};

    insertTuple(new Tuple(strA));
  }
  

//---------------------------------------------------------------------------
// Subscription add events
//---------------------------------------------------------------------------

  public void informAddedInvalSubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                           SubscriptionSet ss) {
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "OverlogPolicy :: informAddedInvalSubscription"+
		   senderNodeId+":"+receiverNodeId+":"+ss);
    }

    String ssStr = ss.toString();
    if(P2VersionHasStringProblem){
      ssStr.replaceAll("/", "-");
    }

    String[] strA = { "informAddedInvalSubscription",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      NodeIdMap.getOverlogId(receiverNodeId).toString(),
                      ssStr};

    insertTuple(new Tuple(strA));
  }

  public void informAttachedInvalSubscription(NodeId senderNodeId, 
                                              NodeId receiverNodeId, 
                                              SubscriptionSet ss) {
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "OverlogPolicy :: informAttachedInvalSubscription"+
		   senderNodeId+":"+receiverNodeId+":"+ss);
    }

    String ssStr = ss.toString();
    if(P2VersionHasStringProblem){
      ssStr.replaceAll("/", "-");
    }

    String[] strA = { "informAttachedInvalSubscription",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      NodeIdMap.getOverlogId(receiverNodeId).toString(),
                      ssStr};

    insertTuple(new Tuple(strA));
  }


  public void informAddedBodySubscription(NodeId senderNodeId, 
                                          NodeId receiverNodeId, 
                                          SubscriptionSet ss){
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "OverlogPolicy :: informAddedBodySubscription "+
		   senderNodeId+":"+receiverNodeId+":"+ss);
    }

    String ssStr = ss.toString();
    if(P2VersionHasStringProblem){
      ssStr.replaceAll("/", "-");
    }

    String[] strA = { "informAddedBodySubscription",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      NodeIdMap.getOverlogId(receiverNodeId).toString(),
                      ssStr};

    insertTuple(new Tuple(strA));
  }

  public void informAddedOutgoingInvalSubscription(NodeId senderNodeId, 
                                                   NodeId receiverNodeId, 
                                                   SubscriptionSet ss){
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "OverlogPolicy :: informAddedOutgoingInvalSubscription"+
		   senderNodeId+":"+receiverNodeId+":"+ss);
    }

    String ssStr = ss.toString();
    if(P2VersionHasStringProblem){
      ssStr.replaceAll("/", "-");
    }

    String[] strA = { "informAddedOutgoingInvalSubscription",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      NodeIdMap.getOverlogId(receiverNodeId).toString(),
                      ssStr};

    insertTuple(new Tuple(strA));
  }

  public void informAddedOutgoingBodySubscription(NodeId senderNodeId, 
                                                  NodeId receiverNodeId, 
                                                  SubscriptionSet ss) {
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "OverlogPolicy :: informAddedOutgoingBodySubscription "
		   +senderNodeId+":"+receiverNodeId+":"+ss);
    }

    String ssStr = ss.toString();
    if(P2VersionHasStringProblem){
      ssStr.replaceAll("/", "-");
    }

    String[] strA = { "informAddedOutgoingBodySubscription",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      NodeIdMap.getOverlogId(receiverNodeId).toString(),
                      ssStr};

    insertTuple(new Tuple(strA));
  }

//---------------------------------------------------------------------------
// Subscription removed events
//---------------------------------------------------------------------------
  

  //Note:  This is triggered when
  //       - SS is removed from a connection
  //       - inval connection dies
  public void informRemovedInvalSubscription(NodeId senderNodeId, 
                                             NodeId receiverNodeId, 
                                             SubscriptionSet ss){
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseInvalSub, "OverlogPolicy :: informRemovedInvalSubscription "+
		   senderNodeId+":"+receiverNodeId+":"+ss);
    }

    String ssStr = ss.toString();
    if(P2VersionHasStringProblem){
      ssStr.replaceAll("/", "-");
    }


    String[] strA = { "informRemovedInvalSubscription",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      NodeIdMap.getOverlogId(receiverNodeId).toString(),
                      ssStr};
    
    insertTuple(new Tuple(strA));
  }

  //Note:  This is triggered when
  //       - body connection dies
  public void informRemovedBodySubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "OverlogPolicy :: informRemovedBodySubscription "
		   +senderNodeId+":"+receiverNodeId+":"+ss);
    }

    
    String ssStr = ss.toString();
    if(P2VersionHasStringProblem){
      ssStr.replaceAll("/", "-");
    }

    String[] strA = { "informRemovedBodySubscription",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      NodeIdMap.getOverlogId(receiverNodeId).toString(),
                      ssStr};
    
    insertTuple(new Tuple(strA));
  }

  public void informRemovedOutgoingInvalSubscription(NodeId senderNodeId, 
                                                     NodeId receiverNodeId, 
                                                     SubscriptionSet ss) {
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "OverlogPolicy :: informRemovedOutgoingInvalSubscription"+
		   senderNodeId+":"+receiverNodeId+":"+ss);
    }

    String ssStr = ss.toString();
    if(P2VersionHasStringProblem){
      ssStr.replaceAll("/", "-");
    }


    String[] strA = { "informRemovedOutgoingInvalSubscription",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      NodeIdMap.getOverlogId(receiverNodeId).toString(),
                      ssStr};
    
    insertTuple(new Tuple(strA));
  }  

  public void informRemovedOutgoingBodySubscription(NodeId senderNodeId, 
                                                    NodeId receiverNodeId, 
                                                    SubscriptionSet ss){
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "OverlogPolicy :: informRemovedOutgoingBodySubscription "
		   +senderNodeId+":"+receiverNodeId+":"+ss);
    }

    String ssStr = ss.toString();
    if(P2VersionHasStringProblem){
      ssStr.replaceAll("/", "-");
    }

    String[] strA = { "informRemovedOutgoingBodySubscription",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      NodeIdMap.getOverlogId(receiverNodeId).toString(),
                      ssStr};
    
    insertTuple(new Tuple(strA));
  }


//---------------------------------------------------------------------------
// Subscription failed
//---------------------------------------------------------------------------
  

  //Note:  This is triggered when
  //       - could not establish a connection to sender
  public void informAddInvalSubscriptionFailed(NodeId senderNodeId, 
                                               NodeId receiverNodeId, 
                                               SubscriptionSet ss){
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "OverlogPolicy :: informAddInvalSubscriptionFailed "+
		   senderNodeId+":"+receiverNodeId+":"+ss);
    }
 
    String ssStr = ss.toString();
    if(P2VersionHasStringProblem){
      ssStr.replaceAll("/", "-");
    }

    String[] strA = { "informAddInvalSubscriptionFailed",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      NodeIdMap.getOverlogId(receiverNodeId).toString(),
                      ssStr};
    
    insertTuple(new Tuple(strA));
  }


  //Note:  This is triggered when
  //       - body connection could not be established
  public void informAddBodySubscriptionFailed(NodeId senderNodeId, 
                                              NodeId receiverNodeId, 
                                              SubscriptionSet ss){
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "OverlogPolicy :: informAddBodySubscriptionFailed "
		   +senderNodeId+":"+receiverNodeId+":"+ss);
    }

    String ssStr = ss.toString();
    if(P2VersionHasStringProblem){
      ssStr.replaceAll("/", "-");
    }

    String[] strA = { "informAddBodySubscriptionFailed",
                      myOverlogId.toString(),
                      NodeIdMap.getOverlogId(senderNodeId).toString(),
                      NodeIdMap.getOverlogId(receiverNodeId).toString(),
                      ssStr};
    
    insertTuple(new Tuple(strA));
  }

//---------------------------------------------------------------------------
// get method
//---------------------------------------------------------------------------
  

  public OverlogId getMyOverlogId(){
    return myOverlogId;
  }
}
//---------------------------------------------------------------------------
/* $Log: OverlogPolicy.java,v $
/* Revision 1.31  2007/11/28 08:11:34  nalini
/* safety policy module and example checked in
/*
/* Revision 1.30  2007/09/12 19:08:12  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.29  2007/07/11 20:25:53  nalini
/* persistent tuple support added
/*
/* Revision 1.28  2007/04/02 21:06:48  zjiandan
/* fix Env.dprintln problems.
/*
/* Revision 1.27  2007/03/16 23:57:35  nalini
/* p2serverinterface added
/*
/* Revision 1.26  2007/03/09 21:46:09  nalini
/* added 2 events: informAddInvalSubscriptionFailed & informAddBodySubscriptionFailed
/*
/* Revision 1.25  2007/03/08 23:05:53  nalini
/* minor dbg changes
/*
/* Revision 1.24  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.22  2007/03/06 18:25:52  zjiandan
/* Add optimization for CatchupInvalIterator, fixed SubscriptionSet, HierInvalTarget
/* and P2Runtime problems. Add aways split when receiving subtree ISStatus in Checkpoint.
/*
/* Revision 1.21  2007/03/02 01:35:29  zjiandan
/* Yeah! Working Simplified TierStore.
/*
/* Revision 1.18  2007/02/12 06:19:49  zjiandan
/* Expose noSyncLog parameter, add more unittest for PRACTIFS, PangaeaFS.
/*
/* Revision 1.17  2007/02/01 06:12:09  zjiandan
/* Add acceptStamp to demandRead so that the sender only sends the data
/* that's at least as new as the acceptStamp.
/*
/* Revision 1.16  2007/01/12 22:29:56  nalini
/* Fixed informRemovedSubscription bug
/*
/* Revision 1.15  2007/01/12 19:12:05  nalini
/* minor updates
/*
/* Revision 1.14  2007/01/10 06:34:01  zjiandan
/* fixed "mount: localhost:practidir: can't read superblock" problem
/* and the Java P2 wrapper string problem.
/*
/* Revision 1.13  2007/01/05 01:18:41  nalini
/* support for sync with VV added
/*
/* Revision 1.12  2006/11/01 19:26:46  zjiandan
/* Integrate PangaeaFS and NFS interface.
/*
/* Revision 1.11  2006/10/25 23:34:38  zjiandan
/* Change URAOverlogNode to make it complete and self-contain to form a base
/* class for any PRACTI case-study system to start with.
/*
/* Revision 1.10  2006/10/18 22:13:21  nalini
/* added checks if overlog has started or not
/*
/* Revision 1.9  2006/10/17 23:40:55  nalini
/* minor bug fixes and separated AS into individual components in overlog tuples
/*
/* Revision 1.8  2006/10/16 20:51:49  nalini
/* P2 tuple handler support added
/*
/* Revision 1.7  2006/10/09 20:03:58  nalini
/* olg preprocess support added
/*
/* Revision 1.6  2006/10/02 23:23:39  nalini
/* synchronization support added
/*
/* Revision 1.5  2006/09/26 20:57:06  nalini
/* minor bug fix
/*
/* Revision 1.4  2006/09/26 20:24:17  nalini
/* added worker queues to handle insert and watch of tuples in overlog policy
/*
/* Revision 1.3  2006/09/26 05:34:09  nalini
/* added sync interface to policy and runtime
/*
/* Revision 1.2  2006/09/24 20:06:31  nalini
/* trying to make overlog and practi work
/*
/* Revision 1.1  2006/09/19 22:18:27  nalini
/* P2 and Practi integration
/*
*/
//---------------------------------------------------------------------------
