//---------------------------------------------------------------------------
/* P2Config.java
 *
 * Parses config file and stores it in an in-mem data structure
 * 
 * (C) Copyright 2004 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.util.*;
import java.io.*;
import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;

public class P2Config {
  static final int MAX_CONFIG_FIELDS = 12;
  static String configFile;
  static Hashtable allConfigurations;  // Hash table of config entries

//-----------------------------------------------------------------------
// Read Configuration from file
//-----------------------------------------------------------------------
  
  public static void readConfig(String configF){
    /*
     * A config file has a number of entries (one per node in the system),
     * each in the following format:
     *
     * NODEID <id>
     *   MAX_SUB_RETRIES <num>
     *   MAX_DR_RETRIES  <num>
     *   TIMEOUT_SUB     <num>
     *   TIMEOUT_DR      <num>  -- the timeout I use for my demand reads 
     *   BODY_WORKERS    <num>
     *   INVAL_WORKERS    <num>
     *   DEMAND_WORKERS    <num>
     *   INFORM_WORKERS    <num> -- # workers to send inform msg to P2Runtime
     *   COMMAND_WORKERS  <num> - # workeres to send commands to Practi
     *   ***END***
     *
     */

    int input;
    NodeId nodeId = null;
    int maxSubRetries = -1;
    int maxDRRetries = -1;
    long timeoutSub = -1;
    long timeoutDr = -1;
    int bodyWorkers = -1;
    int invalWorkers = -1;
    int demandWorkers = -1;
    int informWorkers = -1;
    int commandWorkers = -1;
    int p2serverPort = -1;
    int p2clientPort = -1;

    configFile = configF;
    allConfigurations = new Hashtable();
    LongOpt[] longopts = new LongOpt[MAX_CONFIG_FIELDS+1];
    longopts[0] = new LongOpt("NODEID", LongOpt.REQUIRED_ARGUMENT, null, 0);
    longopts[1] = new LongOpt("MAX_SUB_RETRIES", LongOpt.REQUIRED_ARGUMENT, null, 1);
    longopts[2] = new LongOpt("MAX_DR_RETRIES", LongOpt.REQUIRED_ARGUMENT, null, 2);
    longopts[3] = new LongOpt("TIMEOUT_SUB", LongOpt.REQUIRED_ARGUMENT, null, 3);
    longopts[4] = new LongOpt("TIMEOUT_DR", LongOpt.REQUIRED_ARGUMENT, null, 4);
    longopts[5] = new LongOpt("BODY_WORKERS", LongOpt.REQUIRED_ARGUMENT, null, 5);
    longopts[6] = new LongOpt("INVAL_WORKERS", LongOpt.REQUIRED_ARGUMENT, null, 6);
    longopts[7] = new LongOpt("DEMAND_WORKERS", LongOpt.REQUIRED_ARGUMENT, null, 7);
    longopts[8] = new LongOpt("INFORM_WORKERS", LongOpt.REQUIRED_ARGUMENT, null, 8);
    longopts[9] = new LongOpt("COMMAND_WORKERS", LongOpt.REQUIRED_ARGUMENT, null, 9);
    longopts[10] = new LongOpt("P2_SERVER_PORT", LongOpt.REQUIRED_ARGUMENT, null, 10);
    longopts[11] = new LongOpt("P2_CLIENT_PORT", LongOpt.REQUIRED_ARGUMENT, null, 11);
    longopts[12] = new LongOpt("***END***", LongOpt.NO_ARGUMENT, null, 12);

    String[] fileInput = parseFile(configF);
    
    Getopt g = new Getopt("Config", fileInput, "", longopts, true);
    while((input = g.getopt()) != -1){
      switch(input){
      case 0:
        int id = (new Integer(g.getOptarg())).intValue();
        nodeId = new NodeId(id);
        p2serverPort = makeDefaultP2ServerPort(nodeId);
        p2clientPort = makeDefaultP2ClientPort(nodeId);
        break;

      case 1:
        maxSubRetries = (new Integer(g.getOptarg())).intValue();
        break;
      case 2:
        maxDRRetries = (new Integer(g.getOptarg())).intValue();
        break;
      case 3:
        timeoutSub = (new Long(g.getOptarg())).longValue();
        break;
      case 4:
        timeoutDr = (new Long(g.getOptarg())).longValue();
        break;
      case 5:
        bodyWorkers = (new Integer(g.getOptarg())).intValue();
        break;
      case 6:
        invalWorkers = (new Integer(g.getOptarg())).intValue();
        break;
      case 7:
        demandWorkers = (new Integer(g.getOptarg())).intValue();
        break;
      case 8:
        informWorkers = (new Integer(g.getOptarg())).intValue();
        break;
      case 9:
        commandWorkers = (new Integer(g.getOptarg())).intValue();
        break;
      case 10:
        p2serverPort = (new Integer(g.getOptarg())).intValue();
        break;
      case 11:
        p2clientPort = (new Integer(g.getOptarg())).intValue();
        break;
      case 12:
        P2ConfigEntry p2ConfigE = new P2ConfigEntry(nodeId, maxSubRetries,
                                                    maxDRRetries, timeoutSub,
                                                    timeoutDr, 
						    bodyWorkers,
                                                    invalWorkers, 
                                                    demandWorkers, 
						    informWorkers, 
                                                    commandWorkers,
                                                    p2serverPort,
                                                    p2clientPort);
        allConfigurations.put(p2ConfigE.getNodeId(), p2ConfigE);
        break;
      case '?':
        System.err.println("The option '" + (char)g.getopt() + "' is not valid.");
        System.exit(-1);
        break;
      default:
        System.err.println("getopt() returned " + input);
        System.exit(-1);
        break;
      }
    }
  }

  private static int makeDefaultP2ServerPort(NodeId nodeId){
    return (4000+(int)nodeId.getIDint());
  }

  private static int makeDefaultP2ClientPort(NodeId nodeId){
    return (4500+(int)nodeId.getIDint());
  }

  private static String[] parseFile(String configFile){
    Vector config = new Vector();
    String line = null;
    int lineCounter = 0;
    try{
      BufferedReader in = new BufferedReader(new FileReader(configFile));
      while((line = in.readLine()) != null){
        //
        //"-" is added infront of each line to construct the format of the input arguments
        //
        StringTokenizer st = new StringTokenizer("-" + line);
        while(st.hasMoreTokens()){
          config.add(st.nextToken());
        }
      }
    } catch (Exception e){
      e.printStackTrace();
    }

    Object[] objA = config.toArray();
    String[] strA = new String[objA.length];
    for(int i = 0; i < objA.length; i++){
      strA[i] = (String) objA[i];
    }
    return strA;
  }
                  

  public final static int getMaxSubRetries(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getMaxSubRetries();
  }


  public final static int getMaxDemandReadRetries(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getMaxDemandReadRetries();
  }

  public final static long getSubTimeout(NodeId nid){
    //System.out.println("nid " + nid);
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getSubTimeout();
  }


  public final static long getDemandReadTimeout(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getDemandReadTimeout();
  }

  public final static int getBodyWorkers(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getBodyWorkers();
  }

   public final static int getInvalWorkers(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getInvalWorkers();
  }

  public final static int getDemandReadWorkers(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getDemandReadWorkers();
  }

  public final static int getInformWorkers(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getInformWorkers();
  }

  public final static int getCommandWorkers(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getCommandWorkers();
  }

  public final static int getP2ServerPort(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getP2ServerPort();
  }

  public final static int getP2ClientPort(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getP2ClientPort();
  }

  public final static String getRMIServiceName(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getRMIServiceName();
  }
  public final static String getRMIUrl(NodeId nid){
    P2ConfigEntry p2configE = (P2ConfigEntry)allConfigurations.get(nid);
    return p2configE.getRMIUrl();
  }

  public static void createEmptyConfig(){
    allConfigurations = new Hashtable();
  }

 public static void 
  addOneNodeConfig(NodeId nodeId, int subRetries, int drRetries,
                   long subTimeout, long drTimeout, int bodyW, 
                   int invalW, int demandW, 
                   int informW, int commandW) {
   addOneNodeConfig(nodeId, subRetries, drRetries, subTimeout, drTimeout,
                    bodyW, invalW, demandW, informW, commandW,
                    makeDefaultP2ServerPort(nodeId), makeDefaultP2ClientPort(nodeId));
 }

  public static void 
  addOneNodeConfig(NodeId nodeId, int subRetries, int drRetries,
                   long subTimeout, long drTimeout, int bodyW, 
                   int invalW, int demandW, 
                   int informW, int commandW,
                   int p2serverPort, int p2clientPort){
    P2ConfigEntry p2ConfigE = new P2ConfigEntry(nodeId, subRetries, drRetries,
                                                subTimeout, drTimeout, bodyW, 
                                                invalW, demandW,
                                                informW, commandW,
                                                p2serverPort, p2clientPort);

    allConfigurations.put(p2ConfigE.getNodeId(), p2ConfigE);
  }
  
  public static void
  makeOneNodeTestConfig(String p2ConfigF, long nodeId, int subRetries, int drRetries,
                        int subTimeout, int drTimeout, int bodyW, int invalW,
                        int demandW, int informW, int commandW){
    makeOneNodeTestConfig(p2ConfigF, nodeId, subRetries, drRetries, subTimeout, drTimeout,
                          bodyW, invalW, demandW, informW, commandW,
                          makeDefaultP2ServerPort(new NodeId(nodeId)), 
                          makeDefaultP2ClientPort(new NodeId(nodeId)));
  }
   
  public static void
  makeOneNodeTestConfig(String p2ConfigF, long nodeId, int subRetries, int drRetries,
                        int subTimeout, int drTimeout, int bodyW, int invalW,
                        int demandW, int informW, int commandW,
                        int p2serverPort, int p2clientPort){
    String s;
    try{
      BufferedWriter out = new BufferedWriter(new FileWriter(p2ConfigF));
        out.write("NODEID " + nodeId + "\n");
        out.write("MAX_SUB_RETRIES " + subRetries + "\n");
        out.write("MAX_DR_RETRIES " + drRetries + "\n");
        out.write("TIMEOUT_SUB " + subTimeout + "\n");
        out.write("TIMEOUT_DR " + drTimeout + "\n");
        out.write("BODY_WORKERS " + bodyW + "\n");
        out.write("INVAL_WORKERS " + invalW + "\n");
        out.write("DEMAND_WORKERS " + demandW + "\n");
        out.write("INFORM_WORKERS " + informW + "\n");
        out.write("COMMAND_WORKERS " + commandW + "\n");
        out.write("P2_SERVER_PORT " + p2serverPort + "\n");
        out.write("P2_CLIENT_PORT " + p2clientPort + "\n");
        out.write("***END***\n");
        out.close();
    } catch(Exception e){
      e.printStackTrace();
    }
  }

  public static void
  writeToFile(String newP2ConfigF){
    try{
      BufferedWriter out = new BufferedWriter(new FileWriter(newP2ConfigF));
      for(Enumeration e = allConfigurations.elements(); e.hasMoreElements(); ){
        P2ConfigEntry pe = (P2ConfigEntry)e.nextElement();
        out.write("NODEID " + pe.getNodeId() + "\n");
        out.write("MAX_SUB_RETRIES " + pe.getMaxSubRetries() + "\n");
        out.write("MAX_DR_RETRIES " + pe.getMaxDemandReadRetries() + "\n");
        out.write("TIMEOUT_SUB " + pe.getSubTimeout() + "\n");
        out.write("TIMEOUT_DR " + pe.getDemandReadTimeout() + "\n");
        out.write("BODY_WORKERS " + pe.getBodyWorkers() + "\n");
        out.write("INVAL_WORKERS " + pe.getInvalWorkers() + "\n");
        out.write("DEMAND_WORKERS " + pe.getDemandReadWorkers() + "\n");   
        out.write("INFORM_WORKERS " + pe.getInformWorkers() + "\n");
        out.write("COMMAND_WORKERS " + pe.getCommandWorkers() + "\n");
        out.write("P2_SERVER_PORT " + pe.getP2ServerPort() + "\n");
        out.write("P2_CLIENT_PORT " + pe.getP2ClientPort() + "\n");
        out.write("***END***\n");      
      }
      out.close();
    } catch(Exception e){
      e.printStackTrace();
    }
  }

  public static void main(String[] args){
    if(args.length >= 1){
      String fName = args[0];
      try{
        BufferedWriter out = new BufferedWriter(new FileWriter(fName));
        out.write("NODEID 0\n");
        out.write("MAX_SUB_RETRIES 1\n");
        out.write("MAX_DR_RETRIES 2\n");
        out.write("TIMEOUT_SUB 100\n");
        out.write("TIMEOUT_DR 200\n");
        out.write("BODY_WORKERS 5\n");
        out.write("INVAL_WORKERS 6\n");
        out.write("DEMAND_WORKERS 8\n");
        out.write("INFORM_WORKERS 9\n");
        out.write("COMMAND_WORKERS 10\n");
        out.write("P2_SERVER_PORT 5555\n");
        out.write("P2_CLIENT_PORT 5554\n");
        out.write("***END***\n");

        out.write("NODEID 10\n");
        out.write("MAX_SUB_RETRIES 11\n");
        out.write("MAX_DR_RETRIES 12\n");
        out.write("TIMEOUT_SUB 1100\n");
        out.write("TIMEOUT_DR 1200\n");
        out.write("BODY_WORKERS 15\n");
        out.write("INVAL_WORKERS 16\n");
        out.write("DEMAND_WORKERS 18\n");
        out.write("INFORM_WORKERS 19\n");
        out.write("COMMAND_WORKERS 20\n");
        out.write("P2_SERVER_PORT 5557\n");
        out.write("P2_CLIENT_PORT 5556\n");
        out.write("***END***\n");

        out.close();
        System.out.println("Written temp config file");
        
        System.out.println("Setting up config");
        setupConfig();

        System.out.println("Reading temp config file");
        P2Config.readConfig(fName);
        NodeId nid = new NodeId((long)0);
        System.out.println("Node ID: 0");
        System.out.println(P2Config.getMaxSubRetries(nid));
        System.out.println(P2Config.getMaxDemandReadRetries(nid));
        System.out.println(P2Config.getSubTimeout(nid));
        System.out.println(P2Config.getDemandReadTimeout(nid));
        System.out.println(P2Config.getBodyWorkers(nid));
        System.out.println(P2Config.getInvalWorkers(nid));
        System.out.println(P2Config.getDemandReadWorkers(nid));
        System.out.println(P2Config.getInformWorkers(nid));
        System.out.println(P2Config.getCommandWorkers(nid));
        System.out.println(P2Config.getP2ServerPort(nid));
        System.out.println(P2Config.getP2ClientPort(nid));

     

        nid = new NodeId((long)10);
        System.out.println("Node ID: 10");
        System.out.println(P2Config.getMaxSubRetries(nid));
        System.out.println(P2Config.getMaxDemandReadRetries(nid));
        System.out.println(P2Config.getSubTimeout(nid));
        System.out.println(P2Config.getDemandReadTimeout(nid));
        System.out.println(P2Config.getBodyWorkers(nid));
        System.out.println(P2Config.getInvalWorkers(nid));
        System.out.println(P2Config.getDemandReadWorkers(nid));
        System.out.println(P2Config.getInformWorkers(nid));
        System.out.println(P2Config.getCommandWorkers(nid));
        System.out.println(P2Config.getP2ServerPort(nid));
        System.out.println(P2Config.getP2ClientPort(nid));

      }catch(Exception e){
        e.printStackTrace();
      }
    }else{
      System.out.println("Usage: java P2Config <filename>");
    }
  }     

  private static void setupConfig(){
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(0),
                            "localhost",
                            9988,
                            9989,
                            9991,
                            9992,
                            9990,
                            "/tmp/ufs0",
                            "/*",
                            -1L,
                            "localhost",
                            9993,
                            9994,
                            -1,
                            1000000,
                            Long.MAX_VALUE,
                            Long.MAX_VALUE);


    Config.addOneNodeConfig(new NodeId(10),
                            "localhost",
                            9888,
                            9889,
                            9891,
                            9892,
                            9890,
                            "/tmp/ufs1",
                            "/*",
                            -1L,
                            "localhost",
                            9893,
                            9894,
                            -1,
                            1000000,
                            Long.MAX_VALUE,
                            Long.MAX_VALUE);
  }
}
//---------------------------------------------------------------------------
/* $Log: P2Config.java,v $
/* Revision 1.8  2007/03/16 23:57:35  nalini
/* p2serverinterface added
/*
/* Revision 1.7  2007/03/09 03:01:38  nalini
/* removed update workers option from P2Config
/*
/* Revision 1.6  2006/11/02 21:11:19  zjiandan
/* Fixed some bugs in URAOverlogNode and TupleHandlers.
/*
/* Revision 1.5  2006/10/02 23:23:39  nalini
/* synchronization support added
/*
/* Revision 1.4  2006/09/24 20:06:31  nalini
/* trying to make overlog and practi work
/*
*/
//---------------------------------------------------------------------------
