//---------------------------------------------------------------------------
/* P2JavaWrapper.C 
 *
 * interfaces with P2. Provides methods to install overlog files, watch
 * tuples, and receive tuples
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

#include "P2JavaWrapper.h" 
#include <jni.h>
#include <iostream>

#ifndef __NOP2__

#if HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include "p2.h"

#include "tuple.h"
#include "plumber.h"
#include "val_str.h"

JNIEnv *_jenv;
jobject _jobj;
P2::DataflowHandle* dataflowPtr;
bool verbose = false;

#endif


#ifndef __NOP2__

//---------------------------------------------------------------------------
//  Method called when a "watched" tuple is received
//---------------------------------------------------------------------------

void watch(P2::DataflowHandle dataflow, TuplePtr tp)
{


  jclass cls =_jenv->GetObjectClass(_jobj);
  jmethodID mid = _jenv->GetMethodID(cls, "received_tuple", "(Ljava/lang/String;)V");

  std::string tpCStr = tp->toString();
  const char* cStr = tpCStr.c_str(); 
  jstring tpStr = _jenv->NewStringUTF(cStr);

  if(verbose) {
    std::cout<<"P2JavaWrapper: received tuple:" << cStr << std::endl;
  }

  _jenv->CallVoidMethod(_jobj, mid, tpStr);
  
}

#endif

//---------------------------------------------------------------------------
// install an overlog node
//---------------------------------------------------------------------------


JNIEXPORT void JNICALL Java_P2JavaWrapper_jni_1installOverlog (JNIEnv *env, jobject obj, 
							       jstring jFilename, jstring jHostname, 
							       jstring jPort){
#ifdef __NOP2__
  std::cout<<"P2 Not Supported in this platform"<<std::endl;
  
#else

  const char *filenameC = env->GetStringUTFChars(jFilename, NULL);
  const char *hostnameC = env->GetStringUTFChars(jHostname, NULL);
  const char *portC = env->GetStringUTFChars(jPort, NULL);
  
  string filename(filenameC);
  string hostname(hostnameC);
  int port = atoi(portC);

  if(verbose){
    std::cout << "Filename: "<< filename <<std::endl;
    std::cout << "Hostname: "<< hostname <<std::endl;
    std::cout << "Port: "<< port <<std::endl;
  }  

  std::vector< std::string > definitions;

  if(verbose) std::cout<<"Preprocessing File"<<std::endl;


  string program = P2::preprocessReadOverLogProgram(filename, 
                                                    filename,
                                                    definitions);

  
  std::ostringstream myAddressBuf;
  myAddressBuf <<  hostname << ":" << port;
  std::string myAddress = myAddressBuf.str();

  if(verbose) std::cout<<"Creating dataflow"<<std::endl;
 

  P2::DataflowHandle handle =
    P2::createDataflow("overlog",
                       myAddress,
                       port,
                       filename,
                       program, 
                       false,
                       false,
                       false);


  dataflowPtr = new P2::DataflowHandle(handle);

 if(verbose) std::cout<<"finished creating dataflow"<<std::endl;
 
  
  env->ReleaseStringUTFChars(jFilename, filenameC);
  env->ReleaseStringUTFChars(jHostname, hostnameC);
  env->ReleaseStringUTFChars(jPort, portC);

#endif
  return;

}


//---------------------------------------------------------------------------
//  Informs the data flow about which tuple to watch 
//---------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_P2JavaWrapper_jni_1watch (JNIEnv *env, jobject obj, jstring jTupleName){

#ifdef __NOP2__
  //std::cout<<"P2 Not Supported in this platform"<<std::endl;

#else 
  const char *tupleName = env->GetStringUTFChars(jTupleName, NULL);
  
  if(verbose) std::cout<< "call to watch : "<< tupleName << std::endl;

  P2::subscribe(*dataflowPtr, tupleName, boost::bind(&watch, *dataflowPtr, _1));
  _jenv=env;
  _jobj=obj;

  env->ReleaseStringUTFChars(jTupleName, tupleName);

#endif
  return;
}


//---------------------------------------------------------------------------
//  inserts a tuple into the data flow
//---------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_P2JavaWrapper_jni_1insert (JNIEnv *env, jobject obj, jobjectArray jTupleStrA){
 
#ifdef __NOP2__
  // std::cout<<"P2 Not Supported in this platform"<<std::endl;
#else
  int length = env->GetArrayLength(jTupleStrA);
  TuplePtr tp = Tuple::mk();

  if(verbose)
  std::cout<<"P2JavaWrapper.C: Trying to insert : <";

    jstring string = (jstring)env->GetObjectArrayElement(jTupleStrA, (jsize) 0 );
    const char *str = env->GetStringUTFChars(string, NULL);
    tp->append(Val_Str::mk(str));
    if(verbose) std::cout << str;
    env->ReleaseStringUTFChars(string, str);

  for( int i = 1; i < length; i++){
    jstring string = (jstring)env->GetObjectArrayElement(jTupleStrA, (jsize) i );
    const char *str = env->GetStringUTFChars(string, NULL);
    tp->append(Val_Str::mk(str));
    if(verbose) std::cout << "," << str;
    env->ReleaseStringUTFChars(string, str);
  }
  if(verbose) std::cout << ">";
  tp->freeze();

  P2::injectTuple(*dataflowPtr, tp, &P2::nullCallback);

  if(verbose)std::cout<<" .... done "<<std::endl;

#endif

  return;
}


//---------------------------------------------------------------------------
//  starts the dataflow
//---------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_P2JavaWrapper_jni_1runOverlog(JNIEnv *env, jobject obj){
#ifdef __NOP2__
  //std::cout<<"P2 Not Supported in this platform"<<std::endl;
#else
  try {
    P2::run();
  } catch (Element::Exception e) {
    TELL_ERROR << "Caught an Element exception '"
               << e.toString()
               << "'\n";
  }
#endif

  return;
}


/*
//---------------------------------------------------------------------------
/* $Log: P2JavaWrapper.C,v $
/* Revision 1.14  2007/09/13 16:02:19  nalini
/* fixed make problem
/*
/* Revision 1.13  2007/09/12 19:08:12  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.12  2007/03/01 08:12:37  nalini
/* added maxBoundHop support for MultiObjWrites
/*
/* Revision 1.11  2007/02/16 16:42:46  nalini
/* debug code added
/*
/* Revision 1.10  2007/01/10 06:34:01  zjiandan
/* fixed "mount: localhost:practidir: can't read superblock" problem
/* and the Java P2 wrapper string problem.
/*
/* Revision 1.9  2007/01/10 02:02:24  zjiandan
/* Fixed rmiregistry problems.
/*
/* Revision 1.8  2006/11/16 03:28:08  nalini
/* junit fix
/*
/* Revision 1.7  2006/11/02 20:16:29  nalini
/* fixed filename string bug and reports error if P2 not supported
/*
/* Revision 1.6  2006/11/01 20:54:20  zjiandan
/* fixed bug with filename string
/*
/* Revision 1.5  2006/10/17 23:40:55  nalini
/* minor bug fixes and separated AS into individual components in overlog tuples
/*
/* Revision 1.4  2006/10/11 19:49:33  nalini
/* updated make file to fix junit error
/*
/* Revision 1.3  2006/10/09 20:03:58  nalini
/* olg preprocess support added
/*
/* Revision 1.2  2006/09/24 20:06:31  nalini
/* trying to make overlog and practi work
/*
/* Revision 1.1  2006/09/19 22:18:27  nalini
/* P2 and Practi integration
/*
 */
//---------------------------------------------------------------------------
