//---------------------------------------------------------------------------
/*  P2JavaWrapper.java
 * 
 *  Interfaces to P2.  Provides methods to install the overlog file,
 *  insert watch tuples, and to receive tuples.
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------
import java.lang.Thread;
import java.io.*;

public class P2JavaWrapper {
 
  static {
    System.loadLibrary("P2JavaWrapper");
  }

  private Watcher watcher;
  private String olgFilename;
  private String olgHostname;
  private String olgPort;
  private String[] watchTuples;
  
  private boolean dbg = false;

//---------------------------------------------------------------------------
// Constructor
//---------------------------------------------------------------------------
  public P2JavaWrapper(String olgFilename, 
                       String olgHostname,
                       String olgPort){
    this.olgFilename = olgFilename;
    this.olgHostname = olgHostname;
    this.olgPort = olgPort;
    this.watchTuples = null;
  }

  public void setWatchTuples(String[] watchTuples) {
    this.watchTuples = watchTuples;
  }

  public void setWatcher(Watcher watcher){
    this.watcher = watcher;
  }

//---------------------------------------------------------------------------
// JNI methods which interface with P2 System
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Installs the overlog file at a certain host
//---------------------------------------------------------------------------

  public native void jni_installAndStartOverlog(String filename, 
                                                String hostname, 
                                                String port,
                                                String[] watchTuples);
//---------------------------------------------------------------------------
// Installs the overlog file at a certain host
//---------------------------------------------------------------------------

  public native void jni_installOverlog(String filename, 
                                        String hostname, 
                                        String port);
//---------------------------------------------------------------------------
// Tell the P2 System to "watch" a tuple
//---------------------------------------------------------------------------
  public native void jni_watch(String tupleName);

//---------------------------------------------------------------------------
// runs the overlog file once it has been installed
//---------------------------------------------------------------------------
  public native void jni_runOverlog();


//---------------------------------------------------------------------------
// inserts a tuple into the Overlog dataflow
//---------------------------------------------------------------------------
  public native void jni_insert(String[] tupleStrA);


    
//---------------------------------------------------------------------------
// if a tuple is received, informs the watcher thread.
// Note: this method is called by the native code.
//
// **** Do Not Change Interface w/o changing P2JavaWrapper.C ****
//---------------------------------------------------------------------------
  public void received_tuple(String tupleStr){
    
    if(dbg) System.out.println("P2JavaWrapper: Received Tuple " + tupleStr);
    if(watcher != null ){
      watcher.receivedTuple(new Tuple(tupleStr));
    }
    else {
      System.out.println("P2JavaWrapper: Received Tuple " + tupleStr);
    }
  }


//---------------------------------------------------------------------------
// startOverlog: 
// - installs the overlog file
// - installs the watch tuples
// - starts the overlog dataflow
//---------------------------------------------------------------------------


  public void startOverlog(){

    jni_installOverlog(olgFilename, olgHostname, olgPort);
    
    if(watchTuples != null) {
      for(int i = 0; i < watchTuples.length; i++){
        jni_watch(watchTuples[i]);
      }
    }

    jni_runOverlog();
     
  }

//---------------------------------------------------------------------------
//  insert tuple - inserts tuple in the overlog 
//---------------------------------------------------------------------------
  public void insertTuple(Tuple tuple){
    Env.dprintln(dbg, "P2JavaWrapper: inserting Tuple " + tuple);
    jni_insert(tuple.getStringA());
  }
 

//---------------------------------------------------------------------------
//  main method - install and runs an overlog file
//  Note: it does not handle watch tuples or insertion of tuples
//---------------------------------------------------------------------------
  public static void main(String[] args) {
    if(args.length < 3) {
      System.out.println("Usage: java P2JavaWrapper <overlogfile> <hostname> <port>");
      System.exit(-1);
    }
    
    P2JavaWrapper javaWrapper = new P2JavaWrapper(args[0], args[1], args[2]);
    javaWrapper.startOverlog();
  }
}

//---------------------------------------------------------------------------
/* $Log: P2JavaWrapper.java,v $
/* Revision 1.7  2007/10/24 03:48:41  zjiandan
/* *** empty log message ***
/*
/* Revision 1.6  2007/10/19 23:08:40  zjiandan
/* Coda demo for SOSP poster 2007.
/*
/* Revision 1.5  2007/09/12 19:08:12  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.4  2007/02/16 16:42:46  nalini
/* debug code added
/*
/* Revision 1.3  2006/10/16 20:51:49  nalini
/* P2 tuple handler support added
/*
/* Revision 1.2  2006/10/09 20:03:58  nalini
/* olg preprocess support added
/*
/* Revision 1.1  2006/09/19 22:18:27  nalini
/* P2 and Practi integration
/*
 */
//---------------------------------------------------------------------------

