//---------------------------------------------------------------------------
/*  P2Runtime.java
 * 
 *  Interfaces to Practi. Accepts P2 directives and carries out semantics
 *  of the directives.  Informs P2 layer of important events
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

//package p2;

import java.io.*;
import java.lang.*;

public class P2Runtime {

  private NodeId myNodeId;
  private Controller controller;
  private Core core;
  private RMIClient rmiClient;
  private RMIServerImpl rmiServer;
  private SocketServer socketServer;
  private Status status;
  private Policy policy;

  private boolean dbg=false;  // prints a lot of debugging out put
 
  
  //
  //  verbose output for different parts of the Runtime 
  //
  public static final boolean verboseDR = false;  // demand reads
  public static final boolean verboseBodySub = false; // body subscriptions
  public static final boolean verboseInvalSub = true; // inval subscriptions
  public static final boolean verboseLocal = false; // local events
  public static final boolean verboseRecvInval = false; //  received Inval

  //
  //  verbose output for handlers
  //
  public static final boolean verboseHandlers = false; 

//---------------------------------------------------------------------------
//  Constructor
//---------------------------------------------------------------------------

  
  public 
  P2Runtime(String localConfigFile, String p2ConfigFile, NodeId myNodeId, 
	    boolean cleanDb, Policy policy, boolean noSyncLog){
      Config.readConfig(localConfigFile);
      P2Config.readConfig(p2ConfigFile);
      setup(myNodeId, cleanDb, policy, noSyncLog);
  }

  public 
  P2Runtime(NodeId myNodeId, 
	    boolean cleanDb, Policy policy, boolean noSyncLog){
    // assumes config and p2 config have already been set up.
    setup(myNodeId, cleanDb, policy, noSyncLog);
  }

  private void
  setup(NodeId myNodeId, 
        boolean cleanDb, Policy policy, boolean noSyncLog){
    try{

      this.myNodeId = myNodeId;
      this.rmiClient = new RMIClient();
      //
      // set filterOn to true
      // if cleanDb = true, cleans the presistent state in the core
      //
      boolean filterOn = true;
      this.core = new Core(rmiClient, filterOn, cleanDb, myNodeId, noSyncLog);
      this.status = new Status();
      this.policy = policy;
      this.controller = new RuntimeController(this, this.status);
      this.rmiServer = new RMIServerImpl(core, controller);
      this.rmiServer.start();
      this.socketServer = new SocketServer(core, controller);
      this.core.recoverLocalState(rmiServer);

    } catch(Exception e){
      e.printStackTrace();
    }
  }

  public void start(){
    try{
       
      //
      // set up pending workers
      //

      assert(P2Config.getInformWorkers(myNodeId) > 0) : "No inform workers in P2Config";
      for(int i = 0; i < P2Config.getInformWorkers(myNodeId); i++){
        InformMsgWorkerThread pThread = 
          new InformMsgWorkerThread(this, this.status);
        pThread.start();
      }
      
      assert(P2Config.getCommandWorkers(myNodeId) > 0) : "No command workers in P2Config";
      for(int i = 0; i < P2Config.getCommandWorkers(myNodeId); i++){
        CmdWorkerThread pThread = 
          new CmdWorkerThread(this.rmiClient, this.core, this.status);
        pThread.start();
      }
 
      assert(P2Config.getBodyWorkers(myNodeId) > 0) : "No Body workers in P2Config";  
      for(int i = 0; i < P2Config.getBodyWorkers(myNodeId); i++){
        PendingBodySubscriptionWorkerThread pThread = 
          new PendingBodySubscriptionWorkerThread(this.rmiClient, this.status, this.myNodeId, this, this.core);
        pThread.start();
      }
 
      assert(P2Config.getInvalWorkers(myNodeId) > 0) : "No Inval workers in P2Config";  
      for(int i = 0; i < P2Config.getInvalWorkers(myNodeId); i++){
        PendingInvalSubscriptionWorkerThread pThread = 
          new PendingInvalSubscriptionWorkerThread(this.rmiClient, this.status, this.myNodeId, this, this.core);
        pThread.start();
      }
 
      assert(P2Config.getDemandReadWorkers(myNodeId) > 0) : "No DemandRead workers in P2Config";  
      for(int i = 0; i < P2Config.getDemandReadWorkers(myNodeId); i++){
        DemandReadWorkerThread dThread = 
          new DemandReadWorkerThread(this.rmiClient, this, this.status, this.myNodeId);
        dThread.start();
      }
   
      System.err.println("P2Runtime: started");
    }catch(Exception e){
      System.err.println(e);
      e.printStackTrace();
    }
  }

  
  //---------------------------------------------------------------------------
  // Attempt clean shutdown. Mostly used for unit tests so that
  // we can run multiple tests sequentially within same process.
  // Not extensively tested
  //---------------------------------------------------------------------------
  public synchronized void shutdown(){
    core.syncStateToDisk();
    core.close();
    core = null;
    rmiServer.close();
    rmiServer = null;
    socketServer.shutdown();
    socketServer = null;
  }

 //---------------------------------------------------------------------------
  // accessor methods
  //---------------------------------------------------------------------------
  public Core
  getCore(){
    return this.core;
  }

  public Controller
  getController(){
    return this.controller;
  }

  public NodeId
  getMyNodeId(){
    return this.myNodeId;
  }

  //---------------------------------------------------------------------------
  // methods to hook to p2
  //---------------------------------------------------------------------------
  public synchronized void
  informLocalReadImprecise(ObjId objId, long offset, long length){
    if(dbg || verboseLocal){
      Env.dprintln(dbg || verboseLocal, "P2Runtime :: informLocalReadImprecise " + 
		   objId + ":" + offset + ":" + length);
    }
    policy.informLocalReadImprecise(objId, offset, length);
  }

  public synchronized void
  informLocalReadInvalid(ObjId objId, long offset, long length, AcceptStamp as){
    if(dbg || verboseLocal){
      Env.dprintln(dbg || verboseLocal, "P2Runtime :: informLocalReadInvalid " + 
		   objId + ":" + offset + ":" + length + ":" + as);
    }
    policy.informLocalReadInvalid(objId, offset, length, as);
  }
  
  public synchronized void
  informLocalWrite(ObjId objId, long offset, long length, AcceptStamp as, 
                   boolean isBound, boolean isEmbargoed){
    if(dbg || verboseLocal){
      Env.dprintln(dbg || verboseLocal, "P2Runtime :: informLocalWrite " + 
		   objId + ":" + offset + ":" + length + ":" + as + ":" + isBound + ":" + as);
    }
    policy.informLocalWrite(objId, offset, length, as, isBound, isEmbargoed);

  }


  public synchronized void
  informLocalDelete(ObjId objId){
    if(dbg || verboseLocal){
      Env.dprintln(dbg || verboseLocal, "P2Runtime :: informLocalDelete " +
		   objId); 
    }
   
    policy.informLocalDelete(objId);
  }
  
   
  public synchronized void
  informReceiveInval(NodeId senderNodeId, ObjId objId, long offset, long length,
                     AcceptStamp as, boolean isBound, boolean isEmbargoed){
    if(dbg || verboseRecvInval){
      Env.dprintln(dbg || verboseRecvInval, "P2Runtime :: informReceiveInval "  +
		   senderNodeId + ":" + objId + ":" + offset  + ":" + length + ":" +
		   as + ":" + isBound + ":" + isEmbargoed) ;
    }    
   policy.informReceiveInval(senderNodeId, objId, offset, length, as, isBound, isEmbargoed);
  }
 
  public synchronized void
  recvSyncReply(NodeId sender, AcceptStamp acceptStamp) {
    if(dbg){
      Env.dprintln(dbg,  "P2Runtime :: recvSyncReply " +
		   sender + ":" + acceptStamp);
    }
    policy.recvSyncReply(sender, acceptStamp);
  }

  //---------------------------------------------------------------------------
  // Practi events which require attention
  //---------------------------------------------------------------------------
  
  public synchronized void
  informSubscribeInvalSucceeded(NodeId senderNodeId, SubscriptionSet ss){
    //
    // It should not be in the any of the queues, but we will
    // try removing it just to be sure
    //
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "P2Runtime :: informSubscribeInvalSucceeded " +
                 senderNodeId + " : " + ss);
    }
    if(ss.isEmpty()) {
      return;
     }
    PendingInvalSubscriptionQueue pInvalQueue = status.getPendingInvalSubscriptionQueue();
    
    
    Subscription sub = new Subscription(senderNodeId, myNodeId, ss);
    pInvalQueue.remove(sub);
    status.getEstablishedInvalSubscriptions().add(sub);
    policy.informAddedInvalSubscription(senderNodeId, myNodeId, ss);
      
  }
  

 public synchronized void
  informSubscribeInvalAttached(NodeId senderNodeId, SubscriptionSet ss){
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "P2Runtime :: invformSubscribeInvalAttached " +
		   senderNodeId + " : " + ss);
    }
    if(ss.isEmpty()) {
      return;
    }
    policy.informAttachedInvalSubscription(senderNodeId, myNodeId, ss);
  }
  

  public synchronized void
  informSubscribeInvalFailed(NodeId senderNodeId, SubscriptionSet ss){
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "P2Runtime :: informSubscribeInvalFailed: " +
		   "sender=" + senderNodeId + " ss=" + ss);
    }
    if(ss.isEmpty()) {
      return;
    }
    Subscription sub = new Subscription(senderNodeId, myNodeId, ss);
 
    // Remove from established inval subscription table &
    // from pending inval subscription queue
    // Note: it can probably only be in one of them
    //
    status.getEstablishedInvalSubscriptions().remove(sub);    
    status.getPendingInvalSubscriptionQueue().remove(sub);
    policy.informRemovedInvalSubscription(senderNodeId, myNodeId, ss);    

  }

  public synchronized void
  informSubscribeInvalFailedMaxRetries(NodeId senderNodeId, SubscriptionSet ss){
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "P2Runtime :: informSubscribeInvalFailedMaxRetries: " +
		   "sender=" + senderNodeId + " ss=" + ss);
    }
    if(ss.isEmpty()) {
      return;
    }
    Subscription sub = new Subscription(senderNodeId, myNodeId, ss);
 
    // Remove from established inval subscription table &
    // from pending inval subscription queue
    // Note: shouldn't be in either one of them, but just to be safe
    //
    status.getEstablishedInvalSubscriptions().remove(sub);    
    status.getPendingInvalSubscriptionQueue().remove(sub);
    policy.informAddInvalSubscriptionFailed(senderNodeId, myNodeId, ss);    

  }

  public synchronized void
  informInvalStreamTerminated(NodeId senderNodeId, SubscriptionSet ss){
    if(dbg || verboseInvalSub){
    Env.dprintln(dbg || verboseInvalSub, "P2Runtime :: informInvalStreamTerminated: " + 
                 "sender=" + senderNodeId  + " ss=" + ss);
    }
    if(ss.isEmpty()) {
      return;
    }
    
    Subscription sub = new Subscription(senderNodeId, myNodeId, ss);
 
    // Remove from established inval subscription table &
    // from pending inval subscription queue
    // Note: it can probably only be in one of them
    //
    status.getEstablishedInvalSubscriptions().remove(sub);    
    status.getPendingInvalSubscriptionQueue().remove(sub);
    policy.informRemovedInvalSubscription(senderNodeId, myNodeId, ss);    
  }


  public synchronized void
  informOutgoingSubscribeInvalInitiated(NodeId targetNodeId, SubscriptionSet ss){
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "P2Runtime :: informOutgoingSubscribeInvalInitiated: " + 
		   "recver=" + targetNodeId  + " ss=" + ss);
    }
    if(ss.isEmpty()) {
      return;
    }
    
    Subscription sub = new Subscription(myNodeId, targetNodeId, ss);
    status.getEstablishedOutgoingInvalSubscriptions().add(sub);

    policy.informAddedOutgoingInvalSubscription(myNodeId, targetNodeId, ss);
     
  }

  public synchronized void
  informOutgoingSubscribeInvalTerminated(NodeId targetNodeId, SubscriptionSet ss){
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "P2Runtime :: informOutgoingSubscribeInvalTerminated: " + 
                 "recver=" + targetNodeId  + " ss=" + ss);
    }
    if(ss.isEmpty()) {
      return;
    }
    
    status.getEstablishedOutgoingInvalSubscriptions().remove(myNodeId, 
                                                             targetNodeId,
                                                             ss); 
    policy.informRemovedOutgoingInvalSubscription(myNodeId,
						  targetNodeId,
						  ss);
  }    


//----------------------------------------------------------------------------
// Body Subscriptions
//----------------------------------------------------------------------------

  public synchronized void
  informSubscribeBodySucceeded(NodeId senderNodeId, SubscriptionSet ss){
    //
    // it remove it the pending body subscription queue 
    // and report it to policy and add it to the established subscription table
    //
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "P2Runtime :: informSubscribeBodySucceeded "+
		   senderNodeId + ":" + ss);
    }
    PendingBodySubscriptionQueue pBodyQueue = status.getPendingBodySubscriptionQueue();
    
    
    Subscription sub = new Subscription(senderNodeId, myNodeId, ss);
    pBodyQueue.remove(sub);
    status.getEstablishedBodySubscriptions().add(sub);
    policy.informAddedBodySubscription(senderNodeId, myNodeId, ss);
  }


  public synchronized void
  informSubscribeBodyRemoved(NodeId senderNodeId, SubscriptionSet ss){
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "P2Runtime :: informSubscribeBodyRemoved " +
		   senderNodeId + ":" + ss);
    // note ss can be null
    }
    Subscription sub = new Subscription(senderNodeId, myNodeId, ss);
    //
    // Remove from established body subscription table &
    // from pending body subscription queue
    // Note: it can probably only be in one of them
    //
    if(ss != null) {
      status.getEstablishedBodySubscriptions().remove(sub);
      status.getPendingBodySubscriptionQueue().remove(sub);
      policy.informRemovedBodySubscription(senderNodeId, myNodeId,ss);
    } else {
      Subscription bSub = status.getEstablishedBodySubscriptions().remove(sub);
      if (bSub != null) {
        policy.informRemovedBodySubscription(senderNodeId, myNodeId,bSub.getSS()); 
      }  
      bSub = status.getPendingBodySubscriptionQueue().remove(sub);
      if (bSub != null) {
        policy.informRemovedBodySubscription(senderNodeId, myNodeId,bSub.getSS()); 
      }  
    }
  }

  public synchronized void
  informBodyStreamTerminated(NodeId senderNodeId){
    //
    // same logic as subscribeBodyRemoved
    //
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "P2Runtime:: informSubscribeBodyTerminated " +
		   senderNodeId);
    }
    informSubscribeBodyRemoved(senderNodeId, null);
  }

 public synchronized void
 informSubscribeBodyFailedMaxRetries(NodeId senderNodeId, SubscriptionSet ss){
   if(dbg || verboseBodySub){
     Env.dprintln(dbg || verboseBodySub, "P2Runtime :: informSubscribeBodyFailedMaxRetries" +
		  senderNodeId + ":" + ss);
    // note ss can be null
   }
    Subscription sub = new Subscription(senderNodeId, myNodeId, ss);
    //
    // Remove from established body subscription table &
    // from pending body subscription queue
    // Note: it shouldn't be in either one, but just to be safe.
    //
    if(ss != null) {
      status.getEstablishedBodySubscriptions().remove(sub);
      status.getPendingBodySubscriptionQueue().remove(sub);
      policy.informAddBodySubscriptionFailed(senderNodeId, myNodeId,ss);
    } else {
      Subscription bSub = status.getEstablishedBodySubscriptions().remove(sub);
      if (bSub != null) {
        policy.informAddBodySubscriptionFailed(senderNodeId, myNodeId,bSub.getSS()); 
      }  
      bSub = status.getPendingBodySubscriptionQueue().remove(sub);
      if (bSub != null) {
        policy.informAddBodySubscriptionFailed(senderNodeId, myNodeId,bSub.getSS()); 
      }  
    }
  }



  public synchronized void
  informOutgoingSubscribeBodyInitiated(NodeId targetNodeId, SubscriptionSet ss){
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "P2Runtime :: informOutgoingSubscribeBodyInitiated "+
		   targetNodeId + ":" + ss);
    }
    Subscription sub = new BodySubscription(myNodeId, targetNodeId, ss);
    status.getEstablishedOutgoingBodySubscriptions().add(sub);
    policy.informAddedOutgoingBodySubscription(myNodeId, targetNodeId, ss);
     
  
  }

  public synchronized void
  informOutgoingSubscribeBodyTerminated(NodeId targetNodeId, SubscriptionSet ss){
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "P2Runtime :: informOutgoingSubscribeBodyTerminated "+
		   targetNodeId + ":" + ss);
    }
    status.getEstablishedOutgoingBodySubscriptions().remove(myNodeId, 
                                                            targetNodeId, 
                                                            ss);
    policy.informRemovedOutgoingBodySubscription(myNodeId, 
						 targetNodeId,
						 ss);
  }

  //---------------------------------------------------------------------------
  // Handling Body Subscriptions
  //---------------------------------------------------------------------------
  
  public synchronized void
  addBodySubscription(NodeId senderNode, SubscriptionSet ss){
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "P2Runtime :: addBodySubscription Called " +
		   senderNode + ":" + ss);
    }
    status.getPendingBodySubscriptionQueue().add(senderNode, myNodeId, ss);
    if(dbg){
      Env.dprintln(dbg,  "P2Runtime :: PendingBodySubsQueue = "+
		   status.getPendingBodySubscriptionQueue().toString());
    }
  }

  public synchronized void 
  removeBodySubscription(NodeId senderNode, SubscriptionSet ss){
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "P2Runtime :: removeBodySubscription Called " +
		   senderNode + ":" + ss);
    }
    try{
      status.getEstablishedBodySubscriptions().remove(senderNode, myNodeId, ss);
      RemoveSubscribeBodyCmd cmd = new RemoveSubscribeBodyCmd(senderNode,
                                                              myNodeId,
                                                              ss);
      status.getPendingPractiCommandQueue().add(cmd);
    }catch(Exception e){
      e.printStackTrace();
    }
  }

  public synchronized void
  removeOutgoingBodySubscription(NodeId targetNodeId, SubscriptionSet ss){
    if(dbg || verboseBodySub){
      Env.dprintln(dbg || verboseBodySub, "P2Runtime :: removeOutgoingBodySubscription Called " +
		   targetNodeId + ":" + ss);
    }
    try{
      status.getEstablishedOutgoingBodySubscriptions().remove(myNodeId,
                                                              targetNodeId,
                                                              ss);
      RemoveSubscribeBodyCmd cmd = new RemoveSubscribeBodyCmd(myNodeId,
                                                              targetNodeId,
                                                              ss);
      status.getPendingPractiCommandQueue().add(cmd);
 
    }catch(Exception e){
      e.printStackTrace();
    }
  }



  //---------------------------------------------------------------------------
  // Handling Inval Subscriptions
  //---------------------------------------------------------------------------
  
  public synchronized void
  addInvalSubscription(NodeId senderNode, SubscriptionSet ss, boolean catchupWithCP){
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "P2Runtime :: addInvalSubscription Called " +
		   senderNode + ":" + ss + ":" + catchupWithCP);
    }
    status.getPendingInvalSubscriptionQueue().add(senderNode, myNodeId, ss, catchupWithCP);
  }

  public synchronized void 
  removeInvalSubscription(NodeId senderNode, SubscriptionSet ss){
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "P2Runtime :: removeInvalSubscription Called " +
		   senderNode + ":" + ss);
    }
    try{
      status.getEstablishedInvalSubscriptions().remove(senderNode, myNodeId, ss);
      RemoveSubscribeInvalCmd cmd = new RemoveSubscribeInvalCmd(senderNode,
                                                                myNodeId,
                                                                ss);
      status.getPendingPractiCommandQueue().add(cmd);
    }catch(Exception e){
      e.printStackTrace();
    }
  }

  public synchronized void
  removeOutgoingInvalSubscription(NodeId targetNodeId, SubscriptionSet ss){
    if(dbg || verboseInvalSub){
      Env.dprintln(dbg || verboseInvalSub, "P2Runtime :: removeOutgoingInvalSubscription Called " +
		   targetNodeId + ":" + ss);
    }
    try {
      status.getEstablishedOutgoingInvalSubscriptions().remove(myNodeId,
                                                               targetNodeId,
                                                               ss);
      
      RemoveSubscribeInvalCmd cmd = new RemoveSubscribeInvalCmd(myNodeId,
								targetNodeId,
								ss);
      status.getPendingPractiCommandQueue().add(cmd);
      
    }catch(Exception e){
      e.printStackTrace();
    }
  }

  //---------------------------------------------------------------------------
  // Handling Demand Reads
  //---------------------------------------------------------------------------
   
  public synchronized void 
      demandRead(NodeId nodeId, ObjId objId, long offset, long length, AcceptStamp as){
        boolean print = (verboseDR || dbg); 
	if(print){
	  Env.dprintln(print, "P2Runtime :: demandRead request called " +
		       nodeId+":"+ objId +":"+offset+":"+length+":"+as);
	}
        status.getPendingDemandReadQueue().add(nodeId, myNodeId, objId, offset,
						 length, as);
  }


  public synchronized void 
  informReceiveDemandReply(NodeId senderNodeId, ObjId objId, long offset,
                           long length, long requestId, AcceptStamp as) {
    boolean print = (verboseDR || dbg); 
    if(print){
      Env.dprintln(print, "P2Runtime :: informReceiveDemandReadReply  for " +
		   senderNodeId+":"+ objId +":"+offset+":"+length+":"+as);
    }
    DemandRead dr = status.getPendingDemandReadQueue().removeRequest(senderNodeId, myNodeId, objId, 
                                                                     offset, length,
                                                                     requestId, as);
    if(dr != null) {
      policy.informDemandReadSuccess(senderNodeId, objId, offset, length, as);
    }
      
  }

  public synchronized void 
  informDemandReadFailedMiss(DemandRead dr){
    boolean print = (verboseDR || dbg); 
    if(print){
      Env.dprintln(print, "P2Runtime :: informDemandReadFailed Miss   for " + dr);
    }
    policy.informDemandReadFailedMiss(dr.getSenderNode(), dr.getObjId(), 
                                      dr.getOffset(), dr.getLength(), dr.getAcceptStamp());

  }

  public synchronized void 
  informDemandReadFailedMaxRetries(DemandRead dr){
    boolean print = (verboseDR || dbg); 
    if(print){
      Env.dprintln(print, "P2Runtime :: informDemandReadFailed MaxRetries   for " + dr);
    }
    policy.informDemandReadFailedMaxRetries(dr.getSenderNode(), dr.getObjId(), 
					    dr.getOffset(), dr.getLength(), dr.getAcceptStamp());
  }
     
  //---------------------------------------------------------------------------
  // Other Commands
  //---------------------------------------------------------------------------
  
  public synchronized void   
  pushUpdate(NodeId targetNodeId, ObjId objId, long offset, long length){
    if(dbg){
      Env.dprintln(dbg, "P2Runtime :: pushUpdate called for " +
		   targetNodeId + ":" + objId + ":" + offset + ":" + length);  
    } 
    PushUpdateCmd cmd = new PushUpdateCmd(objId, offset, length, myNodeId, targetNodeId);
    status.getPendingPractiCommandQueue().add(cmd);
  }
    
  public synchronized void 
  unbind(ObjId objId, long offset, long length, AcceptStamp as){
    if(dbg){
      Env.dprintln(dbg, "P2Runtime :: unbind called for " +
		   objId + ":" + offset + ":" + length + ":" + as); 
    }  
    UnbindCmd cmd = new UnbindCmd(objId, offset, length, as);
    status.getPendingPractiCommandQueue().add(cmd);
  }

  public synchronized void 
  debargo (ObjId objId, long offset, long length, AcceptStamp as) {
    if(dbg){
      Env.dprintln(dbg, "P2Runtime :: dbargocalled for " +
		   objId + ":" + offset + ":" + length + ":" + as);
    }   
    DebargoCmd cmd = new DebargoCmd(objId, offset, length, as);
    status.getPendingPractiCommandQueue().add(cmd);
  }
  
  public synchronized void
  requestSync(NodeId remoteNodeId, AcceptStamp as){
    if(dbg){
      Env.dprintln(dbg, "P2Runtime :: requestSync called for " +
		   remoteNodeId + ":" + as);  
    }
    RequestSyncCmd cmd = new RequestSyncCmd(remoteNodeId, myNodeId,  as);
    status.getPendingPractiCommandQueue().add(cmd);
  }

 public synchronized void
  requestSync(NodeId remoteNodeId, AcceptVV aVV){
    if(dbg){
      Env.dprintln(dbg, "P2Runtime :: requestSync called for " +
		   remoteNodeId + ":" + aVV); 
    } 
    RequestSyncCmd cmd = new RequestSyncCmd(remoteNodeId, myNodeId,  aVV);
    status.getPendingPractiCommandQueue().add(cmd);
  }

  public synchronized void
  sync(NodeId remoteNodeId){
    if(dbg){
      Env.dprintln(dbg, "P2Runtime :: sync called for " +
		   remoteNodeId);  
    }
    requestSync(remoteNodeId, core.getCurrentVV());
  }



  //---------------------------------------------------------------------------
  // Retrieving the list of subscriptions
  //---------------------------------------------------------------------------
  
  public SubscriptionTable
  getEstablishedBodySubscriptions(){
    return status.getEstablishedBodySubscriptions();
  }

  public SubscriptionTable
  getEstablishedInvalSubscriptions(){
    return status.getEstablishedInvalSubscriptions();
  }

  public SubscriptionTable
  getEstablishedOutgoingBodySubscriptions(){
    return status.getEstablishedOutgoingBodySubscriptions();
  }

  public SubscriptionTable
  getEstablishedOutgoingInvalSubscriptions(){
    return status.getEstablishedOutgoingInvalSubscriptions();
  }

}



//---------------------------------------------------------------------------
/* $Log: P2Runtime.java,v $
/* Revision 1.32  2007/12/18 17:30:47  nalini
/* updated safetymodule
/*
/* Revision 1.31  2007/09/12 19:08:12  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.30  2007/06/05 21:14:57  nalini
/* changed debugging output
/*
/* Revision 1.29  2007/06/05 20:49:24  nalini
/* exposed LOG|CP option for invalSubscriptions to P2Runtime and Overlog layer
/*
/* Revision 1.28  2007/04/05 21:49:25  nalini
/* trying to get subscribeBWUnit to work
/*
/* Revision 1.27  2007/04/02 22:10:01  nalini
/* junit fixes
/*
/* Revision 1.26  2007/04/02 21:06:48  zjiandan
/* fix Env.dprintln problems.
/*
/* Revision 1.25  2007/03/16 23:57:35  nalini
/* p2serverinterface added
/*
/* Revision 1.24  2007/03/12 03:37:07  nalini
/* Chain Replication exp1 files added
/*
/* Revision 1.23  2007/03/09 21:46:09  nalini
/* added 2 events: informAddInvalSubscriptionFailed & informAddBodySubscriptionFailed
/*
/* Revision 1.22  2007/03/08 23:04:21  nalini
/* minor bug fix
/*
/* Revision 1.21  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.20  2007/03/07 16:52:23  nalini
/* Redid lost changes
/*
/* Revision 1.19  2007/03/07 00:32:50  zjiandan
/* Correct DemandRead success inform event reporting.
/*
/* Revision 1.17  2007/03/02 01:35:29  zjiandan
/* Yeah! Working Simplified TierStore.
/*
/* Revision 1.16  2007/02/12 06:19:49  zjiandan
/* Expose noSyncLog parameter, add more unittest for PRACTIFS, PangaeaFS.
/*
/* Revision 1.15  2007/02/01 06:12:09  zjiandan
/* Add acceptStamp to demandRead so that the sender only sends the data
/* that's at least as new as the acceptStamp.
/*
/* Revision 1.14  2007/01/12 22:29:56  nalini
/* Fixed informRemovedSubscription bug
/*
/* Revision 1.13  2007/01/12 19:12:05  nalini
/* minor updates
/*
/* Revision 1.12  2007/01/05 01:18:41  nalini
/* support for sync with VV added
/*
/* Revision 1.11  2006/11/02 21:11:19  zjiandan
/* Fixed some bugs in URAOverlogNode and TupleHandlers.
/*
/* Revision 1.10  2006/10/31 21:43:33  nalini
/* added control msgs to body streams
/*
/* Revision 1.9  2006/10/13 20:49:31  nalini
/* minor fixes
/*
/* Revision 1.8  2006/10/11 16:58:46  dahlin
/* Added PicShare tests that stress test garbage collection
/*
/* Revision 1.7  2006/10/02 23:23:39  nalini
/* synchronization support added
/*
/* Revision 1.6  2006/09/26 05:34:09  nalini
/* added sync interface to policy and runtime
/*
/* Revision 1.5  2006/09/19 22:18:27  nalini
/* P2 and Practi integration
/*
/* Revision 1.4  2006/09/12 22:18:05  dahlin
/* Working to get the unit tests to all run. Up to RandomAccessState now go through. Note that to encourage people to run RASUnit, I have changed the parameters to --quick-- versions that are less extensive tests.
/*
/* Revision 1.3  2006/08/16 23:24:53  nalini
/* informSubscribeXXFailed triggered when reached maxRetries
/*
/* Revision 1.2  2006/08/15 21:46:24  dahlin
/* Added PicShare Reader and a simple unit test.
/*
/* Revision 1.1  2006/08/09 18:52:16  dahlin
/* Fixed Makefile to be portable to cygwin by fixing classpath definitions; fixed P2Runtime/Runtime v. java.lang.Runtime conflict by moving P2Runtime/Runtime to P2Runtime/P2Runtime
/*
*/
//---------------------------------------------------------------------------
