// interfaces with the P2Server.C to install Overlog
import java.io.*;
import java.net.*;
public class P2ServerInterface {

  private static int P2_SERVER_PORT = 54545;
  private static int CLIENT_PORT = 54555;

  private Watcher watcher;
  private String olgFilename;
  private String olgHostname;
  private String olgPort;
  private String[] watchTuples;

  Process serverProcess;
  TupleListener listener;
  
  private boolean dbg = false;


//---------------------------------------------------------------------------
// Constructor
//---------------------------------------------------------------------------

  public P2ServerInterface(int p2ServerPort,
                           int clientPort,
                           String olgFilename,
                           String olgHostname,
                           String olgPort){
    
    this.P2_SERVER_PORT = p2ServerPort;
    this.CLIENT_PORT = clientPort;
    this.olgFilename = olgFilename;
    this.olgHostname = olgHostname;
    this.olgPort = olgPort;
    this.watchTuples = null;

    startP2Server();


    listener = new TupleListener();
    Thread t = new Thread(listener);
    t.start();
  }

  public void setWatchTuples(String[] watchTuples) {
    this.watchTuples = watchTuples;
  }

  public void setWatcher(Watcher watcher){
    this.watcher = watcher;
  }
 



//---------------------------------------------------------------------------
// starts the server
//---------------------------------------------------------------------------
  public synchronized void startP2Server() {

    try{
      Env.dprintln(dbg, "P2ServerInterface: starting P2Server");
      serverProcess = Runtime.getRuntime().exec("P2Server " + P2_SERVER_PORT + " " 
                                         + CLIENT_PORT + " "+ olgFilename + " "+
                                         olgHostname + " "+olgPort);
      Thread dn = new DevOut(serverProcess.getInputStream(), "server stdout");
      dn.start();
      dn = new DevOut(serverProcess.getErrorStream(), "server stderr");
      dn.start();
      wait(3000);
      Env.dprintln(dbg, "P2ServerInterface: started");

    }catch(Exception e) {
      e.printStackTrace();
      assert(false);
    }
  }

//---------------------------------------------------------------------------
// Destroyer
//---------------------------------------------------------------------------
  protected void finalize() {
    if(serverProcess != null){
      serverProcess.destroy();
    }
  }

//---------------------------------------------------------------------------
// Installs the overlog file
//---------------------------------------------------------------------------
  public void installOverlog() {
    // sendToServer("1" + olgFilename);
    //sendToServer("2" + olgHostname);
    //sendToServer("3" + olgPort);
    sendToServer("4");
  }

//---------------------------------------------------------------------------
// Installs the watch tuples
//---------------------------------------------------------------------------
  public void setupWatchTuples() {
    if(watchTuples != null) {
      for(int i = 0; i < watchTuples.length; i++){
        sendToServer("5" + watchTuples[i]);
      }
    }
  }

//---------------------------------------------------------------------------
// Tell the server to run overlog
//---------------------------------------------------------------------------
  public void runOverlog() {
    sendToServer("6");
  }

//---------------------------------------------------------------------------
// startOverlog: 
// - installs the overlog file
// - installs the watch tuples
// - starts the overlog dataflow
//---------------------------------------------------------------------------

  public void startOverlog(){
    installOverlog();
    setupWatchTuples();
    runOverlog();  
  }


//---------------------------------------------------------------------------
// Shutdown
//  - shuts down the server process
//---------------------------------------------------------------------------
  public void shutdown() {
    // shuts down the server
    serverProcess.destroy();
    listener.stopListener();
  }

//---------------------------------------------------------------------------
// sends a string to the P2 Server
//---------------------------------------------------------------------------
  private static void sendToServer(String s){
    try{
      Socket socket = new Socket("localhost", P2_SERVER_PORT);
      socket.setReuseAddress(true);
      PrintWriter out = new PrintWriter(socket.getOutputStream(), true);
      out.print(s);
      out.close();
      socket.close();
    }catch(Exception e) {
      e.printStackTrace();
    }
  }

//---------------------------------------------------------------------------
// if a tuple is received, informs the watcher thread.
//---------------------------------------------------------------------------
  public void received_tuple(String tupleStr){
    if(watcher != null ){
      watcher.receivedTuple(new Tuple(tupleStr));
    }
    else {
      System.out.println("P2ServerInterface: Received Tuple " + tupleStr);
    }
  }


//---------------------------------------------------------------------------
//  insert tuple - inserts tuple in the overlog 
//---------------------------------------------------------------------------
  public void insertTuple(Tuple tuple){
    Env.dprintln(dbg, "P2ServerInterface: inserting Tuple " + tuple);
    sendToServer("7"+tuple.toString());
  }
 

//---------------------------------------------------------------------------
//  main method - install and runs an overlog file
//  Note: it does not handle watch tuples or insertion of tuples
//---------------------------------------------------------------------------
  public static void main(String[] args) {
    if(args.length < 5) {
      System.out.println("Usage: java P2JavaWrapper <serverPort> <clientPort>" +
                         "<overlogfile> <hostname> <olgPort>");
      System.exit(-1);
    }
    try{
    P2ServerInterface serverInterface = new P2ServerInterface((new Integer(args[0])).intValue(),
                                                              (new Integer(args[1])).intValue(),
                                                              args[2],
                                                              args[3],
                                                              args[4]);
    serverInterface.startOverlog();
    }catch(Exception e) {
      e.printStackTrace();
    }
  }

//---------------------------------------------------------------------------
// set up a server to listen for recieved tuples
//---------------------------------------------------------------------------
  class TupleListener implements Runnable {
    boolean stop = false;
    ServerSocket serverSocket; 

    public void stopListener(){
      stop = true;
      if(serverSocket != null) {
        try{
          serverSocket.close();
        }catch(Exception e) {
          e.printStackTrace();
        }
      }
    }

    public void run() {
 
      try{
        serverSocket = new ServerSocket(CLIENT_PORT);
	serverSocket.setReuseAddress(true);
       
        while(!stop) {
          Socket clientSocket = serverSocket.accept();
          BufferedReader in = new BufferedReader(
                                  new InputStreamReader(
                                    clientSocket.getInputStream()));
          String tupleStr = in.readLine();
          in.close();
          clientSocket.close();
          received_tuple(tupleStr);
        }
        serverSocket.close();
      }catch(Exception e) {
        e.printStackTrace();
      }

    }
  }   

//---------------------------------------------------------------------------
/* DevOut
 * Consume stdin and stderr of  subprocesses so they
 * don't block when the buffer is full and spits it on the screen
 */
//---------------------------------------------------------------------------

  class DevOut extends Thread
  {
    InputStream is;
    String who;

    DevOut(InputStream is, String who){
      this.is = is;
      this.who = who;
    }
  
    public void run(){
      try{
        InputStreamReader isr = new InputStreamReader(is);
        BufferedReader br = new BufferedReader(isr);
        String line = null;
        while((line = br.readLine()) != null){
          System.out.println(who + ": " + line);
        }
      }
      catch(IOException e){
        //e.printStackTrace();
      }
    }
  }
}