//---------------------------------------------------------------------------
/* PendingDemandReadQueue.java
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.util.Iterator;

public class PendingDemandReadQueue  {
  private TreeMapValueSearch pending;
  private long numRequest;
  private long sequence;
  private static final boolean dbg = false;
  

  //---------------------------------------------------------------------------
  // Constructor
  //---------------------------------------------------------------------------
  public 
  PendingDemandReadQueue(){
    pending = new TreeMapValueSearch(new TimeKey(-1, -1));
    numRequest=0;
    sequence = 0;
  }

  //---------------------------------------------------------------------------
  //  grab lock; add demand reads to the queue
  //---------------------------------------------------------------------------
  public synchronized void
  add(NodeId senderNodeId, NodeId receiverNodeId, ObjId objId, long offset,
      long length, AcceptStamp as){

    DemandRead dr = new DemandRead(senderNodeId, receiverNodeId, objId, offset,
                                   length, numRequest++, as);
    doAdd(0, dr);
  }

  //---------------------------------------------------------------------------
  //  grab lock; add demand reads to the queue
  //---------------------------------------------------------------------------
  public synchronized void
  add(long timeoutValue, DemandRead dr){
    doAdd(timeoutValue, dr);
  }

  //---------------------------------------------------------------------------
  //  adds demand reads to the queue; internal; assumes lock is held
  //---------------------------------------------------------------------------
  private void doAdd(long timeoutValue, DemandRead dr){
    if(dbg){
      Env.dprintln(dbg, "PendingDemandReadQueue::doAdd(" + dr.getSenderNode().toString() 
		   + ", " + dr.getReceiverNode().toString() 
		   + ", " + dr.getObjId().toString()
		   + ", offset=" + dr.getOffset()
		   + ", len=" + dr.getLength()
		   + ", reqId=" + dr.getRequestId()
		   + ") retry " + dr.getNumTries());
    }
    long wakeupTimeMS = System.currentTimeMillis() + timeoutValue;
    TimeKey key = new TimeKey(wakeupTimeMS, sequence);
    sequence++;
    pending.put(key, dr);
    notify();
    return;
  }

  //---------------------------------------------------------------------------
  // Remove the demand read matching dr from the pending list.
  //---------------------------------------------------------------------------
  public synchronized DemandRead
  remove(DemandRead dr){
    if(dbg){
      Env.dprintln(dbg, "PendingDemandReadQueue::remove(" + dr.getSenderNode().toString() 
		   + ", " + dr.getReceiverNode().toString() 
		   + ", " + dr.getObjId().toString()
		   + ", offset=" + dr.getOffset()
		   + ", len=" + dr.getLength()
		   + ", reqId=" + dr.getRequestId()
		   + ") retry " + dr.getNumTries());
    }
    return (DemandRead)pending.removeByValue(dr);
  }

  //---------------------------------------------------------------------------
  // Remove the demand read matching the parameters from the pending list.
  //---------------------------------------------------------------------------
  public synchronized DemandRead
  remove(NodeId senderNodeId, NodeId receiverNodeId, ObjId objId, long offset,
         long length, long requestId, AcceptStamp as){
     DemandRead dr = new DemandRead(senderNodeId, receiverNodeId, objId, offset,
                                    length, requestId, as);
     if(dbg){
       Env.dprintln(dbg, "PendingDemandReadQueue::remove(" + senderNodeId.toString() 
		    + ", " + receiverNodeId.toString() 
		    + ", " + objId.toString()
		    + ", offset=" + offset
		    + ", len=" + length
		    + ", reqId=" + dr.getRequestId()
		    + ")");
     }
     return (DemandRead)pending.removeByValue(dr);
   }

  //---------------------------------------------------------------------------
  // Removes the demand read with the same requestId
  //---------------------------------------------------------------------------
  public synchronized DemandRead
  removeRequest(NodeId senderNodeId, NodeId receiverNodeId, ObjId objId, long offset,
         long length, long requestId, AcceptStamp as){
    DemandRead ret = null;
    if(dbg){
      Env.dprintln(dbg, "PendingDemandReadQueue::removeRequest(" + senderNodeId.toString() 
		   + ", " + receiverNodeId.toString() 
		   + ", " + objId.toString()
		   + ", offset=" + offset
		   + ", len=" + length
		   + ", reqId=" + requestId
		   + ")");
    
     Env.dprintln(dbg, "PendingDemandReadQueue:: removeRequest--Queue state b4"
                  + this.toString());
    }
     Iterator i = pending.values().iterator();
     while (i.hasNext()){
       DemandRead dr = (DemandRead) i.next();
       if(dr.getRequestId() == requestId) {
         assert(dr.getSenderNode().equals(senderNodeId) &&
                dr.getReceiverNode().equals(receiverNodeId));
         ret=dr;
	 i.remove();
       }
     }
     if(dbg){
       Env.dprintln(dbg, "PendingDemandReadQueue:: removeRequest--Queue state after"
                  + this.toString());
     }
     return ret;
   }

//   //---------------------------------------------------------------------------
//   // Find and return a copy of the DemandRead that matches the parameters.
//   //---------------------------------------------------------------------------
//   public synchronized DemandRead
//   find(NodeId senderNodeId, NodeId receiverNodeId, ObjId objId, long offset,
//          long length){
//     DemandRead dr = new DemandRead(senderNodeId, receiverNodeId, objId, offset,
//                                    length);
//     return (DemandRead)pending.findByValue(dr).clone();
//   }

  //---------------------------------------------------------------------------
  // Wait until the next event's time has come and (1) update the event's
  // retry count and next timeout and (2) return a copy of the event
  // so that the caller can issue the request. We do the update
  // within the lock (rather than, say, remove and reinsert the updated
  // event) in order to make this update atomic and avoid the race
  // between the reply arriving and removing the timeout for the demand read and
  // the timeout being inserted.
  //---------------------------------------------------------------------------
  public synchronized DemandRead
  getNextAndUpdateTimeout(long maxRetries, long timeoutMS){
    DemandRead dr = removeNext();
    DemandRead ret = (DemandRead) dr.clone();
    if(dr.getNumTries() <= maxRetries){
      dr.incNumTries();
      doAdd(timeoutMS, dr);
    }
    return ret;
  }


  //---------------------------------------------------------------------------
  // Wait until next events time has come and return next event to consider.
  // Note this private method is invoked by getNextAndUpdateTimeout to 
  // atomically look at next event *and* insert the timeout for the processing
  // of that event.
  //---------------------------------------------------------------------------
  private DemandRead removeNext(){
    long delay = getDelay();
    while(delay > 0){
      try{
        wait(delay); // Wait for timeout or signal
      }
      catch(InterruptedException e){
      }
      delay = getDelay();
    }
    assert(pending.size() > 0);
    TimeKey k = (TimeKey)pending.firstKey();
    DemandRead ret = (DemandRead)pending.remove(k);
    return ret;
  }


  //---------------------------------------------------------------------------
  // Return the delay we shouldsleep until the first enqueued timeout occurs.
  // Or return a really large number if no timeouts are pending.
  //---------------------------------------------------------------------------
  private long getDelay(){
    if(pending.size() == 0){
      return Long.MAX_VALUE;
    }
    long now = System.currentTimeMillis();
    TimeKey k = (TimeKey)pending.firstKey();
    long when = k.getTimeMS();
    return when - now;
  }
 
  public String toString() {
    return "";
    //commented by zjd, there's concurrent issues.
    //which throws java.util.ConcurrentModificationException
    //return "PendingDemandReadQueue: " + pending.toString();
  } 
}

//---------------------------------------------------------------------------
/* $Log: PendingDemandReadQueue.java,v $
/* Revision 1.8  2007/04/02 21:06:48  zjiandan
/* fix Env.dprintln problems.
/*
/* Revision 1.7  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.6  2007/03/06 18:25:52  zjiandan
/* Add optimization for CatchupInvalIterator, fixed SubscriptionSet, HierInvalTarget
/* and P2Runtime problems. Add aways split when receiving subtree ISStatus in Checkpoint.
/*
/* Revision 1.5  2007/02/01 06:12:09  zjiandan
/* Add acceptStamp to demandRead so that the sender only sends the data
/* that's at least as new as the acceptStamp.
/*
/* Revision 1.4  2006/09/24 20:06:31  nalini
/* trying to make overlog and practi work
/*
*/
//---------------------------------------------------------------------------
