//---------------------------------------------------------------------------
/* PersistentTupleInterface
 * 
 * Implements interface to read and write tuples 
 * to objs stored in the core
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.io.*;

public class PersistentTupleInterface {

  private OverlogPolicy overlogPolicy;
  private LocalInterface li;
  private ObjReaderWriterFactory objRWFactory;
  private TuplePollQueue pQueue;
  private int pollWorkers = 5;

  private boolean dbg = false;


  //-------------------------------------------------------------------------
  // Constructor
  //-------------------------------------------------------------------------  

  public PersistentTupleInterface(P2Runtime runtime, OverlogPolicy policy) {
    overlogPolicy = policy;
    //
    // create a separate local interface to read and write tuples to coare
    // note we use a null controller because we do not expect any read misses
    //
    li = new LocalInterface(new LocalController(), runtime.getCore());
    objRWFactory = new ObjReaderWriterFactory(li);
    
    pQueue = new TuplePollQueue();

    for(int i=0; i<pollWorkers; i++){
      Thread t = new TuplePollQueueWorkerThread(pQueue, this);
      t.start();
    }
  }


  //-------------------------------------------------------------------------
  //  read tuples from objId and insert it into policy
  //-------------------------------------------------------------------------  

  public void readTuples(ObjId objId) {
    ObjReaderWriter objRW = objRWFactory.getObjReaderWriter(objId);
    
    try{
      while (true) {
        String str = objRW.readLine();
        if(OverlogPolicy.P2VersionHasStringProblem){
          // replace all "/" with "-"
          str.replace('/', '-');
        }

        Tuple tp = new Tuple (str);
        if(dbg){
          Env.dprintln(dbg, "PersistentTupleInterface: going to insert" + tp);
        }
        overlogPolicy.insertTuple(tp);
      }
    }catch(EOFException eof) {
      // reached the end of file
      // do nothing
        if(dbg){
          Env.dprintln(dbg, "PersistentTupleInterface: readTuples reached EOF");
        }

    } catch(IOException e) {
      e.printStackTrace();
      assert(false):"Unexpected IOException";
    }
  }


  //-------------------------------------------------------------------------
  // called by tupleRead handler
  //-------------------------------------------------------------------------  

      
  public void tupleRead(ObjId objId) {
    if(dbg){
      Env.dprintln(dbg, "PeristentTupleInterface: tupleRead called for" + objId);
    }
    readTuples(objId);
  }


  //-------------------------------------------------------------------------
  //  read tuples and add it to poll queue
  //   -- note: if you call tupleRead, and then tupleReadAndWatch, 
  //      tupleReadAndWatch will returnn newer tuples, unless stopWatch was
  //      called
  // 
  //-------------------------------------------------------------------------  

  public void tupleReadAndWatch(ObjId objId, long pollTime) {
    if(dbg) {
      Env.dprintln(dbg, "PeristentTupleInterface: tupleReadAndWatch called for" + objId);
    }
    readTuples(objId);
    PollItem pItem = new PollItem(objId, pollTime);
    pQueue.add(pItem, false);
  }

  //-------------------------------------------------------------------------
  // writes a tuple to the objId specified
  //-------------------------------------------------------------------------  

  public void tupleWrite(ObjId objId, Tuple tp) {
    if(dbg) {
      Env.dprintln(dbg, "PeristentTupleInterface: tupleWrite called for" + objId +
        "with tp = " + tp);
    }
    try{
      ObjReaderWriter objRW = objRWFactory.getObjReaderWriter(objId);      
      objRW.appendLn(tp.toString());
    }catch(IOException e) {
      e.printStackTrace();
      assert(false):"Unexpected IOException";
    }
  }


  //-------------------------------------------------------------------------
  //  remove the objReaderWriter -- this has the effect of resetting the
  //                                readerWriter Obj
  //   and removes it from a the poll queue
  //-------------------------------------------------------------------------  
  
  public void tupleStopWatch(ObjId objId) {
  if(dbg) {
      Env.dprintln(dbg, "PeristentTupleInterface: tupleStopWatch called for" + objId);
  }
    objRWFactory.removeObjReaderWriter(objId);
    pQueue.remove(objId);
  }

  //-------------------------------------------------------------------------
  // deletes the obj which has the tuples
  //-------------------------------------------------------------------------  
  public void tupleDelete(ObjId objId){
    if(dbg) {
      Env.dprintln(dbg, "PeristentTupleInterface: tupleDelete called for" + objId);
    }
    objRWFactory.removeObjReaderWriter(objId);
    pQueue.remove(objId);
    
    try{
      li.delete(objId);
    }catch(IOException e) {
      e.printStackTrace();
      assert(false): "Unexpected IOError";
    }
  }

}

//---------------------------------------------------------------------------
/* $Log: PersistentTupleInterface.java,v $
/* Revision 1.3  2007/09/12 19:08:12  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.2  2007/07/13 20:10:39  nalini
/* tuple*handlers debugged
/*
/* Revision 1.1  2007/07/11 20:25:53  nalini
/* persistent tuple support added
/*
*/
//---------------------------------------------------------------------------
