//---------------------------------------------------------------------------
/* RuntimeController.java
 *
 * Implements the controller to hook into the p2 runtime
 *  
 * for relevant inform methods - inserts a message into 
 * a queue.
 */
//--------------------------------------------------------------------------

public class RuntimeController implements Controller
{
  
  private BlockedQueue informQueue;
  private boolean dbg = false; // prints lots of debug information

 // selectively prints output
  private static boolean verboseDR = P2Runtime.verboseDR;
  private static boolean verboseBodySub = P2Runtime.verboseBodySub;
  private static boolean verboseInvalSub = P2Runtime.verboseInvalSub;
  private static boolean verboseLocal = P2Runtime.verboseLocal;
  private static boolean verboseRecvInval = P2Runtime.verboseRecvInval;



//--------------------------------------------------------------------------
// Constructor
//--------------------------------------------------------------------------

  public RuntimeController(P2Runtime rTime, Status status){
    informQueue = status.getPendingControllerInformQueue();
  }


//--------------------------------------------------------------------------
// inform methods related to local events
//--------------------------------------------------------------------------
 public void informLocalWrite(ObjId objId, long offset, long length,
                               AcceptStamp as) {  
   // We now have local copy of data

   InformLocalWriteMsg msg = new InformLocalWriteMsg (objId, offset,
                                                      length,  as,
                                                      false, false);
   informQueue.add(msg);

  }

  public void informLocalWrite(ObjId objId, long offset, long length,
                               AcceptStamp as, boolean isBound, 
                               boolean isEmbargoed){ // We now have local copy of data

    InformLocalWriteMsg msg = new InformLocalWriteMsg (objId, offset,
                                                       length, as,
                                                       isBound,  isEmbargoed);
   
   informQueue.add(msg);
  }


  public void informLocalDelete(ObjId objId){ // No one has this data anymore
    
    
     InformLocalDeleteMsg msg = new InformLocalDeleteMsg(objId);

     informQueue.add(msg);
    
  }

  public void informLocalReadImprecise(ObjId objId, long offset, long length){
    Env.dprintln(dbg || verboseLocal, "RuntimeController :: informLocalReadImprecise " +
                 objId + ":" + offset + ":" + length);

    InformLocalReadImpreciseMsg msg =  
      new InformLocalReadImpreciseMsg (objId, offset, length);
     
    informQueue.add(msg);
  }

  public void informLocalReadNoBody(ObjId objId, long offset, long length){
   
    // For now leave it as such -- need to update it
  }

  public void informLocalReadInvalid(ObjId objId, long offset, long length,
				     AcceptStamp as){
   
     
    InformLocalReadInvalidMsg msg = 
      new InformLocalReadInvalidMsg(objId, offset, length, as);
    
    informQueue.add(msg);
  }

  // Receive sync reply (once we get enough for a given
  // write, can unbind it)
  //
  public void recvSyncReply(AcceptStamp as, NodeId senderNodeId){
    //Env.dprintln(dbg, "RuntimeController :: recvSyncReply " + as + ":" + senderNodeId);
    InformRecvSyncReplyMsg msg = new InformRecvSyncReplyMsg(senderNodeId, as);

    informQueue.add(msg);
  }


//--------------------------------------------------------------------------
// inform methods related to inval streams
//--------------------------------------------------------------------------

  public void informReceiveInval(GeneralInv inv, NodeId senderNodeId){ 

    

    //
    // informs runtime of only precise invals
    //
    // currenly MultiObjPreciseInval does not extend PreciseInv so 
    //  we check for it first
    if(! (inv instanceof MultiObjPreciseInv)) {
      if(inv.isPrecise()){
	PreciseInv pInv = (PreciseInv) inv;
	InvalTarget invalTarget = pInv.getInvalTarget();
	if(invalTarget instanceof ObjInvalTarget){
	  ObjInvalTarget objInvalTarget = (ObjInvalTarget)invalTarget;
	  
	  InformReceiveInvalMsg msg = 
	    new InformReceiveInvalMsg(senderNodeId, 
				      objInvalTarget.getObjId(),
				      objInvalTarget.getOffset(),
				      objInvalTarget.getLength(),
				      pInv.getAcceptStamp(),
				      pInv.isBound(),
				      pInv.isEmbargoed());
      
	  informQueue.add(msg);
	}
      }
    }else if (inv instanceof MultiObjPreciseInv) {
      // handle MultiObjPreciseInv
      MultiObjPreciseInv mInv = (MultiObjPreciseInv) inv;
      MultiObjInvalTarget multiTarget = (MultiObjInvalTarget) mInv.getInvalTarget();
      
      //only report the invalided ones and not the deleted ones
      MOITIterator moitIter = multiTarget.getIterator(true);
      while(moitIter.hasNext()){
	ObjInvalTarget objInvalTarget = moitIter.getNext();
	InformReceiveInvalMsg msg = 
	  new InformReceiveInvalMsg(senderNodeId, 
				    objInvalTarget.getObjId(),
				    objInvalTarget.getOffset(),
				    objInvalTarget.getLength(),
				    mInv.getAcceptStamp(),
				    mInv.hasBoundEntry(objInvalTarget),
				    mInv.isEmbargoed());
      
	  informQueue.add(msg);
	}
    }
 }

  public void
  informSubscribeInvalSucceeded(NodeId senderNodeId,
                                NodeId receiverNodeId,
                                SubscriptionSet ss,
                                VV vvStart){
       
    InformSubscribeInvalAttachedMsg msg = 
      new InformSubscribeInvalAttachedMsg(senderNodeId, ss);    

    informQueue.add(msg);
  }


  public void
  informSubscribeInvalFailed(NodeId senderNodeId,
                             NodeId receiverNodeId,
                             SubscriptionSet ss,
                             VV vvStart){
    
    InformSubscribeInvalFailedMsg msg = 
      new InformSubscribeInvalFailedMsg(senderNodeId, ss);
    
      informQueue.add(msg);
  }
  
  public void
  informInvalStreamTerminated(NodeId senderNodeId,
			      SubscriptionSet ss,
			      VV vvStart,
			      boolean placeholderWriterSet,
			      StreamId sId){
    
    InformInvalStreamTerminatedMsg msg = 
      new InformInvalStreamTerminatedMsg(senderNodeId, ss);   
    
    informQueue.add(msg);
  }

  public void 
  informOutgoingSubscribeInvalInitiated(NodeId receiverNodeId, 
                                        SubscriptionSet ss){
   
    
    InformOutgoingSubscribeInvalInitiatedMsg msg = 
      new InformOutgoingSubscribeInvalInitiatedMsg(receiverNodeId, ss);
    
    informQueue.add(msg);
  }


  public void 
  informOutgoingSubscribeInvalTerminated(NodeId receiverNodeId, 
                                        SubscriptionSet ss){

       
    InformOutgoingSubscribeInvalTerminatedMsg msg = 
      new InformOutgoingSubscribeInvalTerminatedMsg(receiverNodeId, ss);
 
    informQueue.add(msg);
  }


  public void 
  informOutgoingInvalStreamTerminated(NodeId receiverNodeId, 
				      SubscriptionSet ss,
				      VV startVV, 
				      boolean placeholderWriterSet){

    //
    // we convert outgoingInvalStreamTerminated to SubscribeInvalTerminated 
    //

    
    InformOutgoingSubscribeInvalTerminatedMsg msg = 
      new InformOutgoingSubscribeInvalTerminatedMsg(receiverNodeId, ss);
 
    informQueue.add(msg);

  }


  

//--------------------------------------------------------------------------
// inform methods related to body streams
//--------------------------------------------------------------------------
 
  public void 
  informSubscribeBodySucceeded(NodeId senderNodeId,
                               SubscriptionSet ss) {
    
  
    InformSubscribeBodySucceededMsg msg = 
      new InformSubscribeBodySucceededMsg(senderNodeId, ss);
    
    informQueue.add(msg);
  }

  public void 
  informSubscribeBodyRemoved(NodeId senderNodeId,
                             SubscriptionSet ss) {
    

    InformSubscribeBodyRemovedMsg msg = 
      new InformSubscribeBodyRemovedMsg(senderNodeId, ss);
    
    informQueue.add(msg);
  }

  public void
  informBodyStreamTerminated(NodeId senderNodeId) {
    

    InformBodyStreamTerminatedMsg msg = 
      new InformBodyStreamTerminatedMsg(senderNodeId);
   
    informQueue.add(msg);
  }

  public void informOutgoingSubscribeBodyInitiated(NodeId targetNodeId, 
                                                   SubscriptionSet ss){
   

    InformOutgoingSubscribeBodyInitiatedMsg msg = 
      new InformOutgoingSubscribeBodyInitiatedMsg(targetNodeId, ss); 
    
    informQueue.add(msg);
  }


  public void informOutgoingSubscribeBodyTerminated(NodeId targetNodeId, 
						    SubscriptionSet ss){
   

    InformOutgoingSubscribeBodyTerminatedMsg msg = 
      new InformOutgoingSubscribeBodyTerminatedMsg(targetNodeId, ss); 
    
    informQueue.add(msg);

  }

  public void informOutgoingBodyStreamTerminated(NodeId receiverNodeId, 
                                                 SubscriptionSet ss,
                                                 boolean placeholderWriterSet){
    //
    //  convert outgoingBodyStreamTerminated to OutgoingSubscribeBodyTerminated
   

    InformOutgoingSubscribeBodyTerminatedMsg msg = 
      new InformOutgoingSubscribeBodyTerminatedMsg(receiverNodeId, ss); 
    
    informQueue.add(msg);

  }

//--------------------------------------------------------------------------
// inform methods for demand Reads
//--------------------------------------------------------------------------
  public void informReceiveDemandReply(NodeId senderNodeId, BodyMsg bMsg){
   
    ObjInvalTarget objTarget = bMsg.getObjInvalTarget();

    InformReceiveDemandReplyMsg msg = 
      new InformReceiveDemandReplyMsg(senderNodeId,
                                      objTarget.getObjId(),
                                      objTarget.getOffset(),
                                      objTarget.getLength(),
                                      bMsg.getRequestId(),
				      bMsg.getAcceptStamp());
  
    informQueue.add(msg);
  }




//--------------------------------------------------------------------------
// inform methods we don't care about for P2 or are deprecated
//--------------------------------------------------------------------------

  public void informInvalStreamInitiated(NodeId senderNodeId, 
                                         SubscriptionSet ss,
                                         VV vvStart, 
                                         boolean placeholderWriterSet){
  }

 
 public void informBodyStreamInitiated(NodeId senderNodeId) {
  
  }

  public void informCommitStreamInitiated(NodeId senderNodeId){
  
  }
  

  public void informUnbindStreamInitiated(NodeId senderNodeId){
    
  }

  public void informSyncRplyStreamInitiated(NodeId senderNodeId){
    
  }

  public void informOutgoingInvalStreamInitiated(NodeId targetNodeId, 
                                                 VV startVV,
                                                 boolean placeholderWriterSet){
    
  }


  public void informOutgoingBodyStreamInitiated(NodeId targetNodeId,
						boolean placeholderWriterSet){
    
  }

  public void informCommitStreamTerminated(NodeId senderNodeId) {
    //Env.dprintln(dbg, "RuntimeController :: informCommitStreamTerminated");
  }


  public void informUnbindStreamTerminated(NodeId senderNodeId){
    //Env.dprintln(dbg, "RuntimeController :: informUnbindStreamTerminated");
  }

  public void informSyncRplyStreamTerminated(NodeId senderNodeId){
    //Env.dprintln(dbg, "RuntimeController :: informSyncRplyStreamTerminated");
  }

 
  public void informReceivePushBody(BodyMsg msg){ // We now have local copy of data
    //Env.dprintln(dbg, "RuntimeController :: informReceivePushBody");
  }

 
  //
  // On a demand miss, the local store asks the controller to
  // get someone to send the data to me. Thought about including
  // an acceptStamp or VV indicating how new the data neeeds to be
  // to be useful, but (1) we don't always know [e.g., a miss for a new
  // object) and (2) it may change between request issue and reply recv
  // time (e.g., new invalidations arrive in the mean time). So, at best
  // it would be a hint -- but not a terribly useful one -- we probably
  // always want to go get the newest one we can lay our hands on anyhow.
  //
  public void informDemandReadMiss(ObjId objId, long offset, long length){
    //Env.dprintln(dbg || verboseDR, "RuntimeController :: informDemandReadMiss");
  }


  //
  // To support controller cache replacement policy
  // [TBD: Add interface to core: "discardBody()" to discard
  // the body of an object from local storage and "removeIS()"
  // to discard an interest set (which allows us to discard
  // the per-object metadata like last known acceptStamp, etc.]
  //
  public void informDemandReadHit(ObjId objId, long offset, long length){
    //Env.dprintln(dbg || verboseDR, "RuntimeController :: informDemandReadHit");
  }

  //
  // We have received a demand read for an interest set that is
  // currently imprecise. The controller may need to identify 
  // an enclosing interest set to make precise (either an interest
  // set that the local core is already tracking or make a call
  // to the local core telling core to begin tracking a new
  // interest set [TBD: add this call to core].) Then, the controller
  // may need to contact remote nodes and tell them to set up
  // a subscription to this node. If objId belongs to an interest
  // set that the core is already tracking, enclosingIS identifies
  // it and lpVV indicates the last precise VV (from which the new
  // subscription must start.) On the other hand, if the
  // object does not belong to an interest set being tracked,
  // enclosingIS and lpVV will be null. In that case, the controller
  // has to tell the core to start tracking a new IS and when
  // the controller subscribes this node to a stream of invals,
  // this stream must start at time 0 OR we need to get a checkpoint
  // from another node [TBD: add checkpoint transfer interface
  // to node and core.]
  //
  // Note: informDemandImprecise brings the ISStatus up to date
  // and informDemandRead brings the object up to date; normally, 
  // if a core hits an imprecise interest set, it calls informDemandImprecise
  // first and then calls informDemandReadMiss immediately (so that
  // both requests can be doing their work in parallel)
  //
  public void informDemandImprecise(ObjId objId, 
                                    PreciseSet enclosingIS, 
                                    VV lpVV) {
    //Env.dprintln(dbg || verboseDR, "RuntimeController :: informDemandImprecise");
  }

  public void informBecameImprecise(PreciseSet is){ // This IS is currently "imprecise"
    //Env.dprintln(dbg, "RuntimeController :: informBecameImprecise");
  }
  public void informBecamePrecise(PreciseSet is){// This IS is currently "precise"
    //Env.dprintln(dbg, "RuntimeController :: informBecamePrecise");
  }
  //-------------------------------------------------------------------
  //-------------------------------------------------------------------
  //    inform methods for send-checkpoint protocol
  //-------------------------------------------------------------------
  //-------------------------------------------------------------------
  
  //-------------------------------------------------------------------
  // called in SubscribeInvalWorker when !(this.omitVV < startVV) i.e
  // gap exist between the sender's omitVV and the requested stream's 
  // startVV. It implies that the ongoing subscribe inval stream is 
  // failed.
  //
  // What expected in the controller is to make the decision whether
  // to subscribe inval from other nodes for the receiver or invoke
  // a subscribeCheckpoint request to the same node optionally followed 
  // by a subscribeInval request with a higher startVV
  //-------------------------------------------------------------------
 
  public void informGapExistForSubscribeInv(NodeId invReceiver,
                                            SubscriptionSet is,
                                            VV startVV,
                                            VV omitVV){
    //Env.dprintln(dbg || verboseInvalSub, "RuntimeController :: informGapExistForSubscribeInv");
  }

  //-------------------------------------------------------------------
  // inform Outgoing Checkpoint Stream initiated called in CPSendWorker
  //-------------------------------------------------------------------
  public void
  informOutgoingCheckpointStreamInitiated(NodeId targetNodeId, 
                                          String cpPath, 
                                          String[] exclChildNames,
                                          VV startVV,
                                          boolean placeholderWriterSet){
    //Env.dprintln(dbg, "RuntimeController :: informOutgoingCheckpointStreamInitiated");
  }
  //-------------------------------------------------------------------
  // inform Outgoing Checkpoint Stream initiated called in CPSendWorker
  //-------------------------------------------------------------------
  public void
  informOutgoingCheckpointStreamTerminated(NodeId receiverNodeId,
                                           String cpPath,
                                           String[] exclChildNames,
                                           VV startVV, 
                                           boolean placeholderWriterSet){
    //Env.dprintln(dbg, "RuntimeController :: informOutgoingCheckpointStreamTerminated");
  }
  //-------------------------------------------------------------------
  // inform a Checkpoint Stream initiated called in CPRecvWorker
  //-------------------------------------------------------------------
  public void
  informCheckpointStreamInitiated(NodeId senderNodeId, 
                                  String cpPath,
                                  String[] exclChildNames,
                                  VV vvStart,
                                  boolean placeholderWriterSet){
    //Env.dprintln(dbg, "RuntimeController :: informCheckpointStreamInitiated");
  }

  //-------------------------------------------------------------------
  // inform a Checkpoint Stream terminated called in CPRecvWorker
  //-------------------------------------------------------------------
  public void
  informCheckpointStreamTerminated(NodeId senderNodeId, 
                                   String cpPath,
                                   String[] exclChildNames,
                                   VV vvStart, 
                                   boolean placeholderWriterSet){
    //Env.dprintln(dbg, "RuntimeController :: informCheckpointStreamTerminated");
  }


  //-------------------------------------------------------------------
  // inform a Checkpoint Stream apply status called in CPRecvWorker
  //-------------------------------------------------------------------
  public void
  informCheckpointStreamReceiveStatus(NodeId senderNodeId, 
                                      String cpPath,
                                      String[] exclChildNames,
                                      VV vvStart, 
                                      boolean applyStatus,
                                      boolean placeholderWriterSet){
    //Env.dprintln(dbg, "RuntimeController :: informCheckpointStreamTerminated"); 
  }

}


//---------------------------------------------------------------------------
/* $Log: RuntimeController.java,v $
/* Revision 1.24  2007/04/02 21:06:48  zjiandan
/* fix Env.dprintln problems.
/*
/* Revision 1.23  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.22  2007/03/06 18:25:52  zjiandan
/* Add optimization for CatchupInvalIterator, fixed SubscriptionSet, HierInvalTarget
/* and P2Runtime problems. Add aways split when receiving subtree ISStatus in Checkpoint.
/*
/* Revision 1.21  2007/02/28 03:20:03  nalini
/* added debugMsg
/*
/* Revision 1.20  2007/02/27 04:50:10  nalini
/* minor updates
/*
/* Revision 1.19  2007/02/01 06:12:09  zjiandan
/* Add acceptStamp to demandRead so that the sender only sends the data
/* that's at least as new as the acceptStamp.
/*
/* Revision 1.18  2007/01/12 22:29:56  nalini
/* Fixed informRemovedSubscription bug
/*
/* Revision 1.17  2007/01/12 19:12:05  nalini
/* minor updates
/*
/* Revision 1.16  2006/11/18 05:23:17  nalini
/* new read interface implemented
/*
/* Revision 1.15  2006/10/31 21:43:33  nalini
/* added control msgs to body streams
/*
/* Revision 1.14  2006/10/19 15:15:02  nalini
/* disabled dgb output
/*
/* Revision 1.13  2006/10/18 21:57:11  nalini
/* URAOverlogNode added to  make it easier to start overlog and practi in one class
/*
/* Revision 1.12  2006/10/17 23:40:55  nalini
/* minor bug fixes and separated AS into individual components in overlog tuples
/*
/* Revision 1.11  2006/10/02 23:23:39  nalini
/* synchronization support added
/*
/* Revision 1.10  2006/09/26 05:34:09  nalini
/* added sync interface to policy and runtime
/*
/* Revision 1.9  2006/08/16 23:24:53  nalini
/* informSubscribeXXFailed triggered when reached maxRetries
/*
/* Revision 1.8  2006/08/15 21:46:24  dahlin
/* Added PicShare Reader and a simple unit test.
/*
/* Revision 1.7  2006/08/09 18:52:16  dahlin
/* Fixed Makefile to be portable to cygwin by fixing classpath definitions; fixed P2Runtime/Runtime v. java.lang.Runtime conflict by moving P2Runtime/Runtime to P2Runtime/P2Runtime
/*
/* Revision 1.6  2006/06/02 22:40:02  nalini
/* merged support for adding and removing ss for outgoing body streams
/*
/* Revision 1.5.2.1  2006/06/02 22:18:39  nalini
/* Supports addition and removeal of SS from Outgoing Body Streams
/*
/* Revision 1.5  2006/04/24 02:00:15  nalini
/* Adding acceptstamp and sendernodeid to informReceiveInval
/*
/* Revision 1.4  2006/04/23 20:06:43  nalini
/* *** empty log message ***
/*
/* Revision 1.1  2006/04/21 02:54:54  nalini
/* Initial version of P2 runtime. (merged with jiandan's callback code)
/*
/* */
//---------------------------------------------------------------------------
