//---------------------------------------------------------------------------
/*  SafetyPolicyModule
 * 
 *  Sets up predicates for safety.  Policy writes need to extend this 
 *  class to set up the predicates.
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.io.*;


public abstract class SafetyPolicyModule {
  LocalInterface li;
  SafetyLivenessBridge slBridge; // for waiting for msg from liveness


  //---------------------------------------------------------------------------
  // setLocalInterface
  //---------------------------------------------------------------------------
  public void setLocalInterface(LocalInterface li) {
    this.li = li;
  }

  //---------------------------------------------------------------------------
  // set safetyLivenessBridge
  //---------------------------------------------------------------------------
  
  public void setSafetyLivenessBridge(SafetyLivenessBridge slBridge){
    this.slBridge = slBridge;
    registerMsgs();
  }



  //---------------------------------------------------------------------------
  // register msgs to listen to
  //---------------------------------------------------------------------------
  
  abstract void registerMsgs();



  //---------------------------------------------------------------------------
  // readAtPredicates
  //---------------------------------------------------------------------------
  
  protected boolean readBlockImprecise = false;
  protected boolean readBlockInvalid = false;
  protected long maxTE = Long.MAX_VALUE;
  protected long timeout = -1;
  protected double priority = Core.DEFAULT_PRIORITY;


  //---------------------------------------------------------------------------
  // readAfterPredicates
  //---------------------------------------------------------------------------
     
  abstract void readAfterPredicate(ObjId objId, long offset, long length, BodyMsg bMsg);
  


  //---------------------------------------------------------------------------
  // writeBeforePredicates
  //---------------------------------------------------------------------------
  
  abstract void writeBeforePredicate(ObjId objId, long offset, long length);
  


   //---------------------------------------------------------------------------
  // writeAfterPredicates
  //---------------------------------------------------------------------------
  
  abstract void writeAfterPredicate(ObjId objId, long offset, long length);
  




  //---------------------------------------------------------------------------
  // read
  //---------------------------------------------------------------------------
 
  public final BodyMsg read(ObjId objId, long offset, long length)
    throws ObjNotFoundException, IOException,
	   EOFException, ReadOfInvalidRangeException, ReadOfHoleException{

    BodyMsg bMsg =  li.read(objId, offset, length, readBlockInvalid, 
                            readBlockImprecise, maxTE, timeout);

    readAfterPredicate(objId, offset, length, bMsg);

    return bMsg;
  }


  //---------------------------------------------------------------------------
  // write
  //---------------------------------------------------------------------------
 

  public final long write(ObjId objId, 
                          long offset, 
                          long length, 
                          ImmutableBytes buffer) throws IOException {
     return this.write(objId, offset, length, priority, buffer);  
   }

  
  public final long write(ObjId objId, 
                          long offset, 
                          long length, 
                          double priority_, 
                          ImmutableBytes buffer) throws IOException {
     
   writeBeforePredicate(objId, offset, length);

   long res = li.write(objId, offset, length, priority_, buffer, false);

   writeAfterPredicate(objId, offset, length);

   return res;

  }

}


//---------------------------------------------------------------------------
/* $Log: SafetyPolicyModule.java,v $
/* Revision 1.2  2007/12/18 17:30:48  nalini
/* updated safetymodule
/*
/* Revision 1.1  2007/11/28 08:11:34  nalini
/* safety policy module and example checked in
/*
*/
//---------------------------------------------------------------------------