//---------------------------------------------------------------------------
/* Subscription.java
 * 
 *  Wrapper for subscriptions
 *
 * (C) Copyright 2004 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.io.*;

public class Subscription implements Serializable, Cloneable{

  protected NodeId senderNode;
  protected NodeId receiverNode;
  protected SubscriptionSet SS;
  protected long numTries;

  //
  // If we get an exception on one of the tries, stash it in
  // mostRecentException. Later, if maxRetries timer fires,
  // you might want to look in mostRecentException to see if anything
  // of interest is in there...
  //
  protected Exception mostRecentException; 

//---------------------------------------------------------------------------
// Constructor
//---------------------------------------------------------------------------
  public 
  Subscription(NodeId sNode, NodeId rNode, SubscriptionSet sSet){
    this.senderNode = sNode;
    this.receiverNode = rNode;
    this.SS = sSet;
    this.numTries = 0;
    this.mostRecentException = null;
  }

//---------------------------------------------------------------------------
//  get Methods
//---------------------------------------------------------------------------

  public NodeId 
  getSenderNode(){
    return this.senderNode;
  }

  public NodeId 
  getReceiverNode(){
    return this.receiverNode;
  }

  public SubscriptionSet 
  getSS(){
    return this.SS;
  }

  public void 
  setSS(SubscriptionSet ss){
    this.SS = ss;
  }

  public long 
  getNumTries(){
    return numTries;
  }

  public synchronized void 
  incNumTries(){
    numTries++;
  }

  public void setMostRecentException(Exception e){
    mostRecentException = e;
  }

  public Exception getMostRecentException(){
    return mostRecentException;
  }

//---------------------------------------------------------------------------
// Return ture if "o" equals this object
//---------------------------------------------------------------------------
  public boolean 
  equals(Object o){
    //
    // Note: we do not consider numTries in the comparison
    //       And we only compare ss if they are not null
    //

    if(this == o) {
      return true;
    }

    if(!(o instanceof Subscription)){
      return false;
    }
    
    Subscription s2 = (Subscription)o;
    
    int res = this.senderNode.compareTo(s2.getSenderNode()); 
    if(res != 0) {
      return false;
    }

    res = this.receiverNode.compareTo(s2.getReceiverNode());
    if(res !=0){
      return false;
    }


    if(this.SS != null && s2.getSS() != null){
      return this.SS.equals(s2.getSS());
    }
    //
    // should return false but this enables body subscriptions 
    // to work well
    return true;
  }
 
//---------------------------------------------------------------------------
// clone
//---------------------------------------------------------------------------
  public Object clone() {
    Subscription s = new Subscription((NodeId)senderNode.clone(),
                                      (NodeId)receiverNode.clone(), 
                                      (SubscriptionSet) SS.clone());
    s.numTries = this.numTries;
    if(this.mostRecentException != null) {
      s.mostRecentException = new Exception (this.mostRecentException);
    }
    return s;
  }
//---------------------------------------------------------------------------
// For testing
//---------------------------------------------------------------------------

  public String toString(){
    String st = "";
    st += senderNode + "->" + receiverNode + "::" + SS + "::" + numTries;
    return st;
  }

  public static void main(String[] args){
    NodeId sNode = new NodeId((long)10); 
    NodeId rNode = new NodeId((long)11); 
    NodeId tNode = new NodeId((long)12); 
    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/a");
    NodeId sNode_b = new NodeId((long)10); 
    NodeId rNode_b = new NodeId((long)11); 
    SubscriptionSet ss_b= SubscriptionSet.makeSubscriptionSet("/a");
        
    
    Subscription s1 = new Subscription(sNode, rNode, ss);
    Subscription s2 = new Subscription(sNode_b, rNode_b, ss);
    Subscription s3 = new Subscription(sNode, tNode, ss);
    Subscription s4 = new Subscription(tNode, sNode, ss); 
    
   
    System.out.println(s1.getSenderNode());
    System.out.println(s1.getReceiverNode());
    System.out.println(s1.getSS());

    if(!s1.equals(s2)) {
      System.out.println("Error in equals - case 1");
      System.exit(-1);
    }

    if(s1.equals(s3)) {
      System.out.println("Error in equals - case 2");
      System.exit(-1);
    }

    if(s1.equals(s4)) {
      System.out.println("Error in equals - case 3");
      System.exit(-1);
    }

    s2.incNumTries();
    if(!s1.equals(s2)) {
      System.out.println("Error in equals after diff numEntries");
      System.exit(-1);
    }

    //
    // test with subclasses of subscription
    //
    BodySubscription bs1 = new BodySubscription(sNode, rNode, ss);
   
    if(!bs1.equals(s1)){
      System.out.println("Error in equal - bs case 1");
    }

    if(!s1.equals(bs1)){
      System.out.println("Error in equal - bs case 2");
    }


    InvalSubscription is1 = new InvalSubscription(sNode, rNode, ss, false);
   
    if(!is1.equals(s1)){
      System.out.println("Error in equal - is case 1");
    }

    if(!s1.equals(is1)){
      System.out.println("Error in equal - is case 2");
    }

    UpdateSubscription us1 = new UpdateSubscription(sNode, rNode, ss);
   
    if(!us1.equals(s1)){
      System.out.println("Error in equal - us case 1");
    }

    if(!s1.equals(us1)){
      System.out.println("Error in equal - us case 2");
    }


    System.out.println("Successful");
  }
}

//---------------------------------------------------------------------------
/* $Log: Subscription.java,v $
/* Revision 1.8  2007/06/05 20:49:24  nalini
/* exposed LOG|CP option for invalSubscriptions to P2Runtime and Overlog layer
/*
/* Revision 1.7  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.6  2006/09/24 20:06:31  nalini
/* trying to make overlog and practi work
/*
*/
//---------------------------------------------------------------------------
