//---------------------------------------------------------------------------
/* SubscriptionTable.java
 *
 * For each sender and receiver pair, there will only be one entry in the 
 * table
 * 
 * (C) Copyright 2004 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.util.*;
import java.io.*;

public class SubscriptionTable implements Serializable{

  private HashSet tSet;
//---------------------------------------------------------------------------
// Constructor
//---------------------------------------------------------------------------
  public 
  SubscriptionTable(){
    tSet = new HashSet();
  }

  public SubscriptionTable(SubscriptionTable st){
    tSet = new HashSet(st.tSet);
  }

//---------------------------------------------------------------------------
//  add subscription
//  if an entry with the same sender and receiver exists
//     merges the subscription set
//  otherwise adds a new entry to the table
//---------------------------------------------------------------------------


  public synchronized void
  add(Subscription s){
    boolean added = false;
    Iterator i = tSet.iterator();
    while(i.hasNext()){
      Subscription ts = (Subscription)i.next();
      if(ts.getSenderNode().equals(s.getSenderNode()) && 
         ts.getReceiverNode().equals(s.getReceiverNode())){
        ts.setSS(ts.getSS().getCompleteUnion(s.getSS()));
        added = true;
      }
    }
    if(!added){
      tSet.add(s);
    }
  }

  public synchronized void
  add(NodeId senderNodeId, NodeId receiverNodeId, SubscriptionSet ss){
    tSet.add(new Subscription(senderNodeId, receiverNodeId, ss));
  }


//---------------------------------------------------------------------------
//  Finds subscription with the same sender and reciever Id
//  and returns the intersection
//---------------------------------------------------------------------------  
  public synchronized Subscription
  find(Subscription s){
    Iterator i = tSet.iterator();
    while(i.hasNext()){
      Subscription ts = (Subscription)i.next();
      if(ts.getSenderNode().equals(s.getSenderNode()) && 
         ts.getReceiverNode().equals(s.getReceiverNode())){
        SubscriptionSet intSS = ts.getSS().getIntersection(s.getSS());
        if(!intSS.isEmpty()){
          return new Subscription(ts.getSenderNode(), ts.getReceiverNode(), intSS);
        }
      }
    }
    return null;
  }


//---------------------------------------------------------------------------
//  removes the intersection of the SS with the same sender and reciever Id
//  and returns the intersection
//  note: if ss of the subscription is null, it will remove the entry
//  which matches the sender and the receiver node
//---------------------------------------------------------------------------  
  public synchronized Subscription
  remove(Subscription s){
    Iterator i = tSet.iterator();
    while(i.hasNext()){
      Subscription ts = (Subscription)i.next();
      if(ts.getSenderNode().equals(s.getSenderNode()) && 
         ts.getReceiverNode().equals(s.getReceiverNode())){
	
	SubscriptionSet rSS = SubscriptionSet.makeEmptySet();
	SubscriptionSet intSS = SubscriptionSet.makeEmptySet();
	if(s.getSS() != null){
	  intSS = ts.getSS().getIntersection(s.getSS());
          try{
            rSS = ts.getSS().remove(intSS, true);
          }catch(IllegalRemoveSubscriptionSetException e){
            assert false : ("" + e);
          }
	}

	if(rSS.isEmpty()){
           i.remove();
	}
        else{
          ts.setSS(rSS);
        }
        return new Subscription(ts.getSenderNode(), ts.getReceiverNode(), intSS);
      }
    }
    return null;
  }


  public synchronized Subscription 
  remove(NodeId senderNode, NodeId receiverNode, SubscriptionSet ss){
    return remove(new Subscription(senderNode, receiverNode, ss));
  }

  public synchronized Subscription  
  remove(NodeId senderNode, NodeId receiverNode){
    Vector subVector = new Vector();

    Iterator i = tSet.iterator();
    while(i.hasNext()){
      Subscription ts = (Subscription)i.next();
      if(ts.getSenderNode().equals(senderNode) && ts.getReceiverNode().equals(receiverNode)){
        i.remove();
        return ts;
      }
    }
    return null;
  }    

  
  public synchronized int 
  size(){
    return tSet.size();
  }


  private Iterator 
  iterator(){
    return tSet.iterator();
  }
//---------------------------------------------------------------------------
//  bulk operation
//---------------------------------------------------------------------------
  public synchronized boolean 
  addAll(SubscriptionTable st){
    return tSet.addAll(st.tSet);
  }

  public synchronized boolean 
  retainAll(SubscriptionTable st){
    return tSet.retainAll(st.tSet);
  }

//---------------------------------------------------------------------------
// For testing
//---------------------------------------------------------------------------
  public synchronized String toString(){
    String ret =  "";
    Object[] st = tSet.toArray();
    for(int i = 0; i < st.length; i++){
      ret += "["+((Subscription)st[i]).toString()+"]";
    }
    return ret;
  }

  public  static void main(String[] args){
    SubscriptionTable st = new SubscriptionTable();
    
    NodeId n1 = new NodeId((long)10); 
    NodeId n1_b = new NodeId((long)10); 
    NodeId n2 = new NodeId((long)11); 
    NodeId n2_b = new NodeId((long)11); 
    NodeId n3 = new NodeId((long)12); 
    SubscriptionSet ss1 = SubscriptionSet.makeSubscriptionSet("/a");
    SubscriptionSet ss1_b= SubscriptionSet.makeSubscriptionSet("/a");
    SubscriptionSet ss2 = SubscriptionSet.makeSubscriptionSet("/b");
    
    Subscription sb1 = new Subscription(n1, n2, ss1);
    Subscription sb1_b = new UpdateSubscription(n1_b, n2_b, ss1_b);
    Subscription sb2 = new Subscription(n2, n3, ss2);
    Subscription sb3 = new Subscription(n1, n2, ss2);
    Subscription sb4 = new Subscription(n3, n2, ss1);

    sb1_b.incNumTries();
    sb1_b.incNumTries();

    System.out.println("Adding three elements:");
    st.add(sb1);
    st.add(sb2);
    st.add(sb3);
    System.out.println("size = " + st.size());
    System.out.println(st);

    System.out.println("Making a copy");
    SubscriptionTable st2 = new SubscriptionTable(st);
    System.out.println("size = " + st2.size());
    System.out.println(st2);
    
    
    System.out.print("Finding Element...");
    if(!st.find(sb1).equals(sb1)){
      System.out.println("Problem in find -- case 1");
      System.exit(-1);
    }
    if(st.find(sb1_b)==null){
      System.out.println("Problem in find -- case 2");
      System.exit(-1);
    }
    System.out.println("Success");

    System.out.println("Removing Element .."+sb1_b);
    st.remove(sb1_b);
    System.out.println("size = " + st.size());
    System.out.println(st);
    
    SubscriptionTable st3 = new SubscriptionTable();
    st3.add(sb3); st3.add(sb4);
    System.out.println("New Subscription Table");
    System.out.println("size = " + st3.size());
    System.out.println(st3);
    System.out.println("Add All .." + st);
    st3.addAll(st);
    System.out.println("size = " + st3.size());
    System.out.println(st3);
    System.out.println("Retain All .." + st2);
    st3.retainAll(st2);
    System.out.println("size = " + st3.size());    
    System.out.println(st3);

    System.out.println("Testing removeIntersecting and Remove all");
    SubscriptionSet ss10 = SubscriptionSet.makeSubscriptionSet("/10/*");
    SubscriptionSet ss10_a= SubscriptionSet.makeSubscriptionSet("/10/a");
    SubscriptionSet ss10_b= SubscriptionSet.makeSubscriptionSet("/10/b");
    SubscriptionSet ss10_c= SubscriptionSet.makeSubscriptionSet("/10/c");
    SubscriptionSet ss11_a= SubscriptionSet.makeSubscriptionSet("/11/a");
    SubscriptionSet ss11_b= SubscriptionSet.makeSubscriptionSet("/11/b");

    Subscription sb10 = new Subscription(n1, n2, ss10);
    Subscription sb10_a = new Subscription(n1, n2, ss10_a);
    Subscription sb10_b = new Subscription(n1, n2, ss10_b);
    Subscription sb11_a = new Subscription(n1, n2, ss11_a);
    Subscription sb11_b = new Subscription(n2, n3, ss11_b);
    Subscription sb11_c = new Subscription(n1, n2, ss10_c);
    Subscription sb11_c1 = new Subscription(n2, n3, ss10_c);

    SubscriptionTable st4 =  new SubscriptionTable();
    st4.add(sb10_a); st4.add(sb10_b); st4.add(sb11_a); st4.add(sb11_b); 
    st4.add(sb11_c); st4.add(sb11_c1);
   
    System.out.println("Original Table: "+st4);
    Subscription fSub = st4.find(sb10);
    System.out.println("find "+ss10);
    System.out.println("...found "+fSub);
                       
    fSub = st4.remove(n1, n2, ss10_c);
    System.out.println("After remove set "+ fSub + ":");
    System.out.println("Table: "+st4);

    fSub = st4.remove(n1, n2, ss10);
    System.out.println("After remove set "+ fSub + ":");
    System.out.println("Table: "+st4);
    
                       
    System.out.println("");
    fSub = st4.remove(n2, n3);
    System.out.println("After remove "+ n2 + "->" + n3);
    System.out.println("Table: "+st4);
    System.out.println("..removed "+fSub);
    

  }

    

}

//---------------------------------------------------------------------------
/* $Log: SubscriptionTable.java,v $
/* Revision 1.6  2006/10/02 23:23:39  nalini
/* synchronization support added
/*
/* Revision 1.5  2006/09/05 04:04:00  nayate
/* Fixed to handle new SubscriptionSet.remove() interface
/*
/* Revision 1.4  2006/06/13 03:49:19  nalini
/* RMI for P2 Runtime Implemented
/*
*/
//---------------------------------------------------------------------------
