//---------------------------------------------------------------------------
/* TuplePollQueue
 * 
 * Implements a timeout queue which keeps track of which objects
 * need to be polled for tuples.
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.util.*;

public class TuplePollQueue {
  private TimeoutQueue tQueue;
  private LinkedList removedItems;
  private int maxListSize = 200; // keeps the list from becoming too large

  private boolean dbg = false;

  //-------------------------------------------------------------------------
  // Constructor
  //-------------------------------------------------------------------------
  public TuplePollQueue(){
    this.tQueue = new TimeoutQueue();
    this.removedItems = new LinkedList();
  }

  //-------------------------------------------------------------------------
  //  adds an objId to the Queue
  // isWorker = true if the item is added by a worker thread
  //-------------------------------------------------------------------------
  public void 
  add(PollItem pItem, boolean isWorker){
    if(dbg) {
      Env.dprintln(dbg, "TuplePollQueue :: added " + pItem + " called");
    }
    
    ObjId objId = pItem.getObjId();
    if(isWorker && removedItems.indexOf(objId) >=0 ){
      // Remove for this item has been called and it should 
      // not be added to the queue
      removedItems.remove(objId);      
      if(dbg) {
        Env.dprintln(dbg, "TuplePollQueue :: " + pItem + "not added");
      }
      
    } else {
      //insert it into the queue
      tQueue.insert(pItem.getPollTime(), pItem);
      // remove any previous removed called for this obj
      if(dbg) {
        Env.dprintln(dbg, "TuplePollQueue :: added " + pItem);
      }
      removedItems.remove(objId);      

    }

  }


  //-------------------------------------------------------------------------
  // Removes poll for objId
  //-------------------------------------------------------------------------
  public void
  remove(ObjId objId){
    if(dbg){
      Env.dprintln(dbg, "TuplePollQueue :: remove " + objId + " called");  
    } 
    
    PollItem pItem = (PollItem) tQueue.remove(new PollItem(objId, 0));
    if (pItem == null) {
      // either the watch was never called or 
      // a worker thread is working on it 
      // put it under removed items so that it does not get
      // added again    
      if(removedItems.size() >= maxListSize) {
        removedItems.removeFirst();
      }
      removedItems.add(objId);
     
    }
      
  }
    

  //-------------------------------------------------------------------------
  // Returns the next objId which needs to be read 
  // Blocks until the timeout has reached
  //-------------------------------------------------------------------------
  public PollItem
  getNext(){
    return (PollItem)tQueue.getNext();
  }
}

//---------------------------------------------------------------------------
/* $Log: TuplePollQueue.java,v $
/* Revision 1.1  2007/07/11 20:25:53  nalini
/* persistent tuple support added
/*
*/
//---------------------------------------------------------------------------

