//---------------------------------------------------------------------------
/* ClientServerUnit.java
 * 
 * Unit test for testing Client-Server Scenario.
 *  - Client sets up subscriptions
 *  - Server makes bound writes
 *  - Client tries to read updated values.
 *
 * (C) Copyright 2004 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;


public class ClientServerUnit extends TestCase {


  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean verbose = false; // Start/end of test
  protected static boolean vverbose = false; // Test internals

  private Process rmiregistry;
  private Process client;
  private Process server;
  private BarrierServer barrierServer;
  protected static int BARRIER_PORT = 5000;

  protected static String CONFIG_PATH = "test" + File.separatorChar + "tmp.ClientServerUnit.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar + "tmp.ClientServerUnit.p2config";
  protected static long READER_NODE_ID = 0;
  protected static long WRITER_NODE_ID = 1;

  protected static int sizeOfWrites = 20;
  protected static int numOfObjs = 10;
  protected static byte initialValue = 65; //"A";
  protected static byte finalValue = 66; //"B"; 

//---------------------------------------------------------------------------
//  Basic Constructor - called by test runners
//---------------------------------------------------------------------------

  public ClientServerUnit(final String s){
    super (s);
  }

//---------------------------------------------------------------------------
//  Set up environment in which tests must run
//---------------------------------------------------------------------------
  protected void setUp() throws Exception {
    super.setUp();

    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }
    

    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    System.out.println("rmiregistry started");
    Thread.sleep(2000);

    //
    // Start barrier server for client & server to co-ordinate
    //
    
    barrierServer = new BarrierServer(BARRIER_PORT, 2, 4);
    barrierServer.start();

    //
    // make the config files
    //
    makePractiConfig(CONFIG_PATH);
    makeP2Config(CONFIG_P2_PATH);

  }

//---------------------------------------------------------------------------
// Clearing up after test is over
//---------------------------------------------------------------------------
  protected void tearDown() throws Exception {
    if(client != null){
      client.destroy();
      System.out.println("Client terminated");
    }
    if(server != null){
      server.destroy();
      Env.dprintln(verbose, "Server terminated");
    }
    
    barrierServer.interrupt();

    rmiregistry.destroy();
    // 
    // the previous Unit Test might not completely kill
    // rmiregistry when the second unit test is about to run
    // 
    // Any unit test setup should clean all rmiregistry
    // before start rmiregistry
    // otherwise it is possible that someone else kills the
    // rmiregistry when this unit just starts it.
    //
    
    Thread.sleep(2000);
    Env.dprintln(verbose, "rmiregistry terminated");
    super.tearDown();
  }

//---------------------------------------------------------------------------
// Test if the client can read the most updated data
//---------------------------------------------------------------------------
  public void testRead(){
    try{
      //
      //  Starts the client. Use make to get paths set up properly.
      //
      // client = Runtime.getRuntime().exec("java -ea ClientServerUnit_Client");
      client = Runtime.getRuntime().exec("make ClientServerUnit_Client.unit");

      Thread dn = new DevNull(client.getInputStream(), "Client stdout");
      dn.start();
      dn = new DevNull(client.getErrorStream(), "Client stderr");
      dn.start();
      System.out.println("Client Started");

      //
      // Starts the server
      //
      // server = Runtime.getRuntime().exec("java -ea ClientServerUnit_Server");
      server = Runtime.getRuntime().exec("make ClientServerUnit_Server.unit");
      dn = new DevNull(server.getInputStream(), "Server stdout");
      dn.start();
      dn = new DevNull(server.getErrorStream(), "Server stderr");
      dn.start();
      System.out.println("Server Started");

      //
      // Waits for the client to end.  If there is error -> test fails
      //
      client.waitFor();
      if(client.exitValue() != 0){
	assert(false): "Error in test Read";
      }
    
    }catch(Exception e){
      e.printStackTrace();
    }
  }
  
//---------------------------------------------------------------------------
//  Makes the config file for the experiment
//---------------------------------------------------------------------------
  private void makePractiConfig(String configPath){
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(READER_NODE_ID),
                            "localhost",
                            9718,
                            9719,
                            9711,
                            9712,
                            9710,
                            "test" + File.separatorChar + "tmp.ClientServerUnit-" + 
			    READER_NODE_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9713,
                            9714,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
   Config.addOneNodeConfig(new NodeId(WRITER_NODE_ID),
                            "localhost",
                            9608,
                            9609,
                            9601,
                            9602,
                            9600,
                           "test" + File.separatorChar + "tmp.ClientServerUnit-" + 
			   WRITER_NODE_ID+".db",
                            "/*",
                            -1L,
                            "localhost",
                            9603,
                            9604,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    
   Config.writeToFile(configPath);
  }

//---------------------------------------------------------------------------
//  Makes the P2 config file for the experiment
//---------------------------------------------------------------------------
  private void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(READER_NODE_ID), 2, 2, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(WRITER_NODE_ID), 2, 2, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.writeToFile(p2ConfigPath);
  }

  public static Test suite(){
    TestSuite suite = new TestSuite(ClientServerUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   PicShareReaderUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "ClientServerUnit";
    System.err.println(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
         vverbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new ClientServerUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);
  }
}



//---------------------------------------------------------------------------
/* $Log: ClientServerUnit.java,v $
/* Revision 1.10  2007/07/13 21:17:19  nalini
/* changed port numbers to make unit test work
/*
/* Revision 1.9  2007/03/09 03:01:38  nalini
/* removed update workers option from P2Config
/*
/* Revision 1.8  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.7  2007/01/10 02:02:24  zjiandan
/* Fixed rmiregistry problems.
/*
/* Revision 1.6  2006/11/15 17:48:46  nalini
/* fixing unit tests
/*
/* Revision 1.5  2006/10/12 17:30:38  nalini
/* changed port numbers
/*
/* Revision 1.4  2006/10/02 23:23:39  nalini
/* synchronization support added
/*
/* Revision 1.3  2006/09/12 22:18:05  dahlin
/* Working to get the unit tests to all run. Up to RandomAccessState now go through. Note that to encourage people to run RASUnit, I have changed the parameters to --quick-- versions that are less extensive tests.
/*
/* Revision 1.2  2006/09/04 14:57:39  dahlin
/* Use make to launch java helper processes in ClientServerUnit to make sure paths set up properly
/*
/* Revision 1.1  2006/08/24 23:08:53  nalini
/* Simple unit test created for P2Runtime
/*
 */
//---------------------------------------------------------------------------
