//---------------------------------------------------------------------------
/* ClientServerUnit_Client.java
 * 
 * Implements the Client side for the ClientServerUnit test
 *  - waits for server to create objects
 *  - establishes subscriptions
 *  - waits for server to update objects
 *  - reads the objects and ensures that we have the updated value
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------
import java.io.*;

public class ClientServerUnit_Client {

  NullReader nReader; 
  int myId;
  ImmutableBytes expectedValue;
  LocalInterface lInterface;
  P2Runtime p2Runtime;
  private static final boolean verbose = false;

//---------------------------------------------------------------------------
//  Basic constructor
//---------------------------------------------------------------------------

  public ClientServerUnit_Client() throws Exception{
    
    nReader = new NullReader(ClientServerUnit.CONFIG_PATH,
			     ClientServerUnit.CONFIG_P2_PATH,
			     new NodeId(ClientServerUnit.READER_NODE_ID));
    myId = (int) ClientServerUnit.READER_NODE_ID;

    byte[] b = new byte[ClientServerUnit.sizeOfWrites];
    for(int i = 0; i < ClientServerUnit.sizeOfWrites; i++){
      b[i] = ClientServerUnit.finalValue;
    }

    expectedValue =  new ImmutableBytes(b);
    lInterface = nReader.getLocalInterface();
    p2Runtime = nReader.getRuntime();

      
  }


//---------------------------------------------------------------------------
//  Makes sure we have received the invals for all the objs written
//---------------------------------------------------------------------------

  public void syncWithServer(){
    while(true){
      try{

	lInterface.sync(new AcceptStamp((long)ClientServerUnit.numOfObjs * 2 - 1 , 
					new NodeId(ClientServerUnit.WRITER_NODE_ID)));
	break;
      }catch(InterruptedException ie){
	//ignore
      }
    }
  }


//---------------------------------------------------------------------------
//  Establishes Subscriptions
//---------------------------------------------------------------------------

  public void establishSubscriptions(){
    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/*");
     p2Runtime.addBodySubscription(new NodeId(ClientServerUnit.WRITER_NODE_ID), 
                                   ss);
     p2Runtime.addInvalSubscription(new NodeId(ClientServerUnit.WRITER_NODE_ID), 
				    ss, false);
    
  }


//---------------------------------------------------------------------------
//  Reads each object and compares it with the expected value
//---------------------------------------------------------------------------

  public void readStuff(){

    try{
      for (int i=0; i < ClientServerUnit.numOfObjs; i++){
	ObjId objId = new ObjId("/"+ i);
	
	while(true){
	  try{
	    ImmutableBytes b = nReader.read(objId, 0, ClientServerUnit.sizeOfWrites);
	    assert(b.equals(expectedValue)): "Read data does not match expected Value" ;
	    break;
	  }catch(ObjNotFoundException e){
	    //ignore
	    //System.out.println("Obj not found exception for "+objId);
	    Thread.currentThread().sleep(50);
	  }
	}
      }
    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }


//---------------------------------------------------------------------------
// getID
//---------------------------------------------------------------------------
  public int getId(){
    return myId;
  }


//---------------------------------------------------------------------------
//  Main method - establishes subscriptions, and reads data received from
//  the server to make sure we have the correct value.
//---------------------------------------------------------------------------

  public static void main(String[] args){
   
     try{
      if (verbose) {
	System.out.println("ClientServerUnit_Client Starting...");
      }

      ClientServerUnit_Client client = new ClientServerUnit_Client();
      BarrierClient c =  new BarrierClient("localhost", 
			     ClientServerUnit.BARRIER_PORT,
			     client.getId());



      c.sendBarrierRequest(0, client.getId()); // Wait to "Go"


      c.sendBarrierRequest(0, client.getId()); // Wait for server to write data
      client.establishSubscriptions();
      if (verbose) {
	System.out.println("ClientServerUnit_Client established subscriptions");
      }


      c.sendBarrierRequest(0, client.getId()); // Wait for server to write new data
      client.syncWithServer();
      if (verbose) {
	System.out.println("ClientServerUnit_Client got all Invals");
      }
      Thread.currentThread().sleep(100);

      try{
	client.readStuff();
	if (verbose) {
	  System.out.println("ClientServerUnit_Client finished reading stuff");
	}


	c.sendBarrierRequest(0, client.getId()); // Finished Reading Data
	if (verbose) {
	  System.out.println("ClientServerUnit_Client Exiting... Bye Bye");
	}

      }catch(java.lang.AssertionError ae){
	ae.printStackTrace();
	//send the last Barrier Request
	c.sendBarrierRequest(0, client.getId()); // Finished Reading Data
	System.exit(-1);
      }

      System.exit(0);

     }catch(Exception e){
       //ignore
     }
  }
    
}

//---------------------------------------------------------------------------
/* $Log: ClientServerUnit_Client.java,v $
/* Revision 1.4  2007/06/05 20:49:24  nalini
/* exposed LOG|CP option for invalSubscriptions to P2Runtime and Overlog layer
/*
/* Revision 1.3  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.2  2006/08/30 21:38:56  nalini
/* Three node unit test added
/*
/* Revision 1.1  2006/08/24 23:08:53  nalini
/* Simple unit test created for P2Runtime
/*
*/
//---------------------------------------------------------------------------
