//---------------------------------------------------------------------------
/* ObjReaderWriterUnit.java
 * 
 * Unit test to test the ObjReaderWriterFactory
 * and the ObjReaderWriter
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;


public class ObjReaderWriterUnit extends TestCase {


  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean verbose = false; // Start/end of test
  protected static boolean vverbose = false; // Test internals
 
  private Process rmiregistry;
  RMIServerImpl rmiServer;
  SocketServer socketServer;
  Core core;

  protected static String CONFIG_PATH = "test" + File.separatorChar + "tmp.ObjReaderWriterUnit.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar + "tmp.ObjReaderWriterUnit.p2config";
  protected static long nodeIdLong = 0;

 
  private ObjReaderWriterFactory factory;
 
//---------------------------------------------------------------------------
//  Basic Constructor - called by test runners
//---------------------------------------------------------------------------

  public ObjReaderWriterUnit(final String s){
    super (s);
  }

//---------------------------------------------------------------------------
//  Set up environment in which tests must run
//---------------------------------------------------------------------------
  protected void setUp() throws Exception {
    super.setUp();
    makePractiConfig(CONFIG_PATH);
    Config.readConfig(CONFIG_PATH);

    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }
    
    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    System.out.println("rmiregistry started");
    Thread.sleep(2000);

    RMIClient rmiClient = new RMIClient();
    core = new Core(rmiClient,true ,true, new NodeId(nodeIdLong)); 
    Controller controller = new LocalController(core);
    LocalInterface li = new LocalInterface(controller, core);
    rmiServer = new RMIServerImpl(core, controller);
    socketServer = new SocketServer(core, controller);
    core.recoverLocalState(rmiServer);

    factory = new ObjReaderWriterFactory(li);
  }

//---------------------------------------------------------------------------
// Clearing up after test is over
//---------------------------------------------------------------------------
  protected void tearDown() throws Exception {

    rmiServer.close();
    rmiServer = null;
    socketServer.shutdown();
    socketServer = null;
    core.syncStateToDisk();
    core.close();
    core = null;

    rmiregistry.destroy();
    Thread.sleep(2000);
    Env.dprintln(verbose, "rmiregistry terminated");
    super.tearDown();
    
  }



//---------------------------------------------------------------------------
// Test factory -- simple test to see if generates objRW correctly
//---------------------------------------------------------------------------
  public void testFactory(){
    if(verbose) {
      System.out.println("Starting testFactory");
    }

    int numObj = 5;
    
    assert(factory.getSize() == 0);

    for(int i = 0; i < numObj; i++){
      ObjId objId = new ObjId("/" + i);
      factory.getObjReaderWriter(objId);
    }    
    if(vverbose) factory.printFactoryTable();
    assert(factory.getSize() == numObj);

    for(int i = 0; i < numObj; i++){
      ObjId objId = new ObjId("/" + i);
      factory.getObjReaderWriter(objId);
    }    
    if(vverbose) factory.printFactoryTable();
    assert(factory.getSize() == numObj);

    for(int i = 0; i < numObj; i++){
      ObjId objId = new ObjId("/" + i);
      factory.removeObjReaderWriter(objId);
    }    
    if(vverbose) factory.printFactoryTable();
    assert(factory.getSize() == 0);
    
  }
  
//---------------------------------------------------------------------------
// Single Obj test  -- tests r/w operations on a single obj
// phase 1 test
// r0a -- EOFexception
// w1 -- value 1
// w2 -- value 2
// r1 -- reads value 1
// r2 -- reads value 2
// r0b -- EOF exception
// w3 -- value 3 (note: it should be big enough to exceed block size = 80)
// r3 -- value 3
// w4 -- value 4 (note: small value)
// r4 -- value 4
// r0c -- EOFexception
// removeObjRW
// phase 2 test:
// Get new ObjRW
// r11 -- value 1
// r22 -- value 2
// r33 -- value 3
// r44 -- value 4
// r00 -- EOF exception
//---------------------------------------------------------------------------
  public void testSingleObj(){
    // set up variables
    String value1 = "abcdefghijklmnopqrstuvwxyz";
    String value2 = "01234567890";
    String value3 = "012345678901234567890123456789" +
                    "012345678901234567890123456789" +
                    "012345678901234567890123456789" +
                    "012345678901234567890123456789" +
                    "012345678901234567890123456789";
    String value4 = "abc";
    String str;

    ObjId objId = new ObjId("/0");

    // phase 1 test
    if(verbose) System.out.println("Starting testSingleObj phase 1");

    ObjReaderWriter objRW = factory.getObjReaderWriter(objId);
    try{
      //r0a
      if(vverbose) System.out.println("r0a");  
      try{
        str = objRW.readLine();
        assert(false): "We expect a EOF exception at r0a";
      }catch(EOFException e){
        //exptected
      }
     
      //w1, w2
      if(vverbose) System.out.println("w1, w2");  
      objRW.appendLn(value1);
      objRW.appendLn(value2);

      //r1a
      if(vverbose) System.out.println("r1a");  
      try{
        str = objRW.readLine();
        if(vverbose) System.out.println("value read" + str);
        assert(str.equals(value1)):"r1a: value1 does not match";
      }catch(EOFException e){
        assert(false):"EOF should not happen at r1a";
      }

      //r2a
      if(vverbose) System.out.println("r2a");  
      try{
        str = objRW.readLine();
        if(vverbose) System.out.println("value read" + str);
        assert(str.equals(value2)):"r2a: value2 does not match";
      }catch(EOFException e){
        assert(false):"EOF should not happen at r2a";
      }
         
      //r0b0
      if(vverbose) System.out.println("r0b0");  
      try{
        str = objRW.readLine();
        assert(false): "We expect a EOF exception at r0b0";
      }catch(EOFException e){
        //exptected
      }

      //r0b1
      if(vverbose) System.out.println("r0b1");  
      try{
        str = objRW.readLine();
        assert(false): "We expect a EOF exception at r0b1";
      }catch(EOFException e){
        //exptected
      }

       
      //w3
      if(vverbose) System.out.println("w3");  
      objRW.appendLn(value3);

      //r3a
      if(vverbose) System.out.println("r3a");  
      try{
        str = objRW.readLine();
        if(vverbose) System.out.println("value read" + str);
        assert(str.equals(value3)):"r3a: value3 does not match";
      }catch(EOFException e){
        assert(false):"EOF should not happen at r3a";
      }

      //w4
      if(vverbose) System.out.println("w");  
      objRW.appendLn(value4);

      //r4a
      if(vverbose) System.out.println("r4a");  
      try{
        str = objRW.readLine();
        if(vverbose) System.out.println("value read" + str);
        assert(str.equals(value4)):"r4a: value3 does not match";
      }catch(EOFException e){
        assert(false):"EOF should not happen at r4a";
      }

      //r0c
      if(vverbose) System.out.println("r0c");  
      try{
        str = objRW.readLine();
        assert(false): "We expect a EOF exception at r0c";
      }catch(EOFException e){
        //exptected
      }
      
      factory.removeObjReaderWriter(objId);


      // phase 2 test
      if(verbose) System.out.println("Starting testSingleObj phase 1");
      
      objRW = factory.getObjReaderWriter(objId);
      // read all the 4 values
      try{
        str = objRW.readLine();
        if(vverbose) System.out.println("value read" + str);
        assert(str.equals(value1)):"r11: value does not match";
          str = objRW.readLine();
        if(vverbose) System.out.println("value read" + str);
        assert(str.equals(value2)):"r22: value does not match";
          str = objRW.readLine();
        if(vverbose) System.out.println("value read" + str);
        assert(str.equals(value3)):"r33: value does not match";
          str = objRW.readLine();
        if(vverbose) System.out.println("value read" + str);
        assert(str.equals(value4)):"r44: value does not match";
      }catch(EOFException e){
        assert(false):"EOF should not happen at r3a";
      }

      //r00
      if(vverbose) System.out.println("r0c");  
      try{
        str = objRW.readLine();
        assert(false): "We expect a EOF exception at r00";
      }catch(EOFException e){
        //exptected
      }

    }catch(IOException ioe) {
      ioe.printStackTrace();
      assert(false):"Unexpected IOException";
    }

  }
 
//---------------------------------------------------------------------------
// multiple obj test -- not yet implemented
//---------------------------------------------------------------------------

//  public void testMultipleObj(){
//  }


//---------------------------------------------------------------------------
//  Makes the config file for the experiment
//---------------------------------------------------------------------------
  private void makePractiConfig(String configPath){
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(nodeIdLong),
                            "localhost",
                            9788,
                            9789,
                            9791,
                            9792,
                            9790,
                            "test" + File.separatorChar + "tmp.ObjReaderWriterUnit.db", 
                            "/*",
                            -1L,
                            "localhost",
                            9793,
                            9794,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    
   Config.writeToFile(configPath);
  }

//---------------------------------------------------------------------------
// Suite
//---------------------------------------------------------------------------


  public static Test suite(){
    TestSuite suite = new TestSuite(ObjReaderWriterUnit.class);
    return suite;
  }


//---------------------------------------------------------------------------
// Main Method
//---------------------------------------------------------------------------

  public static void main(String s[]) {
    String name = "ObjReaderWriterUnit";
    System.err.println(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
         vverbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new ObjReaderWriterUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);

    System.exit(0);
  }
}



//---------------------------------------------------------------------------
/* $Log: ObjReaderWriterUnit.java,v $
/* Revision 1.1  2007/07/11 20:25:53  nalini
/* persistent tuple support added
/*
 */
//---------------------------------------------------------------------------
