//---------------------------------------------------------------------------
/* PersistentTupleInterfaceUnit.java
 * 
 * To test the persistent tuple interface
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;


public class PersistentTupleInterfaceUnit extends TestCase {


  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean verbose = false; // Start/end of test
  protected static boolean vverbose = false; // Test internals
 

  protected static String CONFIG_PATH = "test" + File.separatorChar + "tmp.PersistentTuplInterfaceUnit.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar + "tmp.PersistentTupleInterfaceUnit.p2config";
  protected static String NODEID_MAP_PATH = "test" + File.separatorChar + "tmp.PersistentTupleInterfaceUnit.map";
  protected static String OVERLOG_PATH = "";
  protected static long nodeIdLong = 0;

  private Process rmiregistry;
  private OverlogPolicy overlogPolicy;
  private P2Runtime runtime;
  private PersistentTupleInterface ptInterface;
  private LinkedList insertedTuples;

 
//---------------------------------------------------------------------------
//  Basic Constructor - called by test runners
//---------------------------------------------------------------------------

  public PersistentTupleInterfaceUnit(final String s){
    super (s);
  }

//---------------------------------------------------------------------------
//  Set up environment in which tests must run
//---------------------------------------------------------------------------
  protected void setUp() throws Exception {
    super.setUp();
    makePractiConfig(CONFIG_PATH);
    makeP2Config(CONFIG_P2_PATH);
    makeNodeIdMap(NODEID_MAP_PATH);


    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }
    
    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    System.out.println("rmiregistry started");
    Thread.sleep(2000);
    
  
    overlogPolicy = new OverlogPolicy(OVERLOG_PATH, 
                                                    new NodeId(nodeIdLong),
                                                    NODEID_MAP_PATH);

    runtime = new P2Runtime(CONFIG_PATH, CONFIG_P2_PATH, 
                                    new NodeId(nodeIdLong), true, 
                                    overlogPolicy, false);

    overlogPolicy.setRuntime(runtime);
    runtime.start();
    insertedTuples = new LinkedList();
    overlogPolicy.startInDebugMode(insertedTuples);

    ptInterface = new PersistentTupleInterface(runtime, overlogPolicy);

  }

//---------------------------------------------------------------------------
// Clearing up after test is over
//---------------------------------------------------------------------------
  protected void tearDown() throws Exception {

    runtime.shutdown();
    rmiregistry.destroy();
    Thread.sleep(2000);
    Env.dprintln(verbose, "rmiregistry terminated");
    super.tearDown();
    
  }



//---------------------------------------------------------------------------
//  Makes the config file for the experiment
//---------------------------------------------------------------------------
  private void makePractiConfig(String configPath){
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(nodeIdLong),
                            "localhost",
                            9788,
                            9789,
                            9791,
                            9792,
                            9790,
                            "test" + File.separatorChar + "tmp.ObjReaderWriterUnit.db", 
                            "/*",
                            -1L,
                            "localhost",
                            9793,
                            9794,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    
   Config.writeToFile(configPath);
  }

//---------------------------------------------------------------------------
//  Makes the P2 config file for the experiment
//---------------------------------------------------------------------------
  private void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(nodeIdLong), 2, 2, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.writeToFile(p2ConfigPath);
  }

 private static void makeNodeIdMap(String nodeIdMapPath) {
    NodeIdMap.createEmptyMap();
    NodeIdMap.add(new NodeId(nodeIdLong), new OverlogId("localhost:5000"));
    NodeIdMap.writeToFile(nodeIdMapPath);
  }

//---------------------------------------------------------------------------
// Suite
//---------------------------------------------------------------------------


  public static Test suite(){
    TestSuite suite = new TestSuite(PersistentTupleInterfaceUnit.class);
    return suite;
  }


//---------------------------------------------------------------------------
// test interface
//---------------------------------------------------------------------------
  public void testInterface(){
    if(verbose) System.out.println("Starting to test interface");
    //set up variables
    ObjId objId = new ObjId("/testObj");
    Tuple tp1 = new Tuple("testtuple(1)");
    Tuple tp2 = new Tuple("testtuple(2)");
    Tuple tp3 = new Tuple("testtuple(3)");
    Tuple tp4 = new Tuple("testtuple(4)");
    Tuple tp5 = new Tuple("testtuple(5)");
    Tuple tp6 = new Tuple("testtuple(6)");
    Tuple tp7 = new Tuple("testtuple(7_");
    Tuple receivedTuple;
    
    // issue read -- expect nothing
    if(vverbose) System.out.println("issuing initial read");
    ptInterface.tupleRead(objId);
    assert(insertedTuples.size() == 0);
    
    //write tuple 1, and tuple 2 then try to read it
    if(vverbose) System.out.println("writing tp1 & tp2");
    ptInterface.tupleWrite(objId, tp1);
    ptInterface.tupleWrite(objId, tp2);
    if(vverbose) System.out.println("tuple read issued");
    ptInterface.tupleRead(objId);
    assert(insertedTuples.size() == 2);
    receivedTuple = (Tuple) insertedTuples.removeFirst();
    if(vverbose) System.out.println("Received tuple = " + receivedTuple);
    assert(receivedTuple.equals(tp1));
    receivedTuple = (Tuple) insertedTuples.removeFirst();
    if(vverbose) System.out.println("Received tuple = " + receivedTuple);
    assert(receivedTuple.equals(tp2));

    //write tp3, issue readAndWatch, write tp4, tp5
    // expect to recieve tp3, tp3, tp5
    if(vverbose) System.out.println("writing tp3");
    ptInterface.tupleWrite(objId, tp3);

    if(vverbose) System.out.println("issuing readandWatch");
    ptInterface.tupleReadAndWatch(objId, 100);
    assert(insertedTuples.size() == 1);

    if(vverbose) System.out.println("writing tp4 & tp5");
    ptInterface.tupleWrite(objId, tp4);
    try{
      Thread.currentThread().sleep(50);
    }catch(InterruptedException e) {};
    ptInterface.tupleWrite(objId, tp5);
    
    //hopefully this is long enough
    try{
      Thread.currentThread().sleep(500);
    }catch(InterruptedException e){} // igonore

    //check if the recevied tuples are correct
    assert(insertedTuples.size() == 3);
    receivedTuple = (Tuple) insertedTuples.removeFirst();
    if(vverbose) System.out.println("Received tuple = " + receivedTuple);
    assert(receivedTuple.equals(tp3));

    receivedTuple = (Tuple) insertedTuples.removeFirst();
    if(vverbose) System.out.println("Received tuple = " + receivedTuple);
    assert(receivedTuple.equals(tp4));

    receivedTuple = (Tuple) insertedTuples.removeFirst();
    if(vverbose) System.out.println("Received tuple = " + receivedTuple);
    assert(receivedTuple.equals(tp5));

    //stopwatch
    if(vverbose) System.out.println("issuing stopWatch");
    ptInterface.tupleStopWatch(objId);
    try{
      Thread.currentThread().sleep(50);
    }catch(InterruptedException e){} // igonore    
    
    // make some writes and make sure they are not inserted
    ptInterface.tupleWrite(objId, tp6);
    ptInterface.tupleWrite(objId, tp7);
    try{
      Thread.currentThread().sleep(500);
    }catch(InterruptedException e){} // igonore
    assert(insertedTuples.size() == 0);
  }
        
//---------------------------------------------------------------------------
// Main Method
//---------------------------------------------------------------------------

  public static void main(String s[]) {
    String name = "PersistentTupleInterfaceUnit";
    System.err.println(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
         vverbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new PersistentTupleInterfaceUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);

    System.exit(0);
  }
}



//---------------------------------------------------------------------------
/* $Log: PersistentTupleInterfaceUnit.java,v $
/* Revision 1.2  2007/09/17 21:53:02  nalini
/* updated persistentTupleUnit to work with new tuple str format
/*
/* Revision 1.1  2007/07/11 20:25:53  nalini
/* persistent tuple support added
/*
 */
//---------------------------------------------------------------------------
