//---------------------------------------------------------------------------
/* StressTestUnit.java
 * 
 * Unit test to have a lot of operations at the core and make sure there
 * is no blocking.
 *  - Client sets up subscriptions
 *  - Server makes unbound writes to objects and client will demandread 
 *  - Server makes bound writes, and will unbind the obj after the write.
 *
 * (C) Copyright 2004 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;


public class StressTestUnit extends TestCase {


  //public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean verbose = false; // Start/end of test
  protected static boolean vverbose = false; // Test internals

  private Process rmiregistry;
  private ProcessThread client;
  private ProcessThread server;
  private BarrierServer barrierServer;
  protected static int BARRIER_PORT = 8000;
  private int barrierCount = 10;

  protected static String CONFIG_PATH = "test" + File.separatorChar + "tmp.StressTestUnit.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar + "tmp.StressTestUnit.p2config";
  protected static long READER_NODE_ID = 0;
  protected static long WRITER_NODE_ID = 1;

  protected static int sizeOfWrites = 100;
  protected static int numOfObjs = 50; // increase value to increase stress
  protected static int numWritePerObj = 10; // increase value to increase stress

  protected static byte phase1Value = 65; //A
  protected static byte phase2Value = 66; //B
  protected static byte phase3Value = 67; //C
  protected static byte phase4Value = 68; //D

//---------------------------------------------------------------------------
//  Basic Constructor - called by test runners
//---------------------------------------------------------------------------

  public StressTestUnit(final String s){
    super (s);
  }

//---------------------------------------------------------------------------
//  Set up environment in which tests must run
//---------------------------------------------------------------------------
  protected void setUp() throws Exception {
    super.setUp();

    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }

    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    System.out.println("rmiregistry started");
    Thread.sleep(2000);

    //
    // Start barrier server for client & server to co-ordinate
    //
    
    barrierServer = new BarrierServer(BARRIER_PORT, 2, barrierCount);
    barrierServer.start();

    //
    // make the config files
    //
    makePractiConfig(CONFIG_PATH);
    makeP2Config(CONFIG_P2_PATH);

  }

//---------------------------------------------------------------------------
// Clearing up after test is over
//---------------------------------------------------------------------------
  protected void tearDown() throws Exception {
    if(client != null){
      client.destroyProcess();
    }
    if(server != null){
      server.destroyProcess();
    }
    
    barrierServer.interrupt();

    rmiregistry.destroy();
    System.out.println("rmiregistry terminated");
    super.tearDown();
  }

//---------------------------------------------------------------------------
// Carry out the tests
//---------------------------------------------------------------------------
  public void testStressTest(){
    try{
      ProcessState waitObj = new ProcessState();
  
      //
      //  Starts the Client. Use make to get paths right.
      //
      client = new ProcessThread("Client",
				 "make StressTestUnit_Client.unit",
				 waitObj);
      client.start();

      //
      //  Starts the Server. Use make to get paths right.
      //
      server = new ProcessThread("Server",
				 "make StressTestUnit_Server.unit",
				 waitObj);
      server.start();


      //
      // wait for at least one of them to end
      //
      waitObj.waitForProcess();

      if(waitObj.exitValue() != 0){
	assert(false): "Error in test";
      }

    }catch(Exception e){
      e.printStackTrace();
    }
  }
  
//---------------------------------------------------------------------------
//  Makes the config file for the experiment
//---------------------------------------------------------------------------
  private void makePractiConfig(String configPath){
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(READER_NODE_ID),
                            "localhost",
                            9988,
                            9989,
                            9991,
                            9992,
                            9990,
                            "test" + File.separatorChar + "tmp.StressTestUnit-" + 
			    READER_NODE_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9993,
                            9994,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
   Config.addOneNodeConfig(new NodeId(WRITER_NODE_ID),
                            "localhost",
                            9888,
                            9889,
                            9891,
                            9892,
                            9890,
                           "test" + File.separatorChar + "tmp.StressTestUnit-" + 
			   WRITER_NODE_ID+".db",
                            "/*",
                            -1L,
                            "localhost",
                            9893,
                            9894,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    
   Config.writeToFile(configPath);
  }

//---------------------------------------------------------------------------
//  Makes the P2 config file for the experiment
//---------------------------------------------------------------------------
  private void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(READER_NODE_ID), 5, 5, 3000, 3000, 2, 2, 2, 5, 5);
   P2Config.addOneNodeConfig(new NodeId(WRITER_NODE_ID), 5, 5, 3000, 3000, 2, 2, 2, 5, 5);
   P2Config.writeToFile(p2ConfigPath);
  }

  public static Test suite(){
    TestSuite suite = new TestSuite(StressTestUnit.class);
    return suite;
  }


  public static void main(String s[]) {
    String name = "StressTestUnit";
    System.err.println(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
         vverbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new StressTestUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);
  }
}



//---------------------------------------------------------------------------
/* $Log: StressTestUnit.java,v $
/* Revision 1.4  2007/04/12 17:09:32  zjiandan
/* fix across unit test mem leak.
/*
/* Revision 1.3  2007/03/09 03:01:38  nalini
/* removed update workers option from P2Config
/*
/* Revision 1.2  2006/11/15 17:48:46  nalini
/* fixing unit tests
/*
/* Revision 1.1  2006/10/13 20:48:22  nalini
/* StressTestUnit added
/*
 */
//---------------------------------------------------------------------------
