//---------------------------------------------------------------------------
/* StressTestUnit_Client.java
 * 
 * Implements the Client side for the StressTestUnit test
 *  - establishes subscriptions
 *  - tries to read last obj written by server
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------
import java.io.*;

public class StressTestUnit_Client {


  int myId;
  NodeId myNodeId;
  NodeId serverNodeId;
  LocalInterface lInterface;
  Object waitObj;
  P2Runtime runtime;
  private static final boolean verbose = true;
  

//---------------------------------------------------------------------------
//  Basic constructor
//---------------------------------------------------------------------------

  public StressTestUnit_Client() throws Exception{
    
    waitObj = new Object();
    myId = (int) StressTestUnit.READER_NODE_ID;
    myNodeId = new NodeId(StressTestUnit.READER_NODE_ID);
    serverNodeId = new NodeId(StressTestUnit.WRITER_NODE_ID);

    Policy policy = new StressTestClientPolicy();
    runtime = new P2Runtime(StressTestUnit.CONFIG_PATH,
                            StressTestUnit.CONFIG_P2_PATH,
                            myNodeId,
                            true, policy, false);
    runtime.start();

    policy.setRuntime(runtime);
 
    lInterface = new LocalInterface(runtime.getController(), runtime.getCore());

  }

  public void shutdown(){
    runtime.shutdown();
  }
  
//---------------------------------------------------------------------------
//  Main method - establishes subscriptions
//  and reads objs in phases
//
//---------------------------------------------------------------------------

  public static void main(String[] args){
    StressTestUnit_Client client = null;
    try{
      if (verbose) {
	System.out.println("StressTestUnit_Client Starting...");
      }
      
      client = new StressTestUnit_Client();
      BarrierClient c =  new BarrierClient("localhost", 
					   StressTestUnit.BARRIER_PORT,
					   client.getId());

      c.sendBarrierRequest(0, client.getId()); // Wait to "Go"
 
      client.establishSubscriptions();
      if (verbose) {
	System.out.println("ClientServerUnit_Client established subscriptions");
      }
      c.sendBarrierRequest(0, client.getId()); // finished establising subscriptions

      client.phase(c, 1);
      client.phase(c, 2);
      client.phase(c, 3);
      client.phase(c, 4);


      if (verbose) {
	System.out.println("StressTestUnit_Client Exiting... Bye Bye");
      }

      System.exit(0); 
    } catch(java.lang.AssertionError ae) {
      ae.printStackTrace();
      if(client!= null){
	client.shutdown();
	client = null;
      }
      System.exit(-1);
    } catch(Exception e){
      //ignore --WHY?
    
    
    }finally{
      if(client != null){
	client.shutdown();
	client = null;
      }
    }
  }
  
//---------------------------------------------------------------------------
// phases
//---------------------------------------------------------------------------

  private void phase(BarrierClient c, int phase) {
    
    if (verbose) {
	System.out.println("StressTestUnit_Client starting phase " + phase + "....");
    }

    c.sendBarrierRequest(0, getId()); // wait for server to write and sync

    readLastWrittenObj(phase);

    if (verbose) {
      System.out.println("StressTestUnit_Client finished reading last written obj");
    }

    c.sendBarrierRequest(0, getId()); // wait for server to write and sync

    if (verbose) {
      System.out.println("StressTestUnit_Client .... finished phase " + phase);
    }
  }

//---------------------------------------------------------------------------
//  Establishes Subscriptions
//---------------------------------------------------------------------------

  public void establishSubscriptions(){
    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/*");
     runtime.addInvalSubscription(new NodeId(StressTestUnit.WRITER_NODE_ID), 
                                  ss, false);
 
     runtime.addBodySubscription(new NodeId(StressTestUnit.WRITER_NODE_ID), 
                                 ss);
    
  }

//---------------------------------------------------------------------------
//  Reads last written obj
//---------------------------------------------------------------------------

  public void readLastWrittenObj(int phase){
    ImmutableBytes expected = getExpectedBytes(phase);

    try{
  
      ObjId objId = new ObjId("/"+ (StressTestUnit.numOfObjs -1));	
      while(true){
        try{
          ImmutableBytes b = lInterface.read(objId, 0, StressTestUnit.sizeOfWrites, true, true).getBody();
          if(verbose) {
            System.out.println("Obj read: "+ b);
          }
          assert(b.equals(expected)): "Read does not match expected in phase" ;
            break;
         
      }catch(ObjNotFoundException e){
          Thread.currentThread().sleep(100);
        }
      }
    }catch(Exception e){
      //  e.printStackTrace();
      // assert(false);
    }
  }

  private ImmutableBytes getExpectedBytes(int phase){
    byte[] exp = new byte[StressTestUnit.sizeOfWrites];

    switch(phase){
    case 1:
      for(int i = 0; i < exp.length; i++){
        exp[i] = StressTestUnit.phase1Value;
      }
      break;

    case 2:
      for(int i = 0; i < exp.length; i++){
        exp[i] = StressTestUnit.phase2Value;
      }
      break;
    
    case 3:
      for(int i = 0; i < exp.length; i++){
        exp[i] = StressTestUnit.phase3Value;
      }
      break;
    
    case 4:
      for(int i = 0; i < exp.length; i++){
        exp[i] = StressTestUnit.phase4Value;
      }
      break;
    }
    return new ImmutableBytes(exp);
  }

    


//---------------------------------------------------------------------------
// getID
//---------------------------------------------------------------------------
  public int getId(){
    return myId;
  }

}

//---------------------------------------------------------------------------
/* $Log: StressTestUnit_Client.java,v $
/* Revision 1.7  2007/06/05 20:49:24  nalini
/* exposed LOG|CP option for invalSubscriptions to P2Runtime and Overlog layer
/*
/* Revision 1.6  2007/04/12 17:09:32  zjiandan
/* fix across unit test mem leak.
/*
/* Revision 1.5  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.4  2007/02/12 06:19:49  zjiandan
/* Expose noSyncLog parameter, add more unittest for PRACTIFS, PangaeaFS.
/*
/* Revision 1.3  2006/12/08 23:13:18  nalini
/* reverted to old read interface
/*
/* Revision 1.1  2006/10/13 20:48:22  nalini
/* StressTestUnit added
/*
*/
//---------------------------------------------------------------------------
