//---------------------------------------------------------------------------
/* StressTestUnit_Server.java
 * 
 * Implements the Server side for the StressTestUnit test
 *  - server to create objects
 *  - wiats for client to establish subscriptions
 *  - server to update objects
 *  - server waits to sync with client
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------
import java.io.*;

public class StressTestUnit_Server {


  int myId;
  NodeId myNodeId;
  NodeId clientNodeId;
  LocalInterface lInterface;
  WaitObj waitObj;
  P2Runtime runtime;
  private static final boolean verbose = true;
  

//---------------------------------------------------------------------------
//  Basic constructor
//---------------------------------------------------------------------------

  public StressTestUnit_Server() throws Exception{
    
    waitObj = new WaitObj();
    myId = (int) StressTestUnit.WRITER_NODE_ID;
    myNodeId = new NodeId(StressTestUnit.WRITER_NODE_ID);
    clientNodeId = new NodeId(StressTestUnit.READER_NODE_ID);

    Policy policy = new StressTestServerPolicy(waitObj);
    runtime = new P2Runtime(StressTestUnit.CONFIG_PATH,
                                      StressTestUnit.CONFIG_P2_PATH,
                                      myNodeId,
			    true, policy, false);
    runtime.start();

    policy.setRuntime(runtime);
 
    lInterface = new LocalInterface(runtime.getController(), runtime.getCore());

  }
  
  public void shutdown(){
    runtime.shutdown();
  }

//---------------------------------------------------------------------------
//  Main method - waits for reader to subscribe,
//  writes objects in phases.
//
//---------------------------------------------------------------------------

  public static void main(String[] args){
    StressTestUnit_Server server = null;
    try{
      if (verbose) {
	System.out.println("StressTestUnit_Server Starting...");
      }
      
      server = new StressTestUnit_Server();
      BarrierClient c =  new BarrierClient("localhost", 
					   StressTestUnit.BARRIER_PORT,
					   server.getId());

      c.sendBarrierRequest(0, server.getId()); // Wait to "Go"
 
      if (verbose) {
	System.out.println("StressTestUnit_Server waiting for client to establish subscriptions");
      }

      c.sendBarrierRequest(0, server.getId()); // Wait for client to establish subscriptions
     
      server.phase(c, 1);
      server.phase(c, 2);
      server.phase(c, 3);
      server.phase(c, 4);

      if (verbose) {
	System.out.println("StressTestUnit_Server Exiting... Bye Bye");
      }
      
      
      System.exit(0); 
    } catch(java.lang.AssertionError ae) {
      ae.printStackTrace();
      if(server != null){
	server.shutdown();
	server = null;
      }
      System.exit(-1);
    } catch(Exception e){
      //ignore  --- WHY? Should at least print the exceptions
      e.printStackTrace();
    }finally{
      if(server != null){
	server.shutdown();
	server = null;
      }
    }
    
  }



//---------------------------------------------------------------------------
// phase
//---------------------------------------------------------------------------
  private void phase(BarrierClient c, int phase) {

    if (verbose) {
      System.out.println("StressTestUnit_Server starting phase " + phase + "....");
    }

    switch (phase) {
    case 1: 
      phase1Write();
      break;
    case 2: 
      phase2Write();
      break;
    case 3:
      phase3Write();
      break;
    case 4:
      phase4Write();
      break;
    }

    if (verbose) {
      System.out.println("StressTestUnit_Server finished writing, waiting to sync");
    }

    syncWithClient();
      
    if (verbose) {
      System.out.println("StressTestUnit_Server sync with client");
    }

      
    c.sendBarrierRequest(0, getId()); //Wait for client to read
    if (verbose) {
      System.out.println("StressTestUnit_Server  waiting for client to read data");
    }

    c.sendBarrierRequest(0, getId()); //Done
    if (verbose) {
      System.out.println("StressTestUnit_Server  ..... finished phase " + phase);
    }
  }

  
//---------------------------------------------------------------------------
// Phase 1: (Unbound) Writes numofObjs each numWritePerObj times.
//---------------------------------------------------------------------------

  public void phase1Write(){

    byte[] b = new byte[StressTestUnit.sizeOfWrites];
    for(int i = 0; i < b.length; i++){
      b[i] = StressTestUnit.phase1Value;
    }

    for (int i=0; i < StressTestUnit.numOfObjs; i++){
      ObjId objId = new ObjId("/"+ i);
      try{
        for(int j=0; j < StressTestUnit.numWritePerObj; j++){
          lInterface.write(objId, 0, b.length, b, false);
        }
      }catch(Exception e){
        e.printStackTrace();
        assert(false);
      }
    }

  }

//---------------------------------------------------------------------------
// Phase 2: (Unbound) Writes sequentially to each obj numWritePerObj times.
//---------------------------------------------------------------------------

  public void phase2Write(){

    byte[] b = new byte[StressTestUnit.sizeOfWrites];
    for(int i = 0; i < b.length; i++){
      b[i] = StressTestUnit.phase2Value;
    }

    for(int j=0; j < StressTestUnit.numWritePerObj; j++){
      for (int i=0; i < StressTestUnit.numOfObjs; i++){
        ObjId objId = new ObjId("/"+ i);
        try{
          lInterface.write(objId, 0, b.length, b, false);
        }catch(Exception e){
          e.printStackTrace();
          assert(false);
        }
      }
    }
  }


//---------------------------------------------------------------------------
// Phase 3: (Bound) Writes numofObjs each numWritePerObj times.
//---------------------------------------------------------------------------

  public void phase3Write(){

    byte[] b = new byte[StressTestUnit.sizeOfWrites];
    for(int i = 0; i < b.length; i++){
      b[i] = StressTestUnit.phase3Value;
    }

    for (int i=0; i < StressTestUnit.numOfObjs; i++){
      ObjId objId = new ObjId("/"+ i);
      try{
        for(int j=0; j < StressTestUnit.numWritePerObj; j++){
          lInterface.write(objId, 0, b.length, b, true);
        }
      }catch(Exception e){
        e.printStackTrace();
        assert(false);
      }
    }

  }

//---------------------------------------------------------------------------
// Phase 4: Bound Writes sequentially to each obj numWritePerObj times.
//---------------------------------------------------------------------------

  public void phase4Write(){

    byte[] b = new byte[StressTestUnit.sizeOfWrites];
    for(int i = 0; i < b.length; i++){
      b[i] = StressTestUnit.phase4Value;
    }

    for(int j=0; j < StressTestUnit.numWritePerObj; j++){
      for (int i=0; i < StressTestUnit.numOfObjs; i++){
        ObjId objId = new ObjId("/"+ i);
        try{
          lInterface.write(objId, 0, b.length, b, true);
        }catch(Exception e){
          e.printStackTrace();
          assert(false);
        }
      }
    }
  }


//---------------------------------------------------------------------------
// Sync with client:
// - reads the AcceptStamp of the last write
// - requests a sync from the client
// - waits until the synch has been received
//---------------------------------------------------------------------------
  public void syncWithClient(){
    try{
      ObjId objId = new ObjId ("/" + (StressTestUnit.numOfObjs-1));
      BodyMsg bMsg = lInterface.read(objId, 0, StressTestUnit.sizeOfWrites,
                                     true, true);

      AcceptStamp as = bMsg.getAcceptStamp();
      
      if(verbose) {
        System.out.println("Obj written: " + bMsg.getBody());
      }
      runtime.requestSync(clientNodeId, as);
      waitObj.waitOnObj();
    } catch(Exception e) {
      e.printStackTrace();
    }
  }
    
//---------------------------------------------------------------------------
// getID
//---------------------------------------------------------------------------
  public int getId(){
    return myId;
  }

}

//---------------------------------------------------------------------------
/* $Log: StressTestUnit_Server.java,v $
/* Revision 1.6  2007/04/12 17:09:32  zjiandan
/* fix across unit test mem leak.
/*
/* Revision 1.5  2007/04/02 22:10:03  nalini
/* junit fixes
/*
/* Revision 1.4  2007/02/12 06:19:49  zjiandan
/* Expose noSyncLog parameter, add more unittest for PRACTIFS, PangaeaFS.
/*
/* Revision 1.3  2006/12/08 23:13:18  nalini
/* reverted to old read interface
/*
/* Revision 1.1  2006/10/13 20:48:22  nalini
/* StressTestUnit added
/*
*/
//---------------------------------------------------------------------------
