//---------------------------------------------------------------------------
/* ThreeNodeTestUnit.java
 * 
 * A  3-node test.
 * -  Initiator = Node 0, Client = Node 1, Server = Node 2
 * -  Initiator sets up subscriptions from itself and client to server
 * -  Server makes bounded writes
 * -  Initiator and client read data
 * -  Initiator removes some subscriptions
 * -  Server makes unbounded writes
 * -  Initiator initiates demand reads
 * -  Initiator and client read data
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;


public class ThreeNodeTestUnit extends TestCase {


  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean verbose = true; // Start/end of test
  protected static boolean vverbose = true; // Test internals

  private Process rmiregistry;
  private ProcessThread node_0;
  private ProcessThread node_1;
  private ProcessThread node_2;
  private ProcessState waitObj;
  private BarrierServer barrierServer;
  protected static int BARRIER_PORT = 5000;

  protected static String CONFIG_PATH = "test" + File.separatorChar + "tmp.ThreeNodeTestUnit.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar + "tmp.ThreeNodeTestUnit.p2config";
  protected static long INITIATOR_NODE_ID = 0;
  protected static long CLIENT_NODE_ID = 1;
  protected static long SERVER_NODE_ID = 2;

  protected static int sizeOfWrites = 20;
  protected static int numOfObjs = 10;
  protected static byte initialValue = 65; //"A";
  protected static byte finalValue = 66; //"B"; 

//---------------------------------------------------------------------------
//  Basic Constructor - called by test runners
//---------------------------------------------------------------------------

  public ThreeNodeTestUnit(final String s){
    super (s);
  }

//---------------------------------------------------------------------------
//  Set up environment in which tests must run
//---------------------------------------------------------------------------
  protected void setUp() throws Exception {
    super.setUp();

    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }

    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    System.out.println("rmiregistry started");
    Thread.sleep(2000);

    //
    // Start barrier server for the 3 nodes to co-ordinate
    //
    
    barrierServer = new BarrierServer(BARRIER_PORT, 3, 9);
    barrierServer.start();

    //
    // make the config files
    //
    makePractiConfig(CONFIG_PATH);
    makeP2Config(CONFIG_P2_PATH);

  }

//---------------------------------------------------------------------------
// Clearing up after test is over
//---------------------------------------------------------------------------
  protected void tearDown() throws Exception {
    if(node_0 != null){
      node_0.destroyProcess();
    }

    if(node_1 != null){
      node_1.destroyProcess();
    }

    if(node_2 != null){
      node_2.destroyProcess();
    }

    barrierServer.interrupt();

    rmiregistry.destroy();
    System.out.println("rmiregistry terminated");
    
    super.tearDown();
  }

//---------------------------------------------------------------------------
// Carry out the tests
//---------------------------------------------------------------------------
  public void testThreeNode(){
    try{
      ProcessState waitObj = new ProcessState();
      //
      //  Starts the initiator. Use make to get paths right.
      //
      node_0 = new ProcessThread("Initiator",
				 "make ThreeNodeTestUnit_Initiator.unit",
				 waitObj);
      node_0.start();
      //
      //  Starts the Client. Use make to get paths right.
      //
      node_1 = new ProcessThread("Client",
				 "make ThreeNodeTestUnit_Client.unit",
				 waitObj);
      node_1.start();

      //
      //  Starts the Server. Use make to get paths right.
      //
      node_2 = new ProcessThread("Server",
				 "make ThreeNodeTestUnit_Server.unit",
				 waitObj);
      node_2.start();


      //
      // wait for at least one of them to end
      //
      waitObj.waitForProcess();

      if(waitObj.exitValue() != 0){
	assert(false): "Error in test";
      }

    }catch(Exception e){
      e.printStackTrace();
    }
  }

//---------------------------------------------------------------------------
//  Makes the config file for the experiment
//---------------------------------------------------------------------------
  private void makePractiConfig(String configPath){
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(INITIATOR_NODE_ID),
                            "localhost",
                            6588,
                            6589,
                            6591,
                            6592,
                            6590,
                            "test" + File.separatorChar + "tmp.ThreeNodeTestUnit-" + 
			    INITIATOR_NODE_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            6593,
                            6594,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
   Config.addOneNodeConfig(new NodeId(CLIENT_NODE_ID),
			   "localhost",
			   6488,
			   6489,
			   6491,
			   6492,
			   6490,
                           "test" + File.separatorChar + "tmp.ThreeNodeTestUnit-" + 
			   CLIENT_NODE_ID+".db",
			   "/*",
			   -1L,
			   "localhost",
			   6493,
			   6494,
			   -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);
   
   Config.addOneNodeConfig(new NodeId(SERVER_NODE_ID),
			   "localhost",
			   6388,
			   6389,
			   6391,
			   6392,
			   6390,
                           "test" + File.separatorChar + "tmp.ThreeNodeTestUnit-" + 
			   SERVER_NODE_ID+".db",
			   "/*",
			   -1L,
			   "localhost",
			   6393,
			   6394,
			   -1,
			   Config.CACHE_SIZE_BYTES_DEFAULT,
			   Config.MAX_LOG_DISK_SIZE_BYTES,
			   Config.MAX_LOG_MEM_SIZE_BYTES);

   Config.writeToFile(configPath);
  }

//---------------------------------------------------------------------------
//  Makes the P2 config file for the experiment
//---------------------------------------------------------------------------
  private void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(INITIATOR_NODE_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(CLIENT_NODE_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(SERVER_NODE_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.writeToFile(p2ConfigPath);
  }

  public static Test suite(){
    TestSuite suite = new TestSuite(ThreeNodeTestUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   PicShareReaderUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "ThreeNodeTestUnit";
    System.err.println(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
         vverbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new ThreeNodeTestUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);
  }
}


//---------------------------------------------------------------------------
/* $Log: ThreeNodeTestUnit.java,v $
/* Revision 1.13  2007/04/12 17:09:32  zjiandan
/* fix across unit test mem leak.
/*
/* Revision 1.12  2007/03/09 03:01:38  nalini
/* removed update workers option from P2Config
/*
/* Revision 1.11  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.10  2007/01/06 02:03:24  nalini
/* minor junit  bug fixes
/*
/* Revision 1.9  2006/12/08 22:33:31  nalini
/* changed port numbers
/*
/* Revision 1.8  2006/11/15 22:03:05  nalini
/* *** empty log message ***
/*
/* Revision 1.7  2006/11/15 17:48:46  nalini
/* fixing unit tests
/*
/* Revision 1.6  2006/10/12 17:30:38  nalini
/* changed port numbers
/*
/* Revision 1.5  2006/10/02 23:23:39  nalini
/* synchronization support added
/*
/* Revision 1.4  2006/09/12 22:18:05  dahlin
/* Working to get the unit tests to all run. Up to RandomAccessState now go through. Note that to encourage people to run RASUnit, I have changed the parameters to --quick-- versions that are less extensive tests.
/*
/* Revision 1.3  2006/09/04 15:32:58  dahlin
/* *** empty log message ***
/*
/* Revision 1.2  2006/08/31 23:43:50  nalini
/* minor code changes
/*
/* Revision 1.1  2006/08/30 21:38:56  nalini
/* Three node unit test added
/*
 */
//---------------------------------------------------------------------------
