//---------------------------------------------------------------------------
/* ThreeNodeTestUnit_Client.java
 * 
 * A  3-node test.
 * -  Initiator sets up subscriptions from itself and client to server
 * -  Server makes bounded writes
 * -  Initiator and client read data
 * -  Initiator removes some subscriptions
 * -  Server makes unbounded writes
 * -  Initiator initiates demand reads
 * -  Initiator and client read data
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------
import java.io.*;

public class ThreeNodeTestUnit_Client {

  NullReader nReader; 
  int myId;
  NodeId myNodeId;

  ImmutableBytes initialBytes;
  ImmutableBytes finalBytes;

  P2Runtime p2Runtime;
  RuntimeRMIServerImpl p2RMIServer;
  RuntimeRMIClient p2RMIClient;


  private static final boolean verbose = false;
  private static final boolean vverbose = false; //lots of debugging information

//---------------------------------------------------------------------------
//  Basic constructor
//---------------------------------------------------------------------------

  public ThreeNodeTestUnit_Client() throws Exception{
    
    nReader = new NullReader(ThreeNodeTestUnit.CONFIG_PATH,
			     ThreeNodeTestUnit.CONFIG_P2_PATH,
			     new NodeId(ThreeNodeTestUnit.CLIENT_NODE_ID));
    
    myId = (int)ThreeNodeTestUnit.CLIENT_NODE_ID;
    myNodeId =  new NodeId(ThreeNodeTestUnit.CLIENT_NODE_ID);   


    byte[] b1 = new byte[ThreeNodeTestUnit.sizeOfWrites];
    byte[] b2 = new byte[ThreeNodeTestUnit.sizeOfWrites];
    for(int i = 0; i < ThreeNodeTestUnit.sizeOfWrites; i++){
      b1[i] = ThreeNodeTestUnit.initialValue;
      b2[i] = ThreeNodeTestUnit.finalValue;

    }
    
    initialBytes = new ImmutableBytes(b1);
    finalBytes =  new ImmutableBytes(b2);
    p2Runtime = nReader.getRuntime();

    p2RMIServer = new RuntimeRMIServerImpl(p2Runtime);
    p2RMIServer.start();
    p2RMIClient = new RuntimeRMIClient(p2Runtime);
      
  }


//---------------------------------------------------------------------------
//  Reads each object and compares it with the expected value
//---------------------------------------------------------------------------

  public void readStuff(ImmutableBytes value){

    try{
      for (int i=0; i < ThreeNodeTestUnit.numOfObjs; i++){
	ObjId objId = new ObjId("/"+ i);
	
	while(true){
	  try{
	    ImmutableBytes b = nReader.read(objId, 0, ThreeNodeTestUnit.sizeOfWrites);
	    assert(b.equals(value)): "Read data does not match expected Value" ;
	    break;
	  }catch(ObjNotFoundException e){
	    //ignore
	    //System.out.println("Obj not found exception for "+objId);
	    Thread.currentThread().sleep(50);
	  }
	}
      }
    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }


//---------------------------------------------------------------------------
// getID
//---------------------------------------------------------------------------
  public int getId(){
    return myId;
  }


//---------------------------------------------------------------------------
//  Main method - reads data twice
//---------------------------------------------------------------------------

  public static void main(String[] args){
   
    int barrierCount = 0; 
    int maxBarrierCount = 9;
    BarrierClient c = null;
    ThreeNodeTestUnit_Client client = null;

    try{
      if (verbose) {
	System.out.println("ThreeNodeTestUnit_Client Starting...");
      }

      client = new ThreeNodeTestUnit_Client();

   
       c =  new BarrierClient("localhost", 
			      ThreeNodeTestUnit.BARRIER_PORT, 
			      client.getId());
       try{

	 if (vverbose) System.out.println("Barrier 1 ...");
	 barrierCount++;
	 c.sendBarrierRequest(0, client.getId()); // Wait to "Go"
      
 
	 if (vverbose) System.out.println("Barrier 2 ...");
	 barrierCount++;
	 c.sendBarrierRequest(0, client.getId()); // Wait for initiator to establishing subscriptions
 

	 if (vverbose) System.out.println("Barrier 3 ...");
	 barrierCount++;
	 c.sendBarrierRequest(0, client.getId()); // Wait for server to make bound writes

    
	 if (vverbose) System.out.println("Barrier 4 ...");
	 barrierCount++;
	 c.sendBarrierRequest(0, client.getId()); // Wait for initiator to check subscriptions
      
	 client.readStuff(client.initialBytes);
	 if(verbose){
	   System.out.println("ThreeNodeTestUnit_Client finished reading data 1");
	 }
 
	 if (vverbose) System.out.println("Barrier 5 ...");
	 barrierCount++;
	 c.sendBarrierRequest(0, client.getId()); // finished reading data
     
	 if (vverbose) System.out.println("Barrier 6 ...");
	 barrierCount++;
	 c.sendBarrierRequest(0, client.getId()); // wiat for initiator to remove subscriptions

	 if (vverbose) System.out.println("Barrier 7 ...");
	 barrierCount++;
	 c.sendBarrierRequest(0, client.getId()); //wait for server to make unbound writes

   
	 if (vverbose) System.out.println("Barrier 8 ...");
	 barrierCount++;
	 c.sendBarrierRequest(0, client.getId()); // wiat for initiator to initiate demand reads
      
	 client.readStuff(client.finalBytes);
	 if(verbose){
	   System.out.println("ThreeNodeTestUnit_Client finished reading data 2");
	 }

	 if (vverbose) System.out.println("Barrier 9 ...");
	 barrierCount++;
	 c.sendBarrierRequest(0, client.getId()); // done reading data

       }catch(java.lang.AssertionError ae){
	 ae.printStackTrace();
	 System.exit(-1);
       }
       System.exit(0);
    }
    catch(Exception e){
      if(vverbose) e.printStackTrace();
      assert(false);
      System.exit(-1);
    }
    System.exit(0);
  }
    
}

//---------------------------------------------------------------------------
/* $Log: ThreeNodeTestUnit_Client.java,v $
/* Revision 1.2  2007/04/02 22:10:04  nalini
/* junit fixes
/*
/* Revision 1.1  2006/08/30 21:38:56  nalini
/* Three node unit test added
/*
*/
//---------------------------------------------------------------------------
