//---------------------------------------------------------------------------
/* ThreeNodeTestUnit_Initiator.java
 * 
 * A  3-node test.
 * -  Initiator sets up subscriptions from itself and client to server
 * -  Server makes bounded writes
 * -  Initiator and client read data
 * -  Initiator removes some subscriptions
 * -  Server makes unbounded writes
 * -  Initiator initiates demand reads
 * -  Initiator and client read data
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------
import java.io.*;

public class ThreeNodeTestUnit_Initiator {

  NullReader nReader; 
  int myId;
  NodeId myNodeId;
  NodeId clientNodeId;
  NodeId serverNodeId;

  ImmutableBytes initialBytes;
  ImmutableBytes finalBytes;

  P2Runtime p2Runtime;
  RuntimeRMIServerImpl p2RMIServer;
  RuntimeRMIClient p2RMIClient;


  private static final boolean verbose = true;
  private static final boolean vverbose = true; //lots of debugging information

//---------------------------------------------------------------------------
//  Basic constructor
//---------------------------------------------------------------------------

  public ThreeNodeTestUnit_Initiator() throws Exception{
    
    nReader = new NullReader(ThreeNodeTestUnit.CONFIG_PATH,
			     ThreeNodeTestUnit.CONFIG_P2_PATH,
			     new NodeId(ThreeNodeTestUnit.INITIATOR_NODE_ID));
    
    myId = (int)ThreeNodeTestUnit.INITIATOR_NODE_ID;
    myNodeId =  new NodeId(ThreeNodeTestUnit.INITIATOR_NODE_ID);   
    clientNodeId =  new NodeId(ThreeNodeTestUnit.CLIENT_NODE_ID);
    serverNodeId =  new NodeId(ThreeNodeTestUnit.SERVER_NODE_ID);


    byte[] b1 = new byte[ThreeNodeTestUnit.sizeOfWrites];
    byte[] b2 = new byte[ThreeNodeTestUnit.sizeOfWrites];
    for(int i = 0; i < ThreeNodeTestUnit.sizeOfWrites; i++){
      b1[i] = ThreeNodeTestUnit.initialValue;
      b2[i] = ThreeNodeTestUnit.finalValue;

    }
    
    initialBytes = new ImmutableBytes(b1);
    finalBytes =  new ImmutableBytes(b2);
    p2Runtime = nReader.getRuntime();

    p2RMIServer = new RuntimeRMIServerImpl(p2Runtime);
    p2RMIServer.start();
    p2RMIClient = new RuntimeRMIClient(p2Runtime);
      
  }



//---------------------------------------------------------------------------
//  Establishes Subscriptions
//---------------------------------------------------------------------------

  public void establishSubscriptions() throws Exception{
    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/*");
   
    p2RMIClient.addInvalSubscription(serverNodeId, myNodeId, ss, false);
    p2RMIClient.addInvalSubscription(serverNodeId, clientNodeId, ss, false);
    p2RMIClient.addBodySubscription(serverNodeId, clientNodeId, ss);

    //
    // wait till subscriptions are established
    //
    Thread.currentThread().sleep(1000);
  }

//---------------------------------------------------------------------------
//  Check Subscriptions
//---------------------------------------------------------------------------

  public void checkSubscriptions() throws Exception {
    //
    // check the subscriptions
    //

    SubscriptionTable st = p2RMIClient.getEstablishedInvalSubscriptions(myNodeId);
    if(vverbose) System.out.println("initiator - inval subs " + st);
    if(st.size() != 1){  
      assert(false);
    }
 
    st = p2RMIClient.getEstablishedBodySubscriptions(myNodeId);
    if(vverbose) System.out.println("initiator - body subs " + st);
    if(st.size() != 0){  
      assert(false);
    }
 
  
    st = p2RMIClient.getEstablishedInvalSubscriptions(clientNodeId);
    if(vverbose)System.out.println("client - inval subs " + st);   
    if(st.size() != 1){
      assert(false);
    }

   
    st = p2RMIClient.getEstablishedBodySubscriptions(clientNodeId);
    if(vverbose)System.out.println("client - body subs " + st);   
    if(st.size() != 1 ){
      assert(false);
    }

    st = p2RMIClient.getEstablishedOutgoingInvalSubscriptions(serverNodeId);
    if(vverbose)System.out.println("server - inval subs " + st);   
    if(st.size() != 2){
      assert(false);
    }

    st = p2RMIClient.getEstablishedOutgoingBodySubscriptions(serverNodeId);
    if(vverbose) System.out.println("server - body subs " + st);   
    if(st.size() != 1 ){
      assert(false);
    }

  }

//---------------------------------------------------------------------------
//  Removes Subscriptions and checks them
//---------------------------------------------------------------------------

  public void removeSubscriptions() throws Exception{
    SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/*");
   
    p2RMIClient.removeInvalSubscription(serverNodeId, myNodeId, ss);
    p2RMIClient.removeBodySubscription(serverNodeId, clientNodeId, ss);

    //
    // wait till subscriptions are established
    //
    Thread.currentThread().sleep(10000);

    //
    // check the subscriptions
    //

    SubscriptionTable st = p2RMIClient.getEstablishedInvalSubscriptions(myNodeId);
    if(vverbose)System.out.println("initiator - inval subs " + st);
    if(st.size() != 0){ 
      assert(false);
    }
 
    st = p2RMIClient.getEstablishedInvalSubscriptions(clientNodeId);
    if(vverbose) System.out.println("client - inval subs " + st);   
    if(st.size() != 1){
      assert(false);
    }

    st = p2RMIClient.getEstablishedBodySubscriptions(clientNodeId);
    if(vverbose)System.out.println("client - body subs " + st);   
    if(st.size() != 0 ){
      assert(false);
    }

    st = p2RMIClient.getEstablishedOutgoingInvalSubscriptions(serverNodeId);
    if(vverbose)System.out.println("server - inval subs " + st);   
    if(st.size() != 1){
      assert(false);
    }

    st = p2RMIClient.getEstablishedOutgoingBodySubscriptions(serverNodeId);
    if(vverbose) System.out.println("server - body subs " + st);   
    if(st.size() != 0 ){
      assert(false);
    }
    
  }




//---------------------------------------------------------------------------
//  Reads each object and compares it with the expected value
//---------------------------------------------------------------------------

  public void readStuff(ImmutableBytes value){

    try{
      for (int i=0; i < ThreeNodeTestUnit.numOfObjs; i++){
	ObjId objId = new ObjId("/"+ i);
	
	while(true){
	  try{
	    ImmutableBytes b = nReader.read(objId, 0, ThreeNodeTestUnit.sizeOfWrites);
	    assert(b.equals(value)): "Read data does not match expected Value" ;
	    break;
	  }catch(ObjNotFoundException e){
	    //ignore
	    //System.out.println("Obj not found exception for "+objId);
	    Thread.currentThread().sleep(50);
	  }
	}
      }
    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }



//---------------------------------------------------------------------------
//  initiates demand reads from the client to the server
//---------------------------------------------------------------------------
    public void initiateDemandReads() throws Exception{
      for(int i = 0; i < ThreeNodeTestUnit.numOfObjs; i++){
	ObjId objId = new ObjId("/"+i);
	p2RMIClient.demandRead(serverNodeId, 
			       clientNodeId, 
			       objId, 0, 
			       ThreeNodeTestUnit.sizeOfWrites,
			       new AcceptStamp(-1, new NodeId(-1)));
      }

      //
      // sleep for a while so that the demand reads can be fulfilled
      //
      Thread.currentThread().sleep(1000);
    }


//---------------------------------------------------------------------------
// getID
//---------------------------------------------------------------------------
  public int getId(){
    return myId;
  }


//---------------------------------------------------------------------------
//  Main method - establishes subscriptions, and reads data received from
//  the server to make sure we have the correct value.
//---------------------------------------------------------------------------

  public static void main(String[] args){
   
    int barrierCount = 0; 
    int maxBarrierCount = 9;
    BarrierClient c = null;
    ThreeNodeTestUnit_Initiator ini = null;

    try{
      if (verbose) {
	System.out.println("ThreeNodeTestUnit_Initiator Starting...");
      }

      ini = new ThreeNodeTestUnit_Initiator();

   
       c =  new BarrierClient("localhost", 
			      ThreeNodeTestUnit.BARRIER_PORT, 
			      ini.getId());

      if (vverbose) System.out.println("Barrier 1 ...");
      barrierCount++;
      c.sendBarrierRequest(0, ini.getId()); // Wait to "Go"

      try{
	ini.establishSubscriptions();
	if (verbose) {
	  System.out.println("ThreeNodeTestUnit_Initiator established subscriptions");
	}

	if (vverbose) System.out.println("Barrier 2 ...");
	barrierCount++;
	c.sendBarrierRequest(0, ini.getId()); // Done establishing subscriptions


	if (vverbose) System.out.println("Barrier 3 ...");
	barrierCount++;
	c.sendBarrierRequest(0, ini.getId()); // Wait for server to make bound writes
      
	Thread.currentThread().sleep(1000);
	ini.checkSubscriptions(); 
	if (verbose) {
	  System.out.println("ThreeNodeTestUnit_Initiator finished checking subscriptions");
	}
      
	if (vverbose) System.out.println("Barrier 4 ...");
	barrierCount++;
	c.sendBarrierRequest(0, ini.getId()); // Finished checking subscriptions
 
	ini.readStuff(ini.initialBytes);
	if(verbose){
	  System.out.println("ThreeNodeTestUnit_Initiator finished reading data");
	}


	if (vverbose) System.out.println("Barrier 5 ...");
	barrierCount++;
	c.sendBarrierRequest(0, ini.getId()); // finished reading data

	ini.removeSubscriptions();
	if (verbose) {
	  System.out.println("ThreeServerUnit_Initiator removed subscriptions");
	}

	if (vverbose) System.out.println("Barrier 6 ...");
	barrierCount++;
	c.sendBarrierRequest(0, ini.getId()); // removed subscriptions

	if (vverbose) System.out.println("Barrier 7 ...");
	barrierCount++;
	c.sendBarrierRequest(0, ini.getId()); //wait for server to make unbound writes

	ini.initiateDemandReads();
	if (verbose) {
	  System.out.println("ThreeServerUnit_Initiator finished initiating DemandReads");
	}

	if (vverbose) System.out.println("Barrier 8 ...");
	barrierCount++;
	c.sendBarrierRequest(0, ini.getId()); // finished initiating demand reads
      

	if (vverbose) System.out.println("Barrier 9...");
	barrierCount++;
	c.sendBarrierRequest(0, ini.getId()); // waiting for client to read data
	
      }catch(java.lang.AssertionError ae){
	ae.printStackTrace();
	System.exit(-1);
      }

      System.exit(0);

    }catch(Exception e){
      if(vverbose) e.printStackTrace();
      System.exit(-1);
    }
    System.exit(0);
  }
    
}

//---------------------------------------------------------------------------
/* $Log: ThreeNodeTestUnit_Initiator.java,v $
/* Revision 1.8  2007/06/05 20:49:24  nalini
/* exposed LOG|CP option for invalSubscriptions to P2Runtime and Overlog layer
/*
/* Revision 1.7  2007/04/02 22:10:04  nalini
/* junit fixes
/*
/* Revision 1.6  2007/03/09 03:01:38  nalini
/* removed update workers option from P2Config
/*
/* Revision 1.5  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.4  2007/02/01 06:12:09  zjiandan
/* Add acceptStamp to demandRead so that the sender only sends the data
/* that's at least as new as the acceptStamp.
/*
/* Revision 1.3  2006/10/02 23:23:39  nalini
/* synchronization support added
/*
/* Revision 1.2  2006/09/13 15:32:51  nalini
/* Minor Updates
/*
/* Revision 1.1  2006/08/30 21:38:56  nalini
/* Three node unit test added
/*
*/
//---------------------------------------------------------------------------
