//---------------------------------------------------------------------------
/* ThreeNodeTestUnit_Server.java
 * 
 * A  3-node test.
 * -  Initiator sets up subscriptions from itself and server to server
 * -  Server makes bounded writes
 * -  Initiator and server read data
 * -  Initiator removes some subscriptions
 * -  Server makes unbounded writes
 * -  Initiator initiates demand reads
 * -  Initiator and server read data
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------
import java.io.*;

public class ThreeNodeTestUnit_Server {

  Writer writer;
  int myId;
  NodeId myNodeId;

  ImmutableBytes initialBytes;
  ImmutableBytes finalBytes;

  P2Runtime p2Runtime;
  RuntimeRMIServerImpl p2RMIServer;
  RuntimeRMIClient p2RMIClient;


  private static final boolean verbose = false;
  private static final boolean vverbose = false; //lots of debugging information

//---------------------------------------------------------------------------
//  Basic constructor
//---------------------------------------------------------------------------

  public ThreeNodeTestUnit_Server() throws Exception{
    
    writer = new Writer(ThreeNodeTestUnit.CONFIG_PATH,
			ThreeNodeTestUnit.CONFIG_P2_PATH,
			new NodeId(ThreeNodeTestUnit.SERVER_NODE_ID));
    
    myId = (int)ThreeNodeTestUnit.SERVER_NODE_ID;
    myNodeId =  new NodeId(ThreeNodeTestUnit.SERVER_NODE_ID);   


    byte[] b1 = new byte[ThreeNodeTestUnit.sizeOfWrites];
    byte[] b2 = new byte[ThreeNodeTestUnit.sizeOfWrites];
    for(int i = 0; i < ThreeNodeTestUnit.sizeOfWrites; i++){
      b1[i] = ThreeNodeTestUnit.initialValue;
      b2[i] = ThreeNodeTestUnit.finalValue;

    }
    
    initialBytes = new ImmutableBytes(b1);
    finalBytes =  new ImmutableBytes(b2);
    p2Runtime = writer.getRuntime();

    p2RMIServer = new RuntimeRMIServerImpl(p2Runtime);
    p2RMIServer.start();
    p2RMIClient = new RuntimeRMIClient(p2Runtime);
      
  }



//---------------------------------------------------------------------------
// Write Objects 
//---------------------------------------------------------------------------

  public void writeStuff(ImmutableBytes imb, boolean bound){
    byte[] b = imb.getCopyBytes();

    for (int i=0; i < ThreeNodeTestUnit.numOfObjs; i++){
      ObjId objId = new ObjId("/"+ i);
      try{
	writer.write(objId, 0, b.length, b, bound);
	}catch(Exception e){
	  e.printStackTrace();
	  assert(false);
	}
    }

  }

//---------------------------------------------------------------------------
// getID
//---------------------------------------------------------------------------
  public int getId(){
    return myId;
  }


//---------------------------------------------------------------------------
//  Main method - writes bound data, then writes unbound data
//---------------------------------------------------------------------------

  public static void main(String[] args){
   
    int barrierCount = 0; 
    int maxBarrierCount = 9;
    BarrierClient c = null;
    ThreeNodeTestUnit_Server server = null;

    try{
      if (verbose) {
	System.out.println("ThreeNodeTestUnit_Server Starting...");
      }

      server = new ThreeNodeTestUnit_Server();

   
      c =  new BarrierClient("localhost", 
			     ThreeNodeTestUnit.BARRIER_PORT, 
			     server.getId());
      try{

	if (vverbose) System.out.println("Barrier 1 ...");
	barrierCount++;
	c.sendBarrierRequest(0, server.getId()); // Wait to "Go"
      
     
	if (vverbose) System.out.println("Barrier 2 ...");
	barrierCount++;
	c.sendBarrierRequest(0, server.getId()); // Wait for initiator to establishing subscriptions

	server.writeStuff(server.initialBytes, true);
	if(verbose) {
	  System.out.println("ThreeNodeTestUnit_Server finished making bound writes");
	}
	if (vverbose) System.out.println("Barrier 3 ...");
	barrierCount++;
	c.sendBarrierRequest(0, server.getId()); // finished bound writes

	if (vverbose) System.out.println("Barrier 4 ...");
	barrierCount++;
	c.sendBarrierRequest(0, server.getId()); // wait for initiator to check subscriptions
      
 
	if (vverbose) System.out.println("Barrier 5 ...");
	barrierCount++;
	c.sendBarrierRequest(0, server.getId()); // wait for client and initiator to read data

 
	if (vverbose) System.out.println("Barrier 6 ...");
	barrierCount++;
	c.sendBarrierRequest(0, server.getId()); // wiat for initiator to remove subscriptions

	server.writeStuff(server.finalBytes, false);
	if(verbose) {
	  System.out.println("ThreeNodeTestUnit_Server finished making unbounded writes");
	}

	if (vverbose) System.out.println("Barrier 7 ...");
	barrierCount++;
	c.sendBarrierRequest(0, server.getId()); //finished making unbound writes


	if (vverbose) System.out.println("Barrier 8 ...");
	barrierCount++;
	c.sendBarrierRequest(0, server.getId()); // wiat for initiator to initiate demand reads
      

	if (vverbose) System.out.println("Barrier 9 ...");
	barrierCount++;
	c.sendBarrierRequest(0, server.getId()); // wait for client to read data

      }catch(java.lang.AssertionError ae){
	ae.printStackTrace();
	System.exit(-1);
      }
      System.exit(0);
    }
    catch(Exception e){
      if(vverbose) e.printStackTrace();
      assert(false);
      System.exit(-1);
    }
    System.exit(0);
  }
    
}

//---------------------------------------------------------------------------
/* $Log: ThreeNodeTestUnit_Server.java,v $
/* Revision 1.2  2007/04/02 22:10:04  nalini
/* junit fixes
/*
/* Revision 1.1  2006/08/30 21:38:56  nalini
/* Three node unit test added
/*
*/
//---------------------------------------------------------------------------
