//---------------------------------------------------------------------------
/* HandlerTest.java
 *
 * Reads objects, if there is a read miss, demands read from server
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.util.*;
import java.io.*;
import java.lang.*;

public class HandlerTest { 

  static private Process rmiRegistry;
  static private BarrierServer barrierServer;
  protected static int BARRIER_PORT = 3000;

  protected static String CONFIG_PATH = "test" + File.separatorChar + "tmp.Handler.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar + "tmp.Handler.p2config";
  protected static String NODEID_MAP_PATH = "test" + File.separatorChar + "tmp.Handler.map";
  protected static String OVERLOG_PATH = "HandlerTest.olg";

  protected static long READER_NODE_ID = 0;
  protected static long WRITER_NODE_ID = 1;

  protected static int sizeOfWrites = 20;
  protected static int numOfObjs = 1;
  protected static byte initialValue = 65; //"A";
  protected static byte secondValue = 66; //"B"; 
  protected static byte thirdValue = 66; //"B"; 

  static P2Runtime runtime;
  static AllReadBlockAllWriteUnboundInterface localInterface;

  private static void setUpEnv() throws Exception{

    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }

    //
    // Start the registry
    //
    rmiRegistry = Runtime.getRuntime().exec("rmiregistry");
    System.out.println("rmiregistry started");
    Thread.sleep(1000);

    //
    // Start barrier server - for client and server to synch with each other
    //
    
    barrierServer = new BarrierServer(BARRIER_PORT, 2, 4);
    barrierServer.start();
    System.out.println("BarrierServer Started");

    //
    // make the config files
    //
    makePractiConfig(CONFIG_PATH);
    makeP2Config(CONFIG_P2_PATH);
    makeNodeIdMap(NODEID_MAP_PATH);
  }

  private static void tearDown() throws Exception {
    rmiRegistry.destroy();
    System.out.println("rmiregistry terminated");
  }


  private static void startOverlogNode() throws Exception {
    OverlogPolicy overlogPolicy = new OverlogPolicy(OVERLOG_PATH, new NodeId(READER_NODE_ID), NODEID_MAP_PATH);
    runtime = new P2Runtime(CONFIG_PATH, CONFIG_P2_PATH, new NodeId(READER_NODE_ID), true, overlogPolicy, false);
    overlogPolicy.setRuntime(runtime);
    runtime.start();
    overlogPolicy.startOverlog();
    Thread.currentThread().sleep(500); // wait for overlog to start

    localInterface = new AllReadBlockAllWriteUnboundInterface(runtime.getController(), runtime.getCore());
  }
    
  
//---------------------------------------------------------------------------
// Make configuration files
//---------------------------------------------------------------------------
  private static void makePractiConfig(String configPath){
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(READER_NODE_ID),
                            "localhost",
                            9988,
                            9989,
                            9991,
                            9992,
                            9990,
                            "test" + File.separatorChar + "tmp.HandlerUnit-" + 
			    READER_NODE_ID + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9993,
                            9994,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
   Config.addOneNodeConfig(new NodeId(WRITER_NODE_ID),
                            "localhost",
                            9888,
                            9889,
                            9891,
                            9892,
                            9890,
                           "test" + File.separatorChar + "tmp.HandlerUnit-" + 
			   WRITER_NODE_ID+".db",
                            "/*",
                            -1L,
                            "localhost",
                            9893,
                            9894,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    
   Config.writeToFile(configPath);
  }

  private static void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(READER_NODE_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(WRITER_NODE_ID), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.writeToFile(p2ConfigPath);
  }

  private static void makeNodeIdMap(String nodeIdMapPath) {
    NodeIdMap.createEmptyMap();
    NodeIdMap.add(new NodeId(READER_NODE_ID), new OverlogId("localhost:5000"));
    NodeIdMap.add(new NodeId(WRITER_NODE_ID), new OverlogId("localhost:5001"));
    NodeIdMap.writeToFile(nodeIdMapPath);
  }


//---------------------------------------------------------------------------
//  Reads each object and prints it out
//---------------------------------------------------------------------------

  public static void readStuff(){

    try{
      for (int i=0; i < numOfObjs; i++){
	ObjId objId = new ObjId("/"+ i);
	
	while(true){
	  try{
	    ImmutableBytes b = localInterface.read(objId, 0, sizeOfWrites);
            System.out.println(b);
	    break;
	  }catch(ObjNotFoundException e){
	    //ignore
	    //System.out.println("Obj not found exception for "+objId);
	    Thread.currentThread().sleep(500);
	  }
	}
      }
    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }

//---------------------------------------------------------------------------
// Main method  - implements the "reader"
//---------------------------------------------------------------------------
  public static void main(String[] args) {
    try{
      
      BarrierClient c =  new BarrierClient("localhost", 
                                          BARRIER_PORT,
                                          0);
      //
      //set up stuff
      //
      setUpEnv();
      c.sendBarrierRequest(0, 0); // Inform helper that stuff has be set up

      System.out.println("Starting Overlog Node");
      startOverlogNode();
      Thread.currentThread().sleep(10000); //wait for overlog to start

      c.sendBarrierRequest(0, 0); // Wait to "Go"
      
       c.sendBarrierRequest(0, 0); // Wait for server to write stuff

      System.out.println("Handler_Read starting to read stuff");
      readStuff();
     
      System.out.println("Handler_Reader finished reading stuff");
	
      c.sendBarrierRequest(0, 0); // Finished Reading Data
      
      System.out.println("Handler_Reader Exiting... Bye Bye");
	
      /*tearDown();*/

    }catch(Exception e) {
      e.printStackTrace();
    }

  }
  
}
//---------------------------------------------------------------------------
/* $Log: HandlerTest.java,v $
/* Revision 1.4  2007/03/09 03:01:38  nalini
/* removed update workers option from P2Config
/*
/* Revision 1.3  2007/03/01 06:38:24  nalini
/* fixed compilation errors
/*
/* Revision 1.2  2007/01/06 01:04:35  nalini
/* minor bug fixes for requestSync(VV)
/*
/* Revision 1.1  2006/10/17 23:39:51  nalini
/* handler test added
/*
*/
//---------------------------------------------------------------------------
