//---------------------------------------------------------------------------
/*  PingTest.java
 * 
 *  Simple test which reports the time it takes to 
 *  make a local Ping and a remote Ping
 * 
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------


public class PingTest {

  public static String localHost = "localhost";
  public static String localPort = "5000";
  public static String remoteHost = "localhost";
  public static String remotePort = "5001";
  public static int barrierPort = 8000;

  private String olgFilename;  
  private P2JavaWrapper p2Wrapper;
  private Watcher watcher;
  private Overlog overlog;

  private long localPingInsertTime;
  private long localPingReplyTime;
  private long remotePingInsertTime;
  private long remotePingReplyTime;

  public long totalLocalPingTime;
  public long totalRemotePingTime;

  public PingTest(String overlog){
    this.olgFilename = overlog;

    p2Wrapper = new P2JavaWrapper(olgFilename,
				  localHost,
				  localPort);

    watcher = new PingTestWatcher(this);
    String[] watchTuples = {"localPingReply", "remotePingReply"};

    p2Wrapper.setWatchTuples(watchTuples);
    p2Wrapper.setWatcher(watcher);

  }
   
  public synchronized void startOverlog() throws Exception{
    // starts overlog in another htread
    wait(1000);
    OverlogThread t = new OverlogThread(p2Wrapper);
    t.start();
    wait(3000);
  }
				       

  public synchronized void localPing() {
    try{     
      String[] stringA = {"localPingRequest", 
			  localHost+ ":" + localPort};
      Tuple tuple = new Tuple(stringA);
      System.out.println("Sending local ping");
      localPingInsertTime = System.currentTimeMillis();
      p2Wrapper.insertTuple(tuple);
      wait();
      totalLocalPingTime += localPingReplyTime - localPingInsertTime;
      System.out.println("LocalPing:" + (localPingReplyTime - localPingInsertTime) + "ms");
    }catch (InterruptedException e) {
    }
  }

  public synchronized void receivedLocalPingReply() {
    localPingReplyTime = System.currentTimeMillis();
    System.out.println("Received local ping reply");
    notifyAll();
  }

  
  public synchronized void remotePing() {
    try{
      String[] stringA = {"remotePingRequest", 
			  localHost+ ":" + localPort,
			  remoteHost+ ":" + remotePort};
      Tuple tuple = new Tuple(stringA);
      System.out.println("Sending remote ping");
      remotePingInsertTime = System.currentTimeMillis();
      p2Wrapper.insertTuple(tuple);
      wait();
      remotePingReplyTime = System.currentTimeMillis();
      totalRemotePingTime += remotePingReplyTime - remotePingInsertTime;
    
      System.out.println("RemotPing:" + (remotePingReplyTime - remotePingInsertTime) + "ms");
    }catch (InterruptedException e) {
    }
  }

  public synchronized void receivedRemotePingReply() {
    remotePingReplyTime = System.currentTimeMillis();
    System.out.println("Received remote ping reply");
    notifyAll();
  }
    
  public static void main(String[] args) {
    if (args.length <1 ) {
      System.out.println("Usage: PingTest <overlogFilename>");
      return;
    }
    
    int NUM_ITER = 500;

    try{ 
      BarrierServer bServer= new BarrierServer(barrierPort, 2, 3);
      bServer.start();
      
      BarrierClient c = new BarrierClient(localHost, barrierPort, 0);
      
      PingTest pingTest = new PingTest(args[0]);
      System.out.println("PingTest Initialized, waiting for Helper to start");
      c.sendBarrierRequest(0, 0); // wait for helper to initialize
      
      System.out.println("Starting overlog");
      pingTest.startOverlog();
      c.sendBarrierRequest(0, 0); // wait for overlog to start
      
      System.out.println("LocalPingTest ...");
      for(int i= 0; i < NUM_ITER; i++){
	pingTest.localPing();
      }



      System.out.println("RemotePingTest ...");
      for(int i=0; i < NUM_ITER; i++){
	pingTest.remotePing();
      }
      c.sendBarrierRequest(0, 0); // Test Ended
      System.out.println("Local Ping Avg =" + (pingTest.totalLocalPingTime/NUM_ITER) + "ms");
      System.out.println("Remote Ping Avg =" + (pingTest.totalRemotePingTime/NUM_ITER) + "ms");
      System.out.println("Test ended");
      System.exit(0);
    } catch(Exception e) {
      e.printStackTrace();
    }
  }

  class PingTestWatcher implements Watcher {
    PingTest test;

    public PingTestWatcher(PingTest test) {
      this.test = test;
    }

    public synchronized void receivedTuple(Tuple tp) {
      if(tp.getItem(0).equals("localPingReply")) {
	test.receivedLocalPingReply();
      } 
      else if(tp.getItem(0).equals("remotePingReply")){
	test.receivedRemotePingReply();
      }
      else {
        System.out.println("Unrecognized tuple received: " + tp);
      }
    } 
  }
}
    
//---------------------------------------------------------------------------
/* $Log: PingTest.java,v $
/* Revision 1.4  2007/09/12 19:07:55  nalini
/* upgraded to p2-0.8.2
/*
/* Revision 1.3  2007/03/18 04:12:22  nalini
/* updated ping test
/*
/* Revision 1.2  2007/03/16 23:57:35  nalini
/* p2serverinterface added
/*
/* Revision 1.1  2007/02/16 16:42:05  nalini
/* Test to measure ping performance added
/*
*/
//---------------------------------------------------------------------------
