//---------------------------------------------------------------------------
/* Subscribe test
 *
 * Does a bunch of writes and then waits for overlog to take over
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.util.*;
import java.io.*;
import java.lang.*;

public class SubscribeTest { 

  static private Process rmiRegistry;
  static private BarrierServer barrierServer;
  protected static int BARRIER_PORT = 3000;

  protected static String CONFIG_PATH = "test" + File.separatorChar + "tmp.Subscribe.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar + "tmp.Subscribe.p2config";
  protected static String NODEID_MAP_PATH = "test" + File.separatorChar + "tmp.Subscribe.map";
  protected static String OVERLOG_PATH = "SubscribeTest.olg";

  protected static long NODE_0 = 0;
  protected static long NODE_1 = 1;

  protected static int sizeOfWrites = 20;
  protected static int numOfObjs = 10;
  protected static byte initialValue = 65; //"A";
 
  static OverlogPolicy overlogPolicy;
  static P2Runtime runtime;
  static AllReadBlockAllWriteUnboundInterface localInterface;
  
  static boolean verbose = true;

  private static void setUpEnv(NodeId nodeId) throws Exception{

    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }

    //
    // Start the registry
    //
    rmiRegistry = Runtime.getRuntime().exec("rmiregistry");
    System.out.println("rmiregistry started");
    Thread.sleep(1000);

    //
    // Start barrier server - for client and server to synch with each other
    //
    if(nodeId.equals(new NodeId(0))){
         barrierServer = new BarrierServer(BARRIER_PORT, 2, 1);
         barrierServer.start();
         System.out.println("BarrierServer Started");
    }
    //
    // make the config files
    //
    makePractiConfig(CONFIG_PATH);
    makeP2Config(CONFIG_P2_PATH);
    makeNodeIdMap(NODEID_MAP_PATH);
  }

  private static void tearDown() throws Exception {
    rmiRegistry.destroy();
    System.out.println("rmiregistry terminated");
  }


  private static void startOverlogNode(NodeId nodeId) throws Exception {
     
    overlogPolicy = new OverlogPolicy(OVERLOG_PATH, nodeId, NODEID_MAP_PATH);
    runtime = new P2Runtime(CONFIG_PATH, CONFIG_P2_PATH, nodeId, true, overlogPolicy, false);
    overlogPolicy.setRuntime(runtime);
    runtime.start();
    overlogPolicy.startOverlog();
    Thread.currentThread().sleep(500); // wait for overlog to start

    localInterface = new AllReadBlockAllWriteUnboundInterface(runtime.getController(), runtime.getCore());
  }
    
  
//---------------------------------------------------------------------------
// Make configuration files
//---------------------------------------------------------------------------
  private static void makePractiConfig(String configPath){
    Config.createEmptyConfig();
    Config.addOneNodeConfig(new NodeId(NODE_0),
                            "localhost",
                            9988,
                            9989,
                            9991,
                            9992,
                            9990,
                            "test" + File.separatorChar + "tmp.HandlerUnit-" + 
			    NODE_0 + ".db",
                            "/*",
                            -1L,
                            "localhost",
                            9993,
                            9994,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
 
   Config.addOneNodeConfig(new NodeId(NODE_1),
                            "localhost",
                            9888,
                            9889,
                            9891,
                            9892,
                            9890,
                           "test" + File.separatorChar + "tmp.HandlerUnit-" + 
			   NODE_1+".db",
                            "/*",
                            -1L,
                            "localhost",
                            9893,
                            9894,
                            -1,
  			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    
   Config.writeToFile(configPath);
  }

  private static void makeP2Config(String p2ConfigPath){

   P2Config.createEmptyConfig();
   P2Config.addOneNodeConfig(new NodeId(NODE_0), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.addOneNodeConfig(new NodeId(NODE_1), 5, 5, 3000, 3000, 2, 2, 2, 2, 2);
   P2Config.writeToFile(p2ConfigPath);
  }

  private static void makeNodeIdMap(String nodeIdMapPath) {
    NodeIdMap.createEmptyMap();
    NodeIdMap.add(new NodeId(NODE_0), new OverlogId("localhost:5000"));
    NodeIdMap.add(new NodeId(NODE_1), new OverlogId("localhost:5001"));
    NodeIdMap.writeToFile(nodeIdMapPath);
  }


//---------------------------------------------------------------------------
//  Reads each object and prints it out
//---------------------------------------------------------------------------

  public static void readStuff(){

    try{
      for (int i=0; i < numOfObjs; i++){
	ObjId objId = new ObjId("/"+ i);
	
	while(true){
	  try{
	    ImmutableBytes b = localInterface.read(objId, 0, sizeOfWrites);
            if(verbose){
              System.out.println(b);
            }
	    break;
	  }catch(ObjNotFoundException e){
	    //ignore
	    //System.out.println("Obj not found exception for "+objId);
	    Thread.currentThread().sleep(500);
	  }
	}
      }
    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }

//---------------------------------------------------------------------------
// Write Objects 
//---------------------------------------------------------------------------

  public static void writeStuff(byte value){

    byte[] b = new byte[sizeOfWrites];
    for(int i = 0; i < sizeOfWrites; i++){
      b[i] = value;
    }

    ImmutableBytes imb = new ImmutableBytes(b);
    for (int i=0; i < numOfObjs; i++){
      ObjId objId = new ObjId("/"+ i);
      try{
	localInterface.write(objId, 0, sizeOfWrites, imb, false);
	}catch(Exception e){
	  e.printStackTrace();
	  assert(false);
	}
    }

  }

//---------------------------------------------------------------------------
// Main method  - implements the "reader"
//---------------------------------------------------------------------------
  public static void main(String[] args) {

    try{

      if(args.length < 1) {
        System.out.println("Usage: SubscribeTest [0|1]");
        System.exit(0);
      }

      // set up nodeId
      NodeId nodeId;
      if (args[0].equals("0")) {
        nodeId = new NodeId(NODE_0);
      } else {
        nodeId = new NodeId(NODE_1);
      }


      //
      //set up stuff
      //
      setUpEnv(nodeId);
      

      // start overlog
      System.out.println("Starting Overlog Node");
      startOverlogNode(nodeId);
      Thread.currentThread().sleep(10000); //wait for overlog to start

      
      // node_0 writes sutff
      if(nodeId.equals(new NodeId(NODE_0))){
        // write stuff
        System.out.println("Writing stuff");
        writeStuff(initialValue);
        
        // tell overlog done writing
        System.out.println("Tell overlog done");
        String[] strA = { "doneWriting", 
                          NodeIdMap.getOverlogId(nodeId).toString()};
           
        overlogPolicy.insertTuple(new Tuple(strA));
      }
      else {
      // node_1 waits to read stuff
        readStuff();
      }

      BarrierClient c =  new BarrierClient("localhost", 
                                           BARRIER_PORT,
                                           (int)nodeId.getIDint());
      c.sendBarrierRequest(0,0);
      System.exit(0);

      /*tearDown();*/

    }catch(Exception e) {
      e.printStackTrace();
    }

  }
  
}
//---------------------------------------------------------------------------
/* $Log: SubscribeTest.java,v $
/* Revision 1.1  2007/09/12 19:07:55  nalini
/* upgraded to p2-0.8.2
/*
*/
//---------------------------------------------------------------------------
