//---------------------------------------------------------------------------
/*  LogOPingTest.java
 * 
 *  LogPPingTest which measures logPStyle ping test
 * 
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.io.*;

public class LogPPingTest {

  public static String olgFilename;  
  public static String test;
  public static String localHostname;
  public static String localPort;
  public static String remoteHostname;
  public static String remotePort;
  public static long numMessages;
  public static long delay;
  public static String outFile;

  private P2JavaWrapper p2Wrapper;
  private Watcher watcher;
  private Overlog overlog;
  private long startTime;
  private long endTime;

  private boolean helperStarted = false;


  public LogPPingTest(String olgFilename, String test,
		      String localHostname, String localPort,
		      String remoteHostname, String remotePort,
		      long numMessages, long delay,
		      String outFile){

    this.olgFilename = olgFilename;
    this.test = test;
    this.localHostname = localHostname;
    this.localPort = localPort;
    this.remoteHostname = remoteHostname;
    this.remotePort = remotePort;
    this.numMessages = numMessages;
    this.delay = delay;
    this.outFile = outFile;

    p2Wrapper = new P2JavaWrapper(olgFilename,
				  localHostname,
				  localPort);

    watcher = new LogPPingTestWatcher(this);
    String[] watchTuples = {"yesAlive", "localPingReply", "remotePingReply"};

    p2Wrapper.setWatchTuples(watchTuples);
    p2Wrapper.setWatcher(watcher);
  }
   
  public synchronized void startOverlog() throws Exception{
    OverlogThread t = new OverlogThread(p2Wrapper);
    t.start();
    wait(3000);
  }

  public synchronized void logPTest() {
    if(numMessages == 0) return;

    startTime = System.currentTimeMillis();
    for(int i=0; i < numMessages; i++){
      insertTuple();
      delay();
    }
    
    endTime = System.currentTimeMillis();
    printResults();
  }

  public synchronized void insertTuple() {
    String[] stringA;

    if (test.equals("local")){
      stringA = new String[] {"localPingRequest", 
			      localHostname+ ":" + localPort};
    } else {
      stringA =  new String[] {"remotePingRequest", 
			       localHostname+ ":" + localPort,
			       remoteHostname+ ":" + remotePort};
    }
    Tuple tuple = new Tuple(stringA);   
    p2Wrapper.insertTuple(tuple); 
  }

  public synchronized void delay() {
    //System.out.println("delay = " + delay);
    if(delay==0) return;
    long delayStart=System.currentTimeMillis();
    long delayEnd=delayStart + delay;
    int a;
    
    while(true) {
      try{
	
        if(System.currentTimeMillis() < delayEnd){
	  a =0; 
          for(int i=0; i < 1000; i ++){
	    // busy wait
	    a++;
	  }
        } else{
          break;
        }
      }catch(Exception e) {
      //ignore and continue
      }
    }
    //System.out.println("Delay start = " + delayStart + "  Now: " + System.currentTimeMillis());
  }

  public synchronized void printResults() {
    try{
      PrintStream out = new PrintStream(new FileOutputStream(outFile, true));
      long totalTime = (endTime-startTime);
      double timePerMsg = (double)totalTime/numMessages;

      out.println(numMessages + "\t" + delay+ "\t" + totalTime + "\t" + timePerMsg);
      out.close();
    }catch(Exception e) {
      e.printStackTrace();
    }
  }
    
  public synchronized void waitForHelper(){
 
    while(!helperStarted){
      String[] stringA = {"areYouAlive", 
                          localHostname+ ":" + localPort,
			  remoteHostname + ":" + remotePort};
      Tuple t = new Tuple(stringA);
      p2Wrapper.insertTuple(t);
      try{
        wait(2000);
      }catch(InterruptedException e) {
        // do nothing and continue
      }
    }
  }

  public synchronized void helperStarted(){
    helperStarted = true;
    notifyAll();
  }
    

  public static void main(String[] args) {
    if (args.length < 9 ) {
      System.out.println("Usage: LogPPingTest <overlogFilename>" +
        "<local/remote> <localhostname> <localport> <remotehostname>" +
        "<remotePort> <numMessages> <delay> <resultFile>");
      return;
    }
    
    String olgFilename = args[0];
    String test;
    if(args[1].equalsIgnoreCase("local")){
      test = "local";
    } else if (args[1].equalsIgnoreCase("remote")){
      test = "remote";
    } else {
      System.out.println("Could not recognize test type. Please specify local or remote");
      return;
    }
    
    String localHostname = args[2];
    String localPort = args[3];
    String remoteHostname = args[4];
    String remotePort = args[5];
    long numMessages = (new Long(args[6])).longValue();
    long delay = (new Long(args[7])).longValue();
    String outFile = args[8];

    try{
      LogPPingTest exp = new LogPPingTest(olgFilename, test,
					  localHostname, localPort,
					  remoteHostname, remotePort,
					  numMessages, delay, outFile);
      exp.startOverlog();
      if(test.equals("remote")){
	System.out.println("Waiting for helper to start");
        exp.waitForHelper();
      }

      System.out.println("Starting experiment");
      exp.logPTest();
      System.out.println("Finished experiment ... quiting");
      System.exit(0);
    } catch(Exception e) {
      e.printStackTrace();
    }
  }

  class LogPPingTestWatcher implements Watcher {
    private LogPPingTest test;

    public LogPPingTestWatcher(LogPPingTest test) {
      this.test = test;
    }

    public synchronized void receivedTuple(Tuple tp) {
      //System.out.println("Watcher: receivedTuple " + tp);
      if(tp.getItem(0).equals("yesAlive")) {
        test.helperStarted();
      }
      else {
        // don't need to do anything with the tuple
      } 
    }
  }
}
    
//---------------------------------------------------------------------------
/* $Log: LogPPingTest.java,v $
/* Revision 1.1  2007/02/20 21:14:02  nalini
/* logP test added
/*
*/
//---------------------------------------------------------------------------
