//---------------------------------------------------------------------------
/* PadreDemandReadPolicy.java
 * 
 *  estabishes inval subscription from serverId on initialization and
 *
 *  initiates demandReads on read misses
 *
 * (C) Copyright 2007 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.io.*;
import java.util.*;

public class PadreDemandReadPolicy extends Policy{

  private boolean dbg = false;

  private SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/*");
  private boolean subscriptionEstablished = false;

  private NodeId serverId;
  


  //
  // Hashtable has its own lock
  //
  private Hashtable pendingReads;
  private Hashtable demandReadsNotIssued;

  public
  PadreDemandReadPolicy(NodeId serverId_){
    this.serverId = serverId_;
    pendingReads = new Hashtable();
    demandReadsNotIssued = new Hashtable();
  }


//---------------------------------------------------------------------------
// connection management  -- we do a roundabout way of connection management
// via inval subscriptions instead of pinging the server to see if it is alive
//  if we can establish a invalsubscription to it, that means it is alive
//---------------------------------------------------------------------------
  public void start(){
    // try to add inval subscription during initialization
    runtime.addInvalSubscription(serverId, ss, false);   
  }


 public void informAddedInvalSubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "InformAddedInvalSubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
    subscriptionEstablished = true;
    issueDemandReads();
  }


  public void informAddInvalSubscriptionFailed(NodeId senderNodeId, 
                                               NodeId receiverNodeId, 
                                               SubscriptionSet ss){
    Env.dprintln(dbg, "informAddInvalSubscriptionFailed"+senderNodeId+":"+receiverNodeId+":"+ss);
    // move pendingReads to notIssuedReads
    demandReadsNotIssued.putAll(pendingReads);
    pendingReads.clear();

   // wait for 5 seconds and then try again

    subscriptionEstablished = false;
    try{
      wait(5000);
    }catch(Exception e) {}
    runtime.addInvalSubscription(serverId, ss, false);
    
  }

  public void informRemovedInvalSubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informRemovedInvalSubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
    subscriptionEstablished = false;
    // do nothing
  }


//---------------------------------------------------------------------------
//  demand read when connected to server
//---------------------------------------------------------------------------
  public void issueDemandReads() {

    Env.dprintln(dbg, "issuing saved demand reads");

    Enumeration e = demandReadsNotIssued.keys();
    
    Object prev;

    while (e.hasMoreElements()) {
      ObjInvalTarget t = (ObjInvalTarget) e.nextElement();
      AcceptStamp inval = (AcceptStamp) demandReadsNotIssued.get(t);

      prev = pendingReads.put(t, inval.getNodeId());
    //
    // If there was no outstanding read request for this object,
    // then generate one. Note that DataStore may generate
    // multiple events per miss, so this lets us filter
    // out duplicates while stile ensuring that if the demand read
    // reply that we eventually get is not good enough, we will
    // eventually re-issue the demand read.
    //
      if(prev == null){
	runtime.demandRead(serverId,t.getObjId(), t.getOffset(), t.getLength(), inval);
      }
    }

    // clear all from demandReadsNotIssued
    demandReadsNotIssued.clear();

  }

//---------------------------------------------------------------------------
//  demand read on local read invalid
//---------------------------------------------------------------------------

  public void informLocalReadInvalid(ObjId objId, long offset, long length,
				     AcceptStamp inval){
    Env.dprintln(dbg, "informLocalReadInvalid:"+objId+":"+offset+":"+length);
    Env.dprintln(dbg, "...going to demandRead it");

    Object prev;
    AcceptStamp as = new AcceptStamp(-1, new NodeId(-1));


    
    ObjInvalTarget t = new ObjInvalTarget(objId, offset, length);
    if(subscriptionEstablished){
      prev = pendingReads.put(t, inval.getNodeId());
      //
      // If there was no outstanding read request for this object,
      // then generate one. Note that DataStore may generate
      // multiple events per miss, so this lets us filter
      // out duplicates while stile ensuring that if the demand read
      // reply that we eventually get is not good enough, we will
      // eventually re-issue the demand read.
      //
      if(prev == null){
	runtime.demandRead(serverId, objId, offset, length, as);
      }
    } else {
    Env.dprintln(dbg, "sever is not accessible, so we put it in the notIssued table");
      // put in the not issued table
      demandReadsNotIssued.put(t, inval);
    }
  }


//---------------------------------------------------------------------------
// shuts down the policy layer
//---------------------------------------------------------------------------
  public void shutdown(){
  }

//---------------------------------------------------------------------------
// interface to interact with safety module
//---------------------------------------------------------------------------

  // sendMsg: called by safety to send a msg to liveness policy
  public void receiveMsg(String[] msg){
  }

//---------------------------------------------------------------------------
// information of practi events
//---------------------------------------------------------------------------

  

  public void informLocalReadImprecise(ObjId objId, long offset, long length){
    Env.dprintln(dbg, "informLocalReadImprecise:"+objId+":"+offset+":"+length);
  }

  public void informLocalWrite(ObjId objId, long offset, long length, AcceptStamp as,
                               boolean isBound, boolean isEmbargoed){
    Env.dprintln(dbg, "informLocalWrite"+objId+":"+offset+":"+length);
  }

  public void informLocalDelete(ObjId objId){
    Env.dprintln(dbg, "informLocalDelete"+objId);
  }

  public void informReceiveInval(NodeId senderNodeId, ObjId objId, long offset, long length,
                                 AcceptStamp as, boolean isBound, boolean isEmbargoed){
    Env.dprintln(dbg, "informReceiveInval"+objId+":"+offset+":"+length);
}


  public void informDemandReadSuccess(NodeId senderNodeId, ObjId objId, long offset,
				      long length,
				      AcceptStamp as) {
      Env.dprintln(dbg, "informDemandReadSuccess"+senderNodeId+":"+objId+":"+offset+":"+length 
		   + ":" + as.toString());
    ObjInvalTarget t = new ObjInvalTarget(objId, offset, length);
    pendingReads.remove(t);
    
  }

  public void informDemandReadFailedMiss(NodeId senderNodeId, ObjId objId, 
                                         long offset, long length, 
					 AcceptStamp as) {
      Env.dprintln(dbg, "informDemandReadFailedMiss"+senderNodeId+":"+objId
		   +":"+offset+":"+length + ":" + as.toString());
  }
 
  public void informDemandReadFailedMaxRetries(NodeId senderNodeId, ObjId objId, 
                                               long offset,long length,
					       AcceptStamp as) {
    Env.dprintln(dbg, "informDemandReadFailedMaxRetries"
		 +senderNodeId+":"+objId+":"+offset+":"+length + ":" + as.toString());
  }


  public void recvSyncReply(NodeId senderNodeId, AcceptStamp acceptStamp) {
    Env.dprintln(dbg, "recvSyncReply "+senderNodeId+":"+acceptStamp);
  }

//---------------------------------------------------------------------------
// Subscription add events
//---------------------------------------------------------------------------


 

  public void informAttachedInvalSubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "InformAttachedInvalSubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

  public void informAddedBodySubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informAddedBodySubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

  public void informAddedOutgoingInvalSubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informAddedOutgoingInvalSubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

  public void informAddedOutgoingBodySubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "InformAddedOutgoingBodySubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

//---------------------------------------------------------------------------
// Subscription removed events
//---------------------------------------------------------------------------

  
  public void informRemovedBodySubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informRemovedBodySbuscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }
  
  public void informRemovedOutgoingInvalSubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informRemovedOutgoingInvalSubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

  public void informRemovedOutgoingBodySubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informRemovedOutgiongBodySubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

//---------------------------------------------------------------------------
// Subscription failed
//---------------------------------------------------------------------------
 


  public void informAddBodySubscriptionFailed(NodeId senderNodeId, 
                                              NodeId receiverNodeId, 
                                              SubscriptionSet ss){
   Env.dprintln(dbg, "informAddBodySubscriptionFailed"+senderNodeId+":"+receiverNodeId+":"+ss);
  }
  
}
//---------------------------------------------------------------------------
/* $Log:
*/
//---------------------------------------------------------------------------
