//---------------------------------------------------------------------------
/* RMITestInitiator.java
 * 
 * A  3-node test.
 * -  Initiator sets up subscriptions from itself and client to server
 * -  Server makes bounded writes
 * -  Initiator and client read data
 * -  Initiator removes some subscriptions
 * -  Server makes unbounded writes
 * -  Initiator initiates demand reads
 * -  Initiator and client read data
 *
 * (C) Copyright 2004 -- See the file COPYRIGHT for additional details
 */
//---------------------------------------------------------------------------

import java.io.*;

public class RMITestInitiator {

  //---------------------------------------------------------------------------
  // Testing functions
  //---------------------------------------------------------------------------
 
  public static void main(String[] args){
    
    if(args.length < 4){
      System.out.println("Usage: RMITestInitiator <bchost> <bcport>" +
                         "<nWrites> <sizeOfWrites>");
      System.exit(-1);
    }

    String bcHost = args[0];
    int bcPort = (new Integer(args[1])).intValue();
    int nWrites = (new Integer(args[2])).intValue();
    int size = (new Integer(args[3])).intValue();

    String path = "randomTest.db";
    String configPath = "randomTest.config";
    String p2ConfigPath = "randomTest.p2.config";   


    try{ 

      NodeId initiatorId = new NodeId(0);
      NodeId clientId =  new NodeId(1);
      NodeId serverId = new NodeId(2);

      NodeId myNodeId = initiatorId;

      BarrierClient bc = new BarrierClient(bcHost,bcPort, 0);
      Policy policy = new NullPolicy();
      P2Runtime runtime = new P2Runtime(configPath, p2ConfigPath, myNodeId, true, policy, false);
      policy.setRuntime(runtime);
      runtime.start();
      AllReadBlockAllWriteUnboundInterface lInterface = 
	new AllReadBlockAllWriteUnboundInterface(runtime.getController(), runtime.getCore());
   
      RuntimeRMIServerImpl runRMIServer = new RuntimeRMIServerImpl(runtime);
      runRMIServer.start();
      RuntimeRMIClient runRMIClient = new RuntimeRMIClient(runtime);


      SubscriptionSet ss = SubscriptionSet.makeSubscriptionSet("/*");

      barrier(bc, "Waiting for everyone to start");
      
   
      runRMIClient.addInvalSubscription(serverId, clientId, ss, false);
      runRMIClient.addBodySubscription(serverId, clientId, ss);
      Thread.currentThread().sleep(500);

      System.out.println("Subscriptions at Initiator:");
      System.out.println("- inval - "+ runRMIClient.getEstablishedInvalSubscriptions(myNodeId));

      System.out.println("Subscriptions at Client:");
      System.out.println("- inval - "+ runRMIClient.getEstablishedInvalSubscriptions(clientId));
      System.out.println("- body - "+ runRMIClient.getEstablishedBodySubscriptions(clientId));

      System.out.println("Outgoing Subscriptions at Server:");
      System.out.println("- inval -"+ runRMIClient.getEstablishedOutgoingInvalSubscriptions(serverId));
      System.out.println("- body -"+ runRMIClient.getEstablishedOutgoingBodySubscriptions(serverId));

      barrier(bc, "Initiator Finished Setting up Subscriptions");

      barrier(bc, "Server Finished Bound Writes");
      
      ObjId o = new ObjId("/0");
      while(true){
	 try{
	     ImmutableBytes imb=lInterface.read(o, 0, size);
	     System.out.println(o + ":" + imb);
	     break;
	 }catch(ObjNotFoundException e){
	     //ignore
	     Thread.currentThread().sleep(50);
	 }
     }      
      barrier(bc, "Finished trying to read data");
      
      runRMIClient.removeBodySubscription(serverId, clientId, ss);
      Thread.currentThread().sleep(500);
      

      System.out.println("Subscriptions at Initiator:");
      System.out.println("- inval - "+ runRMIClient.getEstablishedInvalSubscriptions(myNodeId));

      System.out.println("Subscriptions at Client:");
      System.out.println("- inval - "+ runRMIClient.getEstablishedInvalSubscriptions(clientId));
      System.out.println("- body - "+ runRMIClient.getEstablishedBodySubscriptions(clientId));

      System.out.println("Outgoing Subscriptions at Server:");
      System.out.println("- inval -"+ runRMIClient.getEstablishedOutgoingInvalSubscriptions(serverId));
      System.out.println("- body -"+ runRMIClient.getEstablishedOutgoingBodySubscriptions(serverId));


      barrier(bc, "Initiator removed Subscriptions");

      barrier(bc, "Server finished Unbound Writes");
      
      runRMIClient.demandRead(serverId,clientId, o, 0, size, new AcceptStamp(-1, new NodeId(-1)));
      Thread.currentThread().sleep(500);

      barrier(bc, "Initiator invoking demand reads");

      barrier(bc, "Finished reading data");

      System.out.println("Bye-Bye...");

      }catch (Exception e){
      e.printStackTrace();
    }
    
  }

  static int numBarrier = 0;

  public static void barrier(BarrierClient bc, String str){
    numBarrier++;
    System.out.println("Barrier "+numBarrier+" : "+str);
    bc.sendBarrierRequest(-1, -1);
  }

}


//---------------------------------------------------------------------------
/* $Log: RMITestInitiator.java,v $
/* Revision 1.5  2007/06/05 20:49:24  nalini
/* exposed LOG|CP option for invalSubscriptions to P2Runtime and Overlog layer
/*
/* Revision 1.4  2007/03/08 21:41:17  nalini
/* total revamp of P2Runtime, update subscriptions removed, retry logic changed
/*
/* Revision 1.3  2007/03/01 06:38:24  nalini
/* fixed compilation errors
/*
/* Revision 1.2  2006/10/10 19:46:00  nalini
/* minor bug fix
/*
/* Revision 1.1  2006/06/13 03:49:19  nalini
/* RMI for P2 Runtime Implemented
/*
 */
//---------------------------------------------------------------------------
