package code;
 /** 
 **/ 
import java.util.HashMap;
import java.util.LinkedList;
import java.util.ListIterator;
import java.util.Random;
import java.util.NoSuchElementException;

public class PendingDemandList
{
    //
    // Hash table is keyed by objId and contains a list of
    // all requests targeted at the specified object
    //
    HashMap hash;
    
public PendingDemandList()
{
    hash = new HashMap(); 
}

/*
 *------------------------------------------------------------------
 *
 * insertAndRemoveRedundant --
 *
 *      If read overlaps another read, make sure that first byte of
 *      this request is issued with existing or new request. 
 *      --> If this offset is less than pending offset, then
 *      issue this request (but shorten the length). If pending offset
 *      is later than this one, shorten then length so they no longer
 *      overlap. Note that it is OK to shorten this length even if
 *      newThis + pending shorter than originalThis; we only have
 *      to guarantee to get first byte.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      Postcondition is that hashtable contains a request that
 *      spans the first byte of the specified request.
 *
 *      Return true if this request "is completely redundant"
 *      i.e., if we did not add anything new to the table. 
 *      False otherwise.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public boolean
insertAndRemoveRedundant(PendingDemandRequest pdr)
{
    LinkedList l = (LinkedList)hash.get(pdr.getObjId());
    if(l == null){
	l = new LinkedList();
	hash.put(pdr.getObjId(), l);
    }
    return insertAndRemoveRedundantList(l, pdr);
}


/*
 *------------------------------------------------------------------
 *
 * insertAndRemoveRedundantList --
 *
 *          Finish the work of insertAndRemoveRedundant on a specific
 *          objId's list.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      Return true if this request was completely redundant
 *      (and discarded rather than added). False otherwise.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private boolean
insertAndRemoveRedundantList(LinkedList l, PendingDemandRequest pdr)
{
    assert(l != null);
    assert(pdr.getLength() > 0); 
    assert(pdr.getOffset() >= 0);
    PendingDemandRequest prev = null, next = null;
    ListIterator i = l.listIterator(0);
    while(i.hasNext()){
	prev = next;
	next = (PendingDemandRequest)i.next();
	assert(next != null);
	//
	// Invariant: List is sorted
	//
	if(prev != null){
	    assert(prev.getOffset() + prev.getLength() <= next.getOffset());
	}

	//
	// Skip to where "next" is after the item to be inserted
	//
	if(next.getOffset() < pdr.getOffset()){
	    continue;
	}
	assert(next.getOffset() >= pdr.getOffset());

	//
	// At this point, prev must begin before me.
	//
	// Because we return when the work is done, 
	// we will never see the case where prev offset
	// is higher than our offset.
	//
	assert(prev == null 
	       || 
	       pdr.getOffset() > prev.getOffset());

	//
	// If we overlap with prev, chop off start of this request
	// so we no longer overlap with prev.
	//
	if(prev != null){
	    assert(prev.getOffset() < pdr.getOffset());
	    if(prev.getOffset() + prev.getLength() > pdr.getOffset()){
		long shortenBy = prev.getOffset() + prev.getLength() 
		    - pdr.getOffset();
		assert(shortenBy > 0);
		if(shortenBy >= pdr.getLength()){
		    return true; // Nothing left to insert
		}
		pdr.chopStart(shortenBy);
		assert(pdr.getLength() > 0); 
		assert(pdr.getOffset() == prev.getOffset() + prev.getLength());
	    }
	}

	//
	// At this point, we are guaranteed that everything that
	// starts earlier is to our left and that we don't
	// overlap with anything to our left.
	//
	assert(prev == null 
	       || pdr.getOffset() >= prev.getOffset() + prev.getLength());

	//
	// If we start at the same place as next, only keep
	// going if we extend beyond next
	//
	if(next.getOffset() == pdr.getOffset()){
	    if(pdr.getLength() > next.getLength()){
		pdr.chopStart(next.getLength());
		assert(pdr.getLength() > 0); 
		assert(pdr.getOffset() == next.getOffset() + next.getLength());
		continue;
	    }
	    else{
		return true; // Nothing left to insert
	    }
	}

	//
	// At this point, we are guaranteed that everything that
	// starts later is to our right.
	//
	assert(next.getOffset() > pdr.getOffset());
	assert(pdr.getLength() > 0); 

	if(pdr.getOffset() + pdr.getLength() > next.getOffset()){
	    long newLength = next.getOffset() - pdr.getOffset();
	    assert(newLength > 0); 
	    assert(newLength <= pdr.getLength());
	    pdr.setLength(newLength);
	    assert(pdr.getLength() > 0); 
	}
	
	//
	// At this point, we are guaranteed that everything that
	// starts earlier is entirely to the left and everything that
	// starts later is entirely to the right.
	//
	assert(next.getOffset() >= pdr.getOffset() + pdr.getLength());
	assert(prev == null 
	       || pdr.getOffset() >= prev.getOffset() + prev.getLength());
	
	l.add(i.previousIndex(), pdr);
	return false; // Added something new
    }

    //
    // We are the highest offset item and (by virtue of the pass
    // through the list above) do not overlap with anything on the list.
    //
    assert(i.hasNext() == false); // reached end of list
    next = null;
    prev = null;
    PendingDemandRequest last;
    try{
	last = (PendingDemandRequest)l.getLast();
    }
    catch(NoSuchElementException e){
	last = null;
    }
    if(last != null){
	if(last.getOffset() + last.getLength() > pdr.getOffset()){
	    long shortenBy = last.getOffset() + last.getLength() 
		- pdr.getOffset();
	    assert(shortenBy > 0);
	    if(shortenBy >= pdr.getLength()){
		return true; // Nothing left to insert
	    }
	    pdr.chopStart(shortenBy);
	    assert(pdr.getOffset() == last.getOffset() + last.getLength());
	}
    }
    assert(last == null
	   || pdr.getOffset() >= last.getOffset() + last.getLength());
    l.addLast(pdr);
    return false;
}



/*
 *------------------------------------------------------------------
 *
 * remove --
 *
 *          Remove any pending request whose first byte
 *          overlaps the BodyMsg. 
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      Postcondition is that there is no longer a pending
 *      request whose first byte is covered by bodyMsg.
 *      (Note that we may cancel multiple requests and 
 *      that not all of them will be fully satisfied. But
 *      every request we cancel will have had its first
 *      byte satisfied.)
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public void
remove(BodyMsg msg)
{
    ObjInvalTarget oit = msg.getObjInvalTarget();
    ObjId o = oit.getObjId();
    long offset = oit.getOffset();
    long length = oit.getLength();
    boolean match;
    do{
	match = removeFirst(o, offset, length);
    }while(match);
}

/*
 *------------------------------------------------------------------
 *
 * remove --
 *
 *          Remove any pending request whose first byte
 *          overlaps the specified request. 
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      Postcondition is that there is no longer a pending
 *      request whose first byte is covered by request.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public void
remove(PendingDemandRequest pdr)
{
    ObjId o = pdr.getObjId();
    long offset = pdr.getOffset();
    long length = pdr.getLength();
    boolean match;
    do{
	match = removeFirst(o, offset, length);
    }while(match);
}



/*
 *------------------------------------------------------------------
 *
 * removeFirst --
 *
 *          Remove the first pending request whose starting
 *          point is overlapped by the specified range.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      Return true if something removed and false if
 *      nothing removed.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private boolean
removeFirst(ObjId objId, long offset, long length)
{
    LinkedList l = (LinkedList)hash.get(objId);
    if(l == null){
	return false;
    }
    ListIterator i = l.listIterator();
    while(i.hasNext()){
	PendingDemandRequest current = (PendingDemandRequest)i.next();
	assert(current != null);
	assert(current.getObjId().equals(objId));
	if(current.getOffset() + current.getLength() < offset){
	    continue;
	}
	assert(current.getOffset() + current.getLength() >= offset);
	if(current.getOffset() >= offset + length){
	    return false;
	}
	assert(current.getOffset() < offset + length
	       && current.getOffset() + current.getLength() >= offset);

	try{
	    i.remove();
	}
	catch(UnsupportedOperationException e){
	    assert(false);
	}
	catch(IllegalStateException f){
	    assert(false);
	}
	return true;
    }
    return false;
}


/*
 *------------------------------------------------------------------
 *
 * isPresent --
 *
 *          Return true if there exists a pending request that overlaps 
 *          the first byte of the specified request.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public boolean
isPresent(PendingDemandRequest r)
{
    PendingDemandRequest g = find(r.getObjId(), r.getOffset());
    if(g == null){
	return false;
    }
    return true;
}


/*
 *------------------------------------------------------------------
 *
 * find --
 *
 *          Find a request that overlaps the first byte of
 *          the specified objId, offset pair
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Private b/c we return a pointer to internal state.
 *
 *------------------------------------------------------------------
 */
private PendingDemandRequest 
find(ObjId oid, long offset)
{
    LinkedList l = (LinkedList)hash.get(oid);
    if(l == null){
	return null;
    }
    int index = findIndex(l, oid, offset);
    if(index < 0){
	return null;
    }
    try{
	return (PendingDemandRequest)l.get(index);
    }
    catch(IndexOutOfBoundsException e){
	assert(false);
	return null;
    }
}

/*
 *------------------------------------------------------------------
 *
 * find --
 *
 *          Find index of a request that overlaps the first byte of
 *          the specified objId, offset pair
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private int
findIndex(LinkedList l, ObjId oid, long offset)
{
    ListIterator i = l.listIterator();
    while(i.hasNext()){
	PendingDemandRequest current = (PendingDemandRequest)i.next();
	if(current.getOffset() <= offset 
	   && current.getOffset() + current.getLength() >= offset){
	    return i.previousIndex();
	}
    }
    return -1;
}


public boolean
equals(Object o)
{
    assert(false); // Equals undefined on this object
    return false;
}
public int
hashCode()
{
    assert(false); // Hash undefined on this object
    return -1;
}


public static void 
main(String s[])
{
    System.err.print("PendingDemandList self test...");
    unitStress();
    System.err.println("PendingDemandList self test PASSES");
    System.exit(0);
}

/*
 *------------------------------------------------------------------
 *
 * unitStress --
 *
 *          Unit testing -- stress test insert overlapping
 *          longic with lots of random offsets and lenghts.
 *          Trying to trigger an assertion failure in the
 *          insert code.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static void 
unitStress()
{
    PendingDemandList pdl = new PendingDemandList();
    Random r = new Random(119);
    ObjId o = new ObjId("/foo");
    long rid = 0;
    long maxOffset = 10000;
    long maxLength = 100;
    int ii;
    long addCount = 0;
    long removeCount = 0;
    for(ii = 0; ii < 10000; ii++){
	long offset = r.nextLong() % maxOffset;
	if(offset < 0){
	    offset = offset * -1;
	}
	assert(offset >= 0);
	long length = r.nextLong() % maxLength;
	if(length == 0){
	    length += 1;
	}
	if(length < 0){
	    length = length * -1;
	}
	assert(length > 0);
	PendingDemandRequest pdr = new PendingDemandRequest(o, offset, 
							    length, rid++,
							    new AcceptStamp(-1, new NodeId(0)));
	//
	// Check for assertion failures in complex merge logic.
	//
	pdl.insertAndRemoveRedundant(pdr);

	//
	// Check postcondition: there exists a request covering
	// the first byte of the inserted request.
	//
	pdr = pdl.find(o, offset);
	assert(pdr != null);
    }
    for(ii = 0; ii < 10000; ii++){
	long offset = r.nextLong() % maxOffset;
	if(offset < 0){
	    offset = offset * -1;
	}
	long length = r.nextLong() % maxLength;
	if(length == 0){
	    length += 1;
	}
	if(length < 0){
	    length = length * -1;
	}
	PendingDemandRequest pdr = new PendingDemandRequest(o, offset, 
							    length, rid++,
							    new AcceptStamp(-1, new NodeId(0)));
	//
	// Make 1/4 requests insert, the rest removes
	//
	long decide = r.nextLong() % 4;
	if(decide == 0){
	    pdl.insertAndRemoveRedundant(pdr);
	    addCount++;
	    continue;
	}
	pdl.remove(pdr);
	removeCount++;

	//
	// Postcondition of remove is that nothing whose first
	// byte is overlapped by pdr remains on the list
	//
	assert(pdl.removeFirst(o, offset, length) == false);
	long jj;
	for(jj = 0; jj < length; jj++){
	    PendingDemandRequest check = pdl.find(o, offset + jj);
	    if(check != null){
		//
		// pdl overlaps with something on list but
		// the first byte of request on list doesn't
		// overlap pdl, so pdl's "arrival" doesn't
		// count as satisfying check.
		//
		if(check.getOffset() >= pdr.getOffset()){
		    assert(check.getOffset() < pdr.getOffset());
		}
		if(check.getOffset() + check.getLength() <= pdr.getOffset()){
		    assert(check.getOffset() + check.getLength() > pdr.getOffset());
		}
	    }
	}
    }
    assert(addCount > 0);
    assert(removeCount > 0);
    assert(removeCount > addCount);

}

}


//---------------------------------------------------------------------------
/* $Log: PendingDemandList.java,v $
/* Revision 1.4  2007/02/01 06:12:09  zjiandan
/* Add acceptStamp to demandRead so that the sender only sends the data
/* that's at least as new as the acceptStamp.
/*
/* Revision 1.3  2004/05/10 21:58:57  dahlin
/* Fixed ^M formatting
/*
/* Revision 1.2  2004/05/10 18:56:31  dahlin
/* Created PendingDemandList and PendingDemandRequest (needed by SDIMSController)
/*
 */
//---------------------------------------------------------------------------
