package code;
//---------------------------------------------------------------
/* PersistentLog
 * 
 * Provide stable on-disk invalidate sequences
 * Use BerkeleyDB to manage data storage
 *
 */
//---------------------------------------------------------------
import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.DatabaseNotFoundException;
import com.sleepycat.je.DeadlockException;

import com.sleepycat.je.Environment;
import com.sleepycat.je.EnvironmentConfig;
import com.sleepycat.je.EnvironmentMutableConfig;
import com.sleepycat.bind.serial.StoredClassCatalog;
import com.sleepycat.je.DatabaseEntry;
import com.sleepycat.bind.serial.SerialBinding;
import com.sleepycat.bind.tuple.LongBinding;
import com.sleepycat.bind.tuple.TupleBinding;
import com.sleepycat.je.Database;
import com.sleepycat.je.DatabaseConfig;
import com.sleepycat.je.Transaction;
import com.sleepycat.je.Cursor;
import com.sleepycat.je.OperationStatus;

import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.BufferedReader;
import java.util.StringTokenizer;
public class PersistentLog{
  private Environment env;
  
  private Database statusDb;//stores omitVV, last gabage collect time etc.

  private Database invalDb; //stores invalidate & unbindmsg sequences
                            //the key is automatically created sequence number
                            //which is unique and incremental
                            //Note: assume the number can go infinity
                            //      we do not deal with number overflow

  private Database classDb; //stores Class Information for Data Binding
  
  private long nextSeq;//sequence number generator for invalDB Record
  private CounterVV omitVV;

  final private static String omitKeyName = "OMITVV";

  final private static String statusDbName = "statusDatabase";
  final private static String invalDbName = "invalidateDatabase";
  final private static String classDbName = "StoredClassCatalog";
  final private static boolean verbose = false;

  // the max diviation the totDiskCharge from actual disk space occupied
  // set this value large enough so that the actual disk space will never
  // grow beyond the specified max disk size for log
  private final static long MAX_LOG_DISK_SIZE_ESTIMATE_DIVIATION = 0;
  private final static long ENTRY_OVERHEAD = 135;
  //private final static double DEFAULT_PAGE_FILL_FACTOR = 0.80;
  private DatabaseEntry omitKey;
  
  
  private TupleBinding logRecordBinding;
  private SerialBinding vvBinding;

  private StoredClassCatalog classCatalog;

  private long totDiskCharge;// keep track of the total data size in log
                             // it might not be accurate as the actual disk
                             // size.

  //-----------------------------------------------------------
  // constructor
  // open/create databases
  // for omitVV, if it's new-created, put the initial VV
  //----------------------------------------------------------- 
  public 
  PersistentLog(String path,
                long cacheSizeBytes,
                boolean reset,
                boolean noSync)
    throws IOException, SecurityException{

    Env.tbd("PersistentLog: estimate the disk size accurately");
    if ( verbose ){
      Env.dprintln(verbose, "PersistentLog::PersistentLog(path="
                 + path
                 + " reset="
                 + reset
                 + ")");
    }
    
    //
    // setup configuration info for env and dbs
    //
    EnvironmentConfig envConfig = new EnvironmentConfig();
    envConfig.setTransactional(true);
    envConfig.setAllowCreate(true);
    envConfig.setCacheSize(cacheSizeBytes);
    EnvironmentMutableConfig mutableConfig = new EnvironmentMutableConfig();

    //
    // For a long time we said here
    //      mutableConfig.setTxnNoSync(true);
    // Unfortunately, this doesn't work -- we rely on PersistentLog as
    // a redo log and we assume that once something is in our local
    // UpdateLog that it is persistent. Unfortunately, commenting 
    // this out slows us down a lot (see mike's notes/2006/8/8b.txt).
    //
    mutableConfig.setTxnNoSync(noSync);

    // We print out a little message here until we figure out how to
    // efficiently sync the log
    System.out.print("DBG: MDD -- PersistentLog -- txnNoSync ");
    if(noSync){
      System.out.println("turned on; recovery won't work!");
    }else{
      System.out.println("turned off; recovery should work...");
    }
    //
    //

    
    DatabaseConfig dbConfig = new DatabaseConfig();
    dbConfig.setTransactional(true);
    dbConfig.setAllowCreate(true);
	
    Transaction txn = null;
    File dbDir = new File(path);
    Cursor cursor = null;
    
    //
    // Initialize environment and databases
    //
    try{
      if(dbDir.exists()&&reset){
        try{
          assert dbDir.isDirectory();
        
          File[] dbFiles = dbDir.listFiles();
          for(int i=0; i<dbFiles.length; i++){
            assert dbFiles[i].isFile();
            dbFiles[i].delete();
          }
        }catch (SecurityException se){
          throw new IOException(se.toString());
        }
      }
      dbDir.mkdirs();
      env = new Environment(dbDir, 
                            envConfig);
      env.setMutableConfig(mutableConfig);

      txn = env.beginTransaction(null, null);
	    
      //
      // open/create classDB
      //
      classDb = env.openDatabase(txn, classDbName, dbConfig);
      classCatalog = new StoredClassCatalog(classDb);

      //initialize binding stuff
      //logRecordBinding = new SerialBinding(classCatalog, LogRecord.class);
      logRecordBinding = new LogRecordBinding();
      vvBinding = new SerialBinding(classCatalog, VV.class);

      //
      // open/create statusDB
      // and initialize omitVV if necessary
      //
      statusDb = env.openDatabase(txn, 
                                  statusDbName,
                                  dbConfig);

      DatabaseEntry key = new DatabaseEntry();
      DatabaseEntry data = new DatabaseEntry(); 

      omitKey = new DatabaseEntry(omitKeyName.getBytes("UTF-8"));
      
      OperationStatus status = statusDb.get(txn, 
                                            omitKey, 
                                            data, 
                                            null );
            
      if ((status != OperationStatus.SUCCESS)||
          reset){//just created or reset
        omitVV = (CounterVV)CounterVV.makeVVAllNegatives();
        vvBinding.objectToEntry(omitVV, data);
        statusDb.put(txn, omitKey, data);
      }else{
        assert(status == OperationStatus.SUCCESS);
        omitVV = (CounterVV)vvBinding.entryToObject(data);
	if ( verbose ){
	  Env.dprintln(verbose, "omitVV already exist: " + omitVV); 
	}
        
      }
  
      //
      // open/create InvalDB
      // and initialize nextSeq
      //
      
      if (reset){
        
        try{
          env.truncateDatabase(txn, invalDbName, false);
          env.compress();
          env.cleanLog();
        }catch(DatabaseNotFoundException dnfe){
          //do nothing.
        }
        
        invalDb = env.openDatabase(txn,
                                 invalDbName,
                                 dbConfig);
        nextSeq = 0;
      }else{
        invalDb = env.openDatabase(txn,
                                 invalDbName,
                                 dbConfig);
        cursor = invalDb.openCursor(txn, null);
	   
        if(cursor.getLast(key, data, null) == OperationStatus.SUCCESS){
          //recover nextSeq
          nextSeq = LongBinding.entryToLong(key)+1;
        } 
        else {
          //
          // Note:
          //      Here we assume that Garbage Collection Thread never deletes
          //      the last invalidate. Make a self test to make sure it works
          //      that way.
          //
          nextSeq = 0;
        }
        cursor.close();
        cursor = null;
      }
      txn.commit();
      txn = null;

    } 
    catch(DatabaseException dbe){
      dbe.printStackTrace();
      String msg = dbe.toString();
      if (cursor != null){
        try{
          cursor.close();
        }catch(Exception ae){
          msg = msg + 
            ("\nNested exception closing cursor:" +  ae.toString());
        } 
        cursor = null;
      }
      if(txn != null){
        try{
          txn.abort();
        }
        catch(Exception ae){
          msg = msg + 
            ("\nNested exception aborting:" +  ae.toString());
        } 
      }
      throw new IOException("Exception opening berkeleyDB Environment: " 
                            + msg);
    }
    
    totDiskCharge = 0;
    try{
      assert dbDir.isDirectory();
      
      File[] dbFiles = dbDir.listFiles();
      for(int i=0; i<dbFiles.length; i++){
        assert dbFiles[i].isFile():dbFiles[i];
        totDiskCharge += dbFiles[i].length(); 
      }
    }catch (SecurityException se){
      throw new IOException(se.toString());
    }
    
  }
    
  //-----------------------------------------------------------
  // get the OmitVV
  //-----------------------------------------------------------
  public AcceptVV 
  getOmitVV(){
    return omitVV.cloneAcceptVV();
  }
    
  //----------------------------------------------------------
  // -- deprecated
  // for garbage collection
  // delete invalidates/unbindmsg till reach either the max
  // number or beyond any part of the given VV.
  // note: this method doesn't clean all the inval less than VV
  //       since it stops as soon as it reaches the first inval beyond
  //       the boundVV. it's possible that there're invs less than 
  //       boundVV after this point.
  // the deleted invalidate.getEndVV() should be <= boundVV
  // this given VV shouldn't be "larger" than the currentVV of 
  // the last checkpoint
  // return the number of inval truncated
  //----------------------------------------------------------
  public synchronized long 
  truncateInval(long invalNum, VV boundVV)
  throws IOException{
    long count = 0;
    
    Transaction txn = null;
    
    DatabaseEntry foundKey = new DatabaseEntry();
    DatabaseEntry foundData = new DatabaseEntry();
    
    Cursor cursor = null;
    CounterVV newOmitVV = new CounterVV();
    newOmitVV.advanceTimestamps(omitVV);
    try{
      txn = env.beginTransaction(null, null);
      cursor = invalDb.openCursor(txn, null);

      OperationStatus status;
   
      //
      // make sure that we don't garbage collect the last one
      //
      long lastkey = -1;
      status = cursor.getLast(foundKey, foundData, null);
      if (status == OperationStatus.SUCCESS){
        lastkey =  LongBinding.entryToLong(foundKey);
      }
      
      status = cursor.getFirst(foundKey, foundData, null);
      while((status == OperationStatus.SUCCESS) 
            &&
            (count < invalNum)){
        
        //exit the loop if it's already reached the last one
        if(LongBinding.entryToLong(foundKey) == lastkey){
          break;
        }
        assert (LongBinding.entryToLong(foundKey) < lastkey);
        
        //LogRecord logRecord = (LogRecord)
        Object foundObject = logRecordBinding.entryToObject(foundData);
        
        //
        // if the current inv.endVV reach the boundVV
        // exit the loop otherwise advance the omitVV
        //
        if (foundObject instanceof GeneralInv) {
          GeneralInv inv = (GeneralInv)foundObject;
          
          if ((boundVV!=null)&&
              (inv.getEndVV().isAnyPartGreaterThan(boundVV))){
            status = cursor.getNext(foundKey, foundData, null);
            continue;
          }
          newOmitVV.advanceTimestamps(inv.getEndVV());
        }
        //
        //overhead for each item is 4 bytes
        // according to http://www.lnupress.com.cn/ref/program/diskspace.html
        //
        this.totDiskCharge -= (foundKey.getSize() + foundData.getSize() 
                               + ENTRY_OVERHEAD);
        cursor.delete();
        count++;
        
        status = cursor.getNext(foundKey, foundData, null);
      }
      cursor.close();
      vvBinding.objectToEntry(newOmitVV, foundData);
      statusDb.put(txn, 
                   omitKey, 
                   foundData);
     
    
      txn.commit();
      txn = null;
      env.compress();
      env.cleanLog();
    } catch (DatabaseException de){
      de.printStackTrace();
      String msg = de.toString();
      if(cursor != null){
        try{
          cursor.close();
        }catch (DatabaseException de2){
          msg = msg + "nested exception when close cursor: " +de2.toString();
        }
          cursor = null;
      }
      try{
        txn.abort();
      }catch(DatabaseException de3){
        msg = msg + "nested exception when aborting: " +de3.toString();
      }
      txn = null;
      throw new IOException("Exception trucating persistentLog: " + msg);
    }
    omitVV.advanceTimestamps(newOmitVV);//now it's safe to update the omitVV
    return count;
  }

  //----------------------------------------------------------
  // for garbage collection
  // delete all invalidates strictly smaller than boundVV and
  // all the unbindmsg. 
  // the deleted invalidate.getEndVV() should be <= boundVV
  // this given boundVV shouldn't be "larger" than the currentVV of 
  // the last checkpoint.
  //
  // Note: we do not cut the invals to the desired omitvv
  //       because it's expensive to rewrite the entries while it doesn't
  //       save any spaces as we still have the same number of invals in
  //       the log. We just update the omitvv in the statusDB. 
  //       when the UpdateLog recovers in-mem-vv from persistentlog,
  //       it will cut the invals to the omitVV.
  //
  // side Effect update OmitVV
  // return the new OmitVV
  //----------------------------------------------------------
  public AcceptVV 
  truncateCompletely(VV newBoundVV)
  throws IOException{
    CounterVV boundVV = new CounterVV(newBoundVV);
    boundVV.advanceTimestamps(omitVV);// we want to get the max of boundVV and
                                      // omitVV as the boundVV since some parts of
                                      // the boundVV might be very small.
    assert boundVV!=null;
   
    /*
    if (omitVV.includes(boundVV)){//nothing needs to be truncated.
      Env.inform("Garbage Collect call with dummy boundVV");
      return omitVV;
    }
    */
    Transaction txn = null;
    
    DatabaseEntry foundKey = new DatabaseEntry();
    DatabaseEntry foundData = new DatabaseEntry();
    
    Cursor cursor = null;
    CounterVV newOmitVV = new CounterVV();
    newOmitVV.advanceTimestamps(omitVV);
    newOmitVV.advanceTimestamps(newBoundVV);
    try{
      txn = env.beginTransaction(null, null);
      cursor = invalDb.openCursor(txn, null);

      OperationStatus status;
   
      //
      // make sure that we don't garbage collect the last one
      //
      long lastkey = -1;
      status = cursor.getLast(foundKey, foundData, null);
      if (status == OperationStatus.SUCCESS){
        lastkey =  LongBinding.entryToLong(foundKey);
      }
      
      status = cursor.getFirst(foundKey, foundData, null);
      while(status == OperationStatus.SUCCESS){
        
        //exit the loop if it's already reached the last one
        if(LongBinding.entryToLong(foundKey) == lastkey){
          break;
        }
        assert (LongBinding.entryToLong(foundKey) < lastkey);
        
        //LogRecord logRecord = (LogRecord)
        Object foundObject = logRecordBinding.entryToObject(foundData);
        
        //
        // if the current inv.endVV reach the boundVV
        // skip this loop otherwise delete it and advance the omitVV
        //
        if (foundObject instanceof GeneralInv) {
          GeneralInv inv = (GeneralInv)foundObject;
          
          if(inv.getEndVV().isAnyPartGreaterThan(boundVV)){
            status = cursor.getNext(foundKey, foundData, null);
            continue;
          }
          
        }
        this.totDiskCharge -= (foundKey.getSize() + foundData.getSize()
                               + ENTRY_OVERHEAD);
        
        cursor.delete();//can be inv <= boundVV or unbindmsg
        
        status = cursor.getNext(foundKey, foundData, null);
      }
      cursor.close();
      vvBinding.objectToEntry(newOmitVV, foundData);
      statusDb.put(txn, 
                   omitKey, 
                   foundData);

      txn.commit();
      txn = null;
      env.compress();
      env.cleanLog();
    } catch (DatabaseException de){
      de.printStackTrace();
      String msg = de.toString();
      if(cursor != null){
        try{
          cursor.close();
        }catch (DatabaseException de2){
          msg = msg + "nested exception when close cursor: " +de2.toString();
        }
          cursor = null;
      }
      try{
        txn.abort();
      }catch(DatabaseException de3){
        msg = msg + "nested exception when aborting: " +de3.toString();
      }
      txn = null;
      throw new IOException("Exception trucating persistentLog: " + msg);
    }
    omitVV.advanceTimestamps(newOmitVV);//now it's safe to update the omitVV
    return omitVV.cloneAcceptVV();
  }

  //----------------------------------------------------------
  // Get an Iterator which provides methods to scan invalDb
  //
  // Note: the desctructor of LogIterator should close the
  //       cursor it used.
  //----------------------------------------------------------
  public LogIterator 
  getLogIterator()
    throws DatabaseException{
    return new LogIterator(invalDb.openCursor(null, null), 
                           logRecordBinding);    
  } 

  //----------------------------------------------------------
  // Append the invalidate into the database
  // 
  // Note: here we use autocommit instead of explicit transaction 
  //       commit to make the append permanent.
  //----------------------------------------------------------
  public void
  append(GeneralInv inv)
    throws DatabaseException{

    if(omitVV.includes(inv.getEndVV())){
      return;//ignore if already omitted
    }
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();

    LongBinding.longToEntry(nextSeq, key);
    nextSeq++;
    logRecordBinding.objectToEntry(inv,
                                   data);
    
    OperationStatus status = invalDb.putNoOverwrite(null, 
                                                    key, 
                                                    data);
    
    assert status!=OperationStatus.KEYEXIST;
    this.totDiskCharge += (key.getSize() + data.getSize()  + ENTRY_OVERHEAD);
    
  }
  //----------------------------------------------------------
  // append the UnbindMsg into the database
  //----------------------------------------------------------
  public void
  append(UnbindMsg ubm)
    throws DatabaseException{
	
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    
    LongBinding.longToEntry(nextSeq, key);
    nextSeq++;
    logRecordBinding.objectToEntry(ubm,
                                   data);
    
    OperationStatus status = invalDb.putNoOverwrite(null, 
                                                    key, 
                                                    data);
    assert status!=OperationStatus.KEYEXIST; 
    this.totDiskCharge += (key.getSize() + data.getSize() 
                           + ENTRY_OVERHEAD); 
  }
  
  //----------------------------------------------------------
  // append the UnbindMsg into the database
  //----------------------------------------------------------
  public void
  append(DebargoMsg ubm)
    throws DatabaseException{
	
    DatabaseEntry key = new DatabaseEntry();
    DatabaseEntry data = new DatabaseEntry();
    
    LongBinding.longToEntry(nextSeq, key);
    nextSeq++;
    logRecordBinding.objectToEntry(ubm,
                                   data);
    
    OperationStatus status = invalDb.putNoOverwrite(null, 
                                                    key, 
                                                    data);
    assert status!=OperationStatus.KEYEXIST; 
    this.totDiskCharge += (key.getSize() + data.getSize() + ENTRY_OVERHEAD);
    
  }
  //-----------------------------------------------------------------------
  // sync log contents to disk
  //-----------------------------------------------------------------------
  public void
  sync()
    throws IOException{
    
    try{
      env.sync();
    }
    catch(DatabaseException e){
      e.printStackTrace();
      throw new IOException(e.toString());
    }
  }

  //------------------------------------------------------------------------
  // return the estimated disk size in a cheap way
  // use accurateDiskSize() if need very accurate disk size
  //------------------------------------------------------------------------
  public long
  diskSize(){
    return this.totDiskCharge+MAX_LOG_DISK_SIZE_ESTIMATE_DIVIATION;
  }

  //------------------------------------------------------------------------
  // return the accurate disk size 
  // -- this operation is more expensive comparing to the diskSize()
  //------------------------------------------------------------------------
  public long
  accurateDiskSize()
    throws IOException{
    Env.performanceWarning("this operation is expensive, use diskSize instead"+
                        " if the accurancy of the disk size is not critical");
    long ret = 0;
    try{
      File dbDir = env.getHome();
      
      assert dbDir.isDirectory();
    
      File[] dbFiles = dbDir.listFiles();
      for(int i=0; i<dbFiles.length; i++){
        assert dbFiles[i].isFile();
        ret += dbFiles[i].length(); 
      }
    }catch (DatabaseException de){
      throw new IOException(de.toString());
    }catch (SecurityException se){
      throw new IOException(se.toString());
    }
    
    return ret;
  }
  
  //--------------------------------------------------------
  // close() -- flush everything to disk and close.
  //--------------------------------------------------------
  public void
  close(){
    try{     
      statusDb.close();
      invalDb.close();
      classDb.close();
      env.sync();
      env.close();
      env = null;
    }
    catch(DatabaseException dbe){
      Env.warn("Exception closing database in PersistentLog: " + 
               dbe.toString());
      return; // Ignore exception
    }
  }

  //===========================================================
  //--------------------Test Methods---------------------------
  //===========================================================

  //---------------------------------------------------------
  // Self Test 1 -- simple test of every API
  //---------------------------------------------------------
  public void 
  simpleTest()
    throws Exception{

    AcceptStamp[] as0 = new AcceptStamp[4];
    as0[0] = new AcceptStamp(-1, new NodeId(9));
    as0[1] = new AcceptStamp(-1, new NodeId(10));
    as0[2] = new AcceptStamp(-1, new NodeId(11));
    as0[3] = new AcceptStamp(-1, new NodeId(12));
    
    CounterVV resultVV = new CounterVV(as0);
    assert ( resultVV.equals(getOmitVV()));

    assert (truncateInval(10, null) == 0);

    LogIterator logIterator = getLogIterator();
    Object nextObj = logIterator.getNext();
    while (nextObj != null){
      assert false;//should be none
    }
    logIterator.close();
    
    AcceptStamp[] as1 = new AcceptStamp[2];
    as1[0] = new AcceptStamp(1, new NodeId(9));
    as1[1] = new AcceptStamp(1, new NodeId(10));
    
    AcceptStamp[] as2 = new AcceptStamp[2];
    as2[0] = new AcceptStamp(2, new NodeId(9));
    as2[1] = new AcceptStamp(2, new NodeId(10));
    
    String[] dir = new String[2];
    dir[0] = "/d1/*";
    dir[1] = "/d2/*";
    
    ImpreciseInv ii = new ImpreciseInv(HierInvalTarget.makeHierInvalTarget(dir),
                                       new AcceptVV(as1),
                                       new AcceptVV(as2));
    //System.err.println("before append, nextSeq = " +nextSeq);
    assert (nextSeq == 0);
    
    append(ii);
    //System.err.println("after first append, nextSeq = " + nextSeq);
    assert (nextSeq == 1);
    as1[0] = new AcceptStamp(3, new NodeId(9));
    as1[1] = new AcceptStamp(3, new NodeId(11));

    as2[0] = new AcceptStamp(4, new NodeId(9));
    as2[1] = new AcceptStamp(4, new NodeId(11));
    ii = new ImpreciseInv(HierInvalTarget.makeHierInvalTarget(dir),
                          new AcceptVV(as1),
                          new AcceptVV(as2));
    append(ii);
    //System.err.println("after second append, nextSeq = " + nextSeq);
    assert (nextSeq == 2);
    UnbindMsg um = new UnbindMsg(new ObjInvalTarget(new ObjId("obj1"), 0, 0), 
                                 new AcceptStamp(100, new NodeId(9)));
    append(um);
    //System.err.println("after third append, nextSeq = " + nextSeq);
    assert (nextSeq == 3);
    as1[0] = new AcceptStamp(5, new NodeId(9));
    as1[1] = new AcceptStamp(5, new NodeId(10));

    as2[0] = new AcceptStamp(6, new NodeId(9));
    as2[1] = new AcceptStamp(6, new NodeId(10));
    ii = new ImpreciseInv(HierInvalTarget.makeHierInvalTarget(dir),
                          new AcceptVV(as1),
                          new AcceptVV(as2));
    append(ii);
    //System.err.println("after fourth append, nextSeq = " + nextSeq);
    assert (nextSeq == 4);

    logIterator = getLogIterator();
    nextObj = logIterator.getNext();
    int count = 0;
    while (nextObj != null){
      //System.err.println(nextObj); //should be 4 records
      count++;
      nextObj = logIterator.getNext();
    }
    assert (count == 4);
    logIterator.close();
    
    AcceptStamp[] as3 = new AcceptStamp[3];
    as3[0] = new AcceptStamp(2, new NodeId(9));
    as3[1] = new AcceptStamp(5, new NodeId(11));
    as3[2] = new AcceptStamp(1, new NodeId(12));

    assert (truncateInval(1, new AcceptVV(as3)) == 1);

    as0[0] = new AcceptStamp(2, new NodeId(9));
    as0[1] = new AcceptStamp(2, new NodeId(10));
    as0[2] = new AcceptStamp(-1, new NodeId(11));
    as0[3] = new AcceptStamp(-1, new NodeId(12));
    resultVV = null;
    resultVV = new CounterVV(as0);
    assert ( resultVV.equals(getOmitVV()));

    logIterator = getLogIterator();
    nextObj = logIterator.getNext();
    count = 0;
    while (nextObj != null){
      //System.err.println(nextObj); 
      count++;
      nextObj = logIterator.getNext();
    }
    logIterator.close();

    assert (count == 3);

    as3[0] = new AcceptStamp(6, new NodeId(9));
    as3[1] = new AcceptStamp(2, new NodeId(10));
    as3[2] = new AcceptStamp(4, new NodeId(11));

    assert (truncateInval(1, new AcceptVV(as3)) == 1);


    as0[0] = new AcceptStamp(4, new NodeId(9));
    as0[1] = new AcceptStamp(2, new NodeId(10));
    as0[2] = new AcceptStamp(4, new NodeId(11));
    as0[3] = new AcceptStamp(-1, new NodeId(12));
    resultVV = null;
    resultVV = new CounterVV(as0);
    assert ( resultVV.equals(getOmitVV()));

    logIterator = getLogIterator();
    nextObj = logIterator.getNext();
    count = 0;
    while (nextObj != null){
      //System.err.println(nextObj); //should be 2 record
      count++;
      nextObj = logIterator.getNext();
    }
    logIterator.close();
    assert (count == 2);
    
    as2[0] = new AcceptStamp(7, new NodeId(9));
    as2[1] = new AcceptStamp(7, new NodeId(10));
 
    assert (truncateInval(3, new AcceptVV(as2)) == 1);
    //System.err.println("omitVV after truncate: " + getOmitVV());
    as0[0] = new AcceptStamp(4, new NodeId(9));
    as0[1] = new AcceptStamp(2, new NodeId(10));
    as0[2] = new AcceptStamp(4, new NodeId(11));
    as0[3] = new AcceptStamp(-1, new NodeId(12));
    resultVV = null;
    resultVV = new CounterVV(as0);
    assert ( resultVV.equals(getOmitVV()));
    
    logIterator = getLogIterator();
    nextObj = logIterator.getNext();
    count = 0;
    while (nextObj != null){
      //System.err.println(nextObj); //should be 1 record
      count++;
      nextObj = logIterator.getNext();
    }
    logIterator.close();
    assert (count == 1);
    
    System.out.println("diskSize: " + this.diskSize());
    System.out.println("accurateDiskSize: " + this.accurateDiskSize()); 
  }

  //-----------------------------------------------------------------------
  // Self Test 2 -- Test Recover
  // Note: this test assumes it follows the above simpletest immediately.
  //-----------------------------------------------------------------------
  public void
  testRecover()
    throws Exception{
    //System.err.println("nextSeq = " + nextSeq); //should be 4
    assert (nextSeq == 4);
    //System.err.println("omitVV: " + getOmitVV());
    AcceptStamp[] as0 = new AcceptStamp[4];
    as0[0] = new AcceptStamp(4, new NodeId(9));
    as0[1] = new AcceptStamp(2, new NodeId(10));
    as0[2] = new AcceptStamp(4, new NodeId(11));
    as0[3] = new AcceptStamp(-1, new NodeId(12));
    CounterVV resultVV = null;
    resultVV = new CounterVV(as0);
    assert ( resultVV.equals(getOmitVV()));
    
    LogIterator logIterator = getLogIterator();
    Object nextObj = logIterator.getNext();
    int count = 0;
    while (nextObj != null){
      //System.err.println(nextObj); //should be 1 record
      count++;
      nextObj = logIterator.getNext();
    }
    logIterator.close();
    assert (count == 1);
    
  }

  //-----------------------------------------------------------------------
  // Test concurrent appends -- assume it right follows the previous 2 tests
  //-----------------------------------------------------------------------
  public void
  testMultiAppends()
  throws Exception{
    //System.err.println("start 3 insert threads...");
    InvalInserter invalInserter1 = new InvalInserter(this, 100, 1);
    InvalInserter invalInserter2 = new InvalInserter(this, 100, 100);
    InvalInserter invalInserter3 = new InvalInserter(this, 100, 1000);
    invalInserter1.start();
    invalInserter2.start();
    invalInserter3.start();

    invalInserter1.join();
    invalInserter2.join();
    invalInserter3.join();
    
    LogIterator logIterator = getLogIterator();
    Object nextObj = logIterator.getNext();
    int count = 0;
    while (nextObj != null){
      //System.err.println(nextObj); 
      count++;
      nextObj = logIterator.getNext();
    }
    logIterator.close();
    assert (count == 301);
  }

  //-----------------------------------------------------------------------
  // Test concurrent Truncate and appends -- note: to avoid assert false
  // it should be called following the previous 3 tests
  //-----------------------------------------------------------------------
  public void
  testTruncateAppend()
    throws Exception{
    //System.err.println("testTruncateAppend start...");
    GarbageCollecter garbageCollecter = new GarbageCollecter(this, 300);
    InvalInserter invalInserter1 = new InvalInserter(this, 100, 1);
    InvalInserter invalInserter2 = new InvalInserter(this, 100, 100);
    InvalInserter invalInserter3 = new InvalInserter(this, 100, 1000);
    garbageCollecter.start();
    invalInserter1.start();
    invalInserter2.start();
    invalInserter3.start();

    garbageCollecter.join();
    invalInserter1.join();
    invalInserter2.join();
    invalInserter3.join();

    LogIterator logIterator = getLogIterator();
    Object nextObj = logIterator.getNext();
    int count = 0;
    while (nextObj != null){
      //System.err.println(nextObj); 
      count++;
      nextObj = logIterator.getNext();
    }
    logIterator.close();
    assert (count == 1);
    //assert (nextSeq == 604);
    // System.err.println("nextSeq: " + nextSeq);
  }

  private ImpreciseInv createImpreciseInv(AcceptStamp[] as1, 
                                        AcceptStamp[] as2){
    String[] dir = new String[2];
    dir[0] = "/d1/*";
    dir[1] = "/d2/*";
    return  new ImpreciseInv(HierInvalTarget.makeHierInvalTarget(dir),
                             new AcceptVV(as1),
                             new AcceptVV(as2));
  }

  //---------------------------------------------------------------------
  // test the new truncateCompletely method
  //    check: does the log truncate all inv < truncateVV ?
  //---------------------------------------------------------------------
  public void
  testTruncateCompletely(){
    
    try {
        //  System.err.println("inserter [" + id + "] inserting " + ii);
      AcceptStamp[] as1 = null;
      AcceptStamp[] as2 = null;
      String[] dir = new String[2];
      dir[0] = "/d1/*";
      dir[1] = "/d2/*";
      ImpreciseInv ii = null;
      int id = 9;
      as1 = new AcceptStamp[2];
      as1[0] = new AcceptStamp(10, new NodeId(id));
      as1[1] = new AcceptStamp(10, new NodeId(id + 1));
      
      as2 = new AcceptStamp[2];
      as2[0] = new AcceptStamp(20, new NodeId(id));
      as2[1] = new AcceptStamp(20, new NodeId(id + 1));
      
      ii = createImpreciseInv(as1, as2);
      append(ii);

      as1 = new AcceptStamp[2];
      as1[0] = new AcceptStamp(100, new NodeId(id));
      as1[1] = new AcceptStamp(100, new NodeId(id + 1));
      
      as2 = new AcceptStamp[2];
      as2[0] = new AcceptStamp(200, new NodeId(id));
      as2[1] = new AcceptStamp(200, new NodeId(id + 1));
      
      ii = createImpreciseInv(as1, as2);
      append(ii);

      as1 = new AcceptStamp[2];
      as1[0] = new AcceptStamp(11, new NodeId(id));
      as1[1] = new AcceptStamp(11, new NodeId(id + 1));
      
      as2 = new AcceptStamp[2];
      as2[0] = new AcceptStamp(11, new NodeId(id));
      as2[1] = new AcceptStamp(11, new NodeId(id + 1));

      ii = createImpreciseInv(as1, as2);
      append(ii);
      
      as1 = new AcceptStamp[2];
      as1[0] = new AcceptStamp(110, new NodeId(id));
      as1[1] = new AcceptStamp(110, new NodeId(id + 1));
      
      as2 = new AcceptStamp[2];
      as2[0] = new AcceptStamp(120, new NodeId(id));
      as2[1] = new AcceptStamp(120, new NodeId(id + 1));

      ii = createImpreciseInv(as1, as2);
      append(ii);

      as1 = new AcceptStamp[2];
      as1[0] = new AcceptStamp(210, new NodeId(id));
      as1[1] = new AcceptStamp(210, new NodeId(id + 1));
      
      as2 = new AcceptStamp[2];
      as2[0] = new AcceptStamp(220, new NodeId(id));
      as2[1] = new AcceptStamp(220, new NodeId(id + 1));

      ii = createImpreciseInv(as1, as2);
      append(ii);

      id = 10;
      as1 = new AcceptStamp[2];
      as1[0] = new AcceptStamp(10, new NodeId(id));
      as1[1] = new AcceptStamp(10, new NodeId(id + 1));
      
      as2 = new AcceptStamp[2];
      as2[0] = new AcceptStamp(20, new NodeId(id));
      as2[1] = new AcceptStamp(20, new NodeId(id + 1));

      ii = createImpreciseInv(as1, as2);
      append(ii);

      id = 9;
      as1 = new AcceptStamp[2];
      as1[0] = new AcceptStamp(190, new NodeId(id));
      as1[1] = new AcceptStamp(10, new NodeId(id + 1));
      
      as2 = new AcceptStamp[2];
      as2[0] = new AcceptStamp(190, new NodeId(id));
      as2[1] = new AcceptStamp(10, new NodeId(id + 1));

      ii = createImpreciseInv(as1, as2);
      append(ii);
      
      System.out.println(this);
      AcceptStamp[] truncAS = new AcceptStamp[3];
      truncAS[0] = new AcceptStamp(20, new NodeId(9));
      truncAS[1] = new AcceptStamp(20, new NodeId(10));
      truncAS[2] = new AcceptStamp(0, new NodeId(11));
      System.err.println(truncateCompletely(new AcceptVV(truncAS)));
      System.out.println(this);
    }catch (Exception e){
      e.printStackTrace();
      assert false;
    }
  
  } 

  //---------------------------------------------------------------------
  // print the whole inval database
  //---------------------------------------------------------------------
  public String 
  toString(){
    String str = "\nInvalDb: \n";
    try{
      
      LogIterator logIterator = getLogIterator();
      Object nextObj = logIterator.getNext();
      int count = 0;

      while (nextObj != null){
        str += nextObj;
        str += "\n";
        count++;
        nextObj = logIterator.getNext();
      }
      logIterator.close();
      str += "\n total records: ";
      str += count;
      str += "\n nextSeq : ";
      str += nextSeq;
      
    }catch(Exception e){
      System.err.println("can't show checkpoint: " + e); 
    }
    str += "\nomitVV:";
    str += omitVV;
    str += "----------------------------";
    str += "diskSize: " + diskSize();
    try{
      str += "fileSize: " + accurateDiskSize();
    }catch (Exception e){
      System.err.println(e.toString());
      assert false;
    }
    return str;
  }
  //-----------------------------------------------------------------------
  // stress test: repeat appending 1000 records, gabage collecting 
  // 1000 records over night to see if the database grows.
  //-----------------------------------------------------------------------
  public void
  stressTestOvernight()
  throws Exception{
    System.err.println("Stress Testing ..." );
    while (true){
      testTruncateAppend();
    }
  }

  //---------------------------------------------------------------------
  // for append try to see if cursor(open in the constructor, close in 
  // destructor) improves performance.
  // Performance Test 1 -- small append with bound inv of body size 5 bytes
  //---------------------------------------------------------------------
  public void 
  perfTest1(){
    int appendNum = 100000;
    long start, end;
    double avgMS;
    byte[] bdy = null;
    bdy = new byte[5];
    bdy[0] = 10;
    bdy[1] = 9;
    bdy[2] = 8;
    bdy[3] = 7;
    bdy[4] = 6;
    ImmutableBytes body = new ImmutableBytes(bdy);
    BoundInval bi = null;
    
    bi = new BoundInval(new ObjId("/d1/bound2"),
                        10,
                        body.getLength(),
                        new AcceptStamp(50, new NodeId(200)),
                        body);
    
    try{
      start = System.currentTimeMillis();
      for(int i = 0; i < appendNum; i++){
        append(bi);
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)appendNum;
      System.err.print("append 1 byte boundinval: " + avgMS + "ms...");
    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }
  
  //---------------------------------------------------------------------
  // for append try to see if cursor(open in the constructor, close in 
  // destructor) improves performance.
  // Performance Test 2 -- append with bound inv of body size 1000 bytes
  //---------------------------------------------------------------------
  public void 
  perfTest2(){
    int appendNum = 1000;
    long start, end;
    double avgMS;
    int bodySize = 1000;
    byte[] bdy = null;
    bdy = new byte[bodySize];
    
    for(int i = 0; i < bodySize; i++){
      bdy[i] = 1;
    }
    ImmutableBytes body = new ImmutableBytes(bdy);
    BoundInval bi = null;
    
    bi = new BoundInval(new ObjId("/d1/bound2"),
                        10,
                        body.getLength(),
                        new AcceptStamp(50, new NodeId(200)),
                        body);
    
    try{
      start = System.currentTimeMillis();
      for(int i = 0; i < appendNum; i++){
        append(bi);
      }
      end = System.currentTimeMillis();
      avgMS = ((double)((double)end - (double)start))/(double)appendNum;
      System.err.print("append 1k byte boundinval: " + avgMS + "ms...");
      
    }catch(Exception e){
      e.printStackTrace();
      assert(false);
    }
  }

  private static void testDiskSize(String path, long cacheSize, 
                                   long maxSize, int type){
    try{
      PersistentLog log = new PersistentLog(path, cacheSize, true, false);
      String typeName = "";
      long start, end;
      double avgMS;
      int bodySize = 10;
      byte[] bdy = null;
      bdy = new byte[bodySize];
    
      for(int i = 0; i < bodySize; i++){
        bdy[i] = 1;
      }
      ImmutableBytes body = new ImmutableBytes(bdy);
      BoundInval bi = null;
      
      bi = new BoundInval(new ObjId("/d1/bound2"),
                          10,
                          body.getLength(),
                          new AcceptStamp(50, new NodeId(200)),
                          body);

      ObjInvalTarget oit;
      oit = new ObjInvalTarget(new ObjId(new String("/a")),
                               0,
                               500);
    
      PreciseInv pi = new PreciseInv(oit,
                                     new AcceptStamp(10, new NodeId(10)),
                                     new AcceptStamp(10, new NodeId(10)),
                                     false);
      
      UnbindMsg um = new UnbindMsg(oit,
                                   new AcceptStamp(11, new NodeId(10)));
      
      AcceptStamp[] as = new AcceptStamp[1];
      as[0] = new AcceptStamp(14, new NodeId(10));
      
      AcceptStamp[] as3 = new AcceptStamp[1];
      as3[0] = new AcceptStamp(100, new NodeId(10));
      
      HierInvalTarget hit = HierInvalTarget.makeHierInvalTarget("/a");
      
      SingleWriterImpreciseInv si = 
        new SingleWriterImpreciseInv(hit,
                                     as[0],
                                     as3[0],
                                     as[0]);
      
      ImpreciseInv ii = 
        new ImpreciseInv(hit,
                         new AcceptVV(as),
                         new AcceptVV(as3),
                         new AcceptVV(as));
      
      
      int appendNum = 0;
      try{
        start = System.currentTimeMillis();
        
        while (log.diskSize()< maxSize){
          switch(type){
          case 0: 
            typeName = "BoundInv";
            log.append(bi);
            break;
          case 1:
            typeName = "PreciseInv";
            log.append(pi);
            break;
          case 2:
            typeName = "UnbindMsg";
            log.append(um);
            break;
          case 3:
            typeName = "SingleWriterImpreciseInv";
            log.append(si);
            break;
          case 4:
            typeName = "ImpreciseInv";
            log.append(ii);
            break;
          }
          appendNum++;
        }
        end = System.currentTimeMillis();
        avgMS = ((double)((double)end - (double)start))/(double)appendNum;
        System.err.print("append 1k byte boundinval: " + avgMS + "ms...");
      
      }catch(Exception e){
        e.printStackTrace();
        assert(false);
      }
      System.out.println(log.diskSize() + ":" + log.accurateDiskSize());
      long diviation = log.accurateDiskSize()-log.diskSize();
      System.out.println("diviation: " + diviation);
      System.out.println("diviation rate:" 
                         + (double)diviation/(double)log.diskSize());
      System.out.println("Type: " + typeName +"\n overhead/record: " 
                         + (double)diviation/(double)appendNum); 
      log.close();
    } catch (Exception e){
      e.printStackTrace();
      System.out.println(e.toString());
    }
  }

  private static void testInterface(String path, long cacheSize){
    boolean end = false;
    try{
      PersistentLog log = new PersistentLog(path, cacheSize, true, false);
      System.out.println("This test interface allows you append impreciseInv for"
                         + "nodes: 9, 10, 11 and specify truncates with timestamp" 
                         + " for 9, 10, 11");
      System.out.println("Enter action:  ");
      
      System.out.println("AppendInv    : a"
                         +" [nodeId1] [nodeId2] [startTime1] [startTime2]" 
                         +"[endTime1] [endTime2]");
      System.out.println("truncate    : t [time1] [time2] [time3]");
      System.out.println("showlog     : s");
      System.out.println("end         : e");
	    
      InputStreamReader tempReader=new InputStreamReader(System.in);
      BufferedReader reader=new BufferedReader(tempReader);
      String[] dir=new String[2];
      dir[0] = "/a";
      dir[1] = "/b";
      while(!end){
        String input=reader.readLine();
        System.out.println(" ");
        System.out.println("Input: " + input);
        if (input == null) return;
        byte[] action = input.getBytes();
        
			
        if(action[0] == 'a'){
         
          StringTokenizer st = new StringTokenizer(input.substring(2));
			
          long nodeId1 = (new Long(st.nextToken())).longValue();
          long nodeId2 = (new Long(st.nextToken())).longValue();
          long startTime1 = (new Long(st.nextToken())).longValue();
          long startTime2 = (new Long(st.nextToken())).longValue();
          long endTime1 = (new Long(st.nextToken())).longValue();
          long endTime2 = (new Long(st.nextToken())).longValue();
		    
          AcceptStamp[] as1 = new AcceptStamp[2];
          as1[0] = new AcceptStamp(startTime1, new NodeId(nodeId1));
          as1[1] = new AcceptStamp(startTime2, new NodeId(nodeId2));
		    
          AcceptStamp[] as2 = new AcceptStamp[2];
          as2[0] = new AcceptStamp(endTime1, new NodeId(nodeId1));
          as2[1] = new AcceptStamp(endTime2, new NodeId(nodeId2));
		    
          ImpreciseInv ii = new ImpreciseInv(HierInvalTarget.makeHierInvalTarget(dir),
                                             new AcceptVV(as1),
                                             new AcceptVV(as2));
          log.append(ii);
          System.out.println(log);	    
        } else if (action[0] == 't'){
          StringTokenizer st = new StringTokenizer(input.substring(2));
		    
          long t1 = (new Long(st.nextToken())).longValue();
          long t2 = (new Long(st.nextToken())).longValue();
          long t3 = (new Long(st.nextToken())).longValue();
		    
          AcceptStamp[] as1 = new AcceptStamp[3];
          as1[0] = new AcceptStamp(t1, new NodeId(9));
          as1[1] = new AcceptStamp(t2, new NodeId(10));
          as1[2] = new AcceptStamp(t3, new NodeId(11));

          AcceptVV acceptVV1 = new AcceptVV(as1);
		    
          log.truncateCompletely(acceptVV1);
          System.out.println(log);
          System.out.println("omitVV: " + log.getOmitVV());
		    
        } else if (action[0] == 's'){   
          System.out.println(log);
        
        } else if (action[0] == 'e'){
          end = true;
        }
      }
       
    } catch (Exception e){
      e.printStackTrace();
      System.out.println("Wrong input format ...");
    }
  }
    
  
  //----------------------------------------------------------------------
  //  Self Test
  //----------------------------------------------------------------------
  public static void
  main(String[] argv)
  throws Exception{

    if(argv.length <2){
      System.out.println("Usage: java -ea PersistentLog [MaxDiskSize] [Type:"
                         + " 0-BoundInv 1-PreciseInv 2-UnbindMsg"
                         + " 3-SingleWriterImpreciseInv 4-ImpreciseInv]");
      System.exit(0);
    }
    Env.verifyAssertEnabled();
    System.err.print("PersistentLog self test...");
    
    String configFile ="/projects/lasr/space0/zjd/" + 
      "universalReplication/code/ufs-local-Persistent-selfTest.config" ;
    
    Config.readConfig(configFile);
    long cacheSize = 1000000;
    String dbpath = "/tmp/ufs";
    AcceptStamp[] as2 = new AcceptStamp[2];
    as2[0] = new AcceptStamp(10, new NodeId(9));
    as2[1] = new AcceptStamp(10, new NodeId(10));

    AcceptVV av = new AcceptVV(as2);
    AcceptStamp[] as1 = new AcceptStamp[3];
    as1[0] = new AcceptStamp(10, new NodeId(9));
    as1[1] = new AcceptStamp(10, new NodeId(10));
    as1[2] = new AcceptStamp(0, new NodeId(11));
    
    AcceptVV tr = new AcceptVV(as1);
    testDiskSize(dbpath, 
                 cacheSize, 
                 new Long(argv[0]).longValue(),
                 new Integer(argv[1]).intValue());
    /*
      //testInterface(dbpath, cacheSize);
    
      PersistentLog plog1 = null;
      PersistentLog plog2 = null;
      try{
      if (argv.length > 0){ //print invalDb
      plog2 = new PersistentLog(dbpath, cacheSize, false);
      System.out.println(plog2);
      
      } else { //self testing
      File dbDir = new File(dbpath);
      
        try{
          File[] dbFiles = dbDir.listFiles();
      
          if (dbFiles != null){
            for (int i = 0; i < dbFiles.length; i ++){
              dbFiles[i].delete();
            }
          }
          
        } catch (SecurityException se){
          assert false; //should never be here.
        }
        plog1 = new PersistentLog(dbpath, cacheSize, true);
        plog1.simpleTest();
        plog1.close();
        plog1 = null;
        plog2 = new PersistentLog(dbpath, cacheSize, false);
        plog2.testRecover();
        
        // plog2.testMultiAppends();
        plog2.testTruncateAppend();
        //plog2.stressTestOvernight();
        //plog2.perfTest1();
        //plog2.perfTest2();
        plog2.close();
        plog1 = new PersistentLog(dbpath, cacheSize, true);
        plog1.testTruncateCompletely();
        
        
      }
    }finally{
      if (plog1 != null){
        plog1.close();
      }
      if (plog2 != null){
        plog2.close();
      }
    }
    System.err.println("...PersistentLog self test succeeds");
    System.exit(0);
    */
  }
}

//-----------------------------------------------------------------
// Garbage Collect thread to keep truncating 10 records units till 
// the total number truncated reachs the given number.
//-----------------------------------------------------------------
class GarbageCollecter extends Thread {
  PersistentLog plog;
  long num;  // total number to truncate.

  //---------------------------------------------------------------
  // Constructor
  //---------------------------------------------------------------
  public GarbageCollecter(PersistentLog plog, long num){
    this.plog = plog;
    this.num = num;
  }
    
  //---------------------------------------------------------------
  // garbage collect
  //---------------------------------------------------------------
  public void
  run(){
    try{
      
      //System.err.println("GarbageCollecter start ... ");
      long count = 0;
      while (count < num){
        count = count + plog.truncateInval(10, null);
        //System.err.println("total truncated num = " + count);
      }
    }catch (Exception e){
      e.printStackTrace();
      assert false;
    }
  }
}

//-----------------------------------------------------------------
// Insert Thread to append the given number of invalidates 
//-----------------------------------------------------------------
class InvalInserter extends Thread{
  PersistentLog plog;
  int invalNum;
  int id;  // the id of this thread, it's also used as the nodeid
           // for acceptstamp

  //--------------------------------------------------------------
  // constructor
  //--------------------------------------------------------------
  public InvalInserter(PersistentLog plog, int invalNum, int id){
    this.plog = plog;
    this.invalNum = invalNum;
    this.id = id;
  }

  //--------------------------------------------------------------
  // Append invalidates
  //--------------------------------------------------------------  
  public void
  run(){
    //System.err.println("inserter : [ " + id + " ] start...");
    AcceptStamp[] as1 = null;
    AcceptStamp[] as2 = null;
    String[] dir = new String[2];
    dir[0] = "/d1/*";
    dir[1] = "/d2/*";
    ImpreciseInv ii = null;
    for(int i = 0; i < invalNum; i++){
      as1 = new AcceptStamp[2];
      as1[0] = new AcceptStamp(i, new NodeId(id));
      as1[1] = new AcceptStamp(i, new NodeId(id + 1));
	    
      as2 = new AcceptStamp[2];
      as2[0] = new AcceptStamp(i + 1, new NodeId(id));
      as2[1] = new AcceptStamp(i + 1, new NodeId(id + 1));
    
      ii = new ImpreciseInv(HierInvalTarget.makeHierInvalTarget(dir),
                            new AcceptVV(as1),
                            new AcceptVV(as2));
      try {
        //  System.err.println("inserter [" + id + "] inserting " + ii);
        plog.append(ii);
      }catch (Exception e){
        e.printStackTrace();
        assert false;
      }
    }
    //System.err.println("inserter : [" + id + "] finish..."); 
  }
}
//---------------------------------------------------------------------------
/* $Log: PersistentLog.java,v $
/* Revision 1.19  2007/04/12 17:09:32  zjiandan
/* fix across unit test mem leak.
/*
/* Revision 1.18  2007/04/02 21:11:38  zjiandan
/* snapshort for sosp2007.
/*
/* Revision 1.17  2006/11/01 17:04:32  nayate
/* Added a parameter to control log sync-ing
/*
/* Revision 1.16  2006/09/06 16:11:55  dahlin
/* *** empty log message ***
/*
/* Revision 1.15  2006/09/01 21:49:11  dahlin
/* PicShareReaderUnit now works. BIG ISSUE -- fixed bug where PersistentLog did not actually sync all transactions to disk. 30x slowdown on small writes. See mike's email of 9/1/2006 and notes/2006/8/8b.txt
/*
/* Revision 1.14  2006/09/01 21:32:24  dahlin
/* PicSharReader test now works for case when writer dies and restarts
/*
/* Revision 1.13  2006/09/01 15:59:01  dahlin
/* Added test of core recovery. It works -- including log replay.
/*
/* Revision 1.12  2005/10/17 19:22:29  zjiandan
/* Implemented Non-blocking CP exchange update.
/*
/* Revision 1.11  2005/10/13 00:24:24  zjiandan
/* remove Config.getMy* fixed Garbage Collection and Checkpoint exchange code
/*
/* Revision 1.10  2005/07/19 02:12:45  zjiandan
/* Add Logic to stop outgoing invalidates streams whenever
/* log garbage collector outperforms InvalIterator, i.e.
/* Log.inMemOmitVV > InvalIterator.currentVV.
/*
/* Revision 1.9  2005/07/18 05:10:23  zjiandan
/* Embargoed Writes etc. features implementation plus
/* log overhead measurement with disk size and in-memory size.
/*
/* Revision 1.8  2005/03/15 21:15:34  zjiandan
/* Automatic GC checked in.
/*
/* Revision 1.7  2005/03/03 20:25:10  zjiandan
/* minor bugs fixed.
/*
/* Revision 1.6  2005/03/01 02:09:13  nayate
/* Modified for new code
/*
/* Revision 1.5  2005/02/28 20:25:59  zjiandan
/* Added Garbage Collection code and part of Checkpoint exchange protocol code
/*
/* Revision 1.4  2005/01/10 03:47:47  zjiandan
/* Fixed some bugs. Successfully run SanityCheck and Partial Replication experiments.
/*
/* Revision 1.3  2004/11/02 22:24:33  zjiandan
/* add utility methods for core recovery self test.
/*
/* Revision 1.2  2004/10/22 18:13:39  zjiandan
/* cleaned csn from UpdateLog, modified DeleteInv.
/* TBD:
/*    1. clean csn from all subclasses of GeneralInv
/*    2. fix *Inv::cloneIntersectInvaltargetChopStartEnd
/*
/* Revision 1.1  2004/10/13 17:41:36  zjiandan
/* Initial implementation of PersistentLog tested with DataStore stubs.
/*
/* TBD: test recovery with integrated DataStore and RandomAccessState.
/**/
//---------------------------------------------------------------------------

