 /** 
/* PicShareConfig.java
 *
 * Read config file and provide interface for accessing it.
 * 
 * Config file has
 *
 *    MODE     READER_FROM_PICSHARE|WRITER_TO_PICSHARE    // Reader Node ID > LOWEST_READER_NODE_ID
 *    ALBUM_NAME  name
 *    PASSWORD    password
 *    INPUT_PATH  path-to-input files // for WRITER only
 *    OUTPUT_PATH path to output files // for READER only
 *
 * Defines layout of where data stored in PRACTI id space
 * H(data_key)/ 
 *             update_list/node[nodeId] -- history of updates
 *             receive_list[reader]/node[writer] -- which updates from 
 *                                                  [writer] has [reader] 
 *                                                  processed?
 *             data/* -- replicated files
 *             metadata/* -- per-object metadata
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
 **/ 


import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;
import java.io.*;
import java.security.MessageDigest;
import java.util.List;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.Enumeration;

 /** 
 *  Config:: parse config file and store it in the in-mem data structure
 
 *           
 
 *           to add a new field:
 
 *              0. Update MAX_CONFIG_FIELDS
 
 *              1. add a member to store the value
 
 *              2. add an element for longopts in readConfig()
 
 *                 longopts[14] =  new LongOpt("COMM_PORT", LongOpt.REQUIRED_ARGUMENT, null, 14);
 
 *              3. add a case like this:
 
 *                  case 14:
 
 * 	           commPort = (new Integer(g.getOptarg())).intValue();
 
 * 	           break ;
 
 *              4. Update sanityCheckConfig to make sure required fields present
 
 **/ 

public class PicShareConfig{

  SubscriptionSet readerTargets;
  //
  // Assume all nodes from 1 to 99 are writers.
  // All nodes 100+ are readers.
  //
  public final static int LOWEST_WRITER_NODE_ID = 1;
  public final static int LOWEST_READER_NODE_ID = 100;

  //
  // Album name and password hashed together to give
  // unique base for PRACTI path
  //
  private String albumName = null;
  private String password = null;

  private String localPath = null;

  //
  // uniqHash is base ID in PRACTI space
  //
  private String uniqHash;

  //
  // Should writer scan body to make sure that it matches
  // what is stored in PRACTI? False --> reduce overhead.
  //
  public final static boolean verifyBody = true;

  public PicShareConfig(String pathToConfigFile, NodeId nid){

    readConfig(pathToConfigFile, nid);
    
    uniqHash = makeUniqHash();

    readerTargets = SubscriptionSet.makeSubscriptionSet("/" + uniqHash + "/data/*:"
                                                        + "/" + uniqHash + "/metadata/*:"
                                                        + "/" + uniqHash + "/config/*");
  }

  //-------------------------------------------------------------------------
  // readConfig
  //-------------------------------------------------------------------------
  private void readConfig(String pathToConfigFile, NodeId myId){
    int MAX_CONFIG_FIELDS = 4;
    LongOpt[] longopts = new LongOpt[MAX_CONFIG_FIELDS+1];
    longopts[0] = new LongOpt("PASSWORD", LongOpt.REQUIRED_ARGUMENT, null, 0);
    longopts[1] = new LongOpt("ALBUM_NAME", LongOpt.REQUIRED_ARGUMENT, null, 1);
    longopts[2] = new LongOpt("INPUT_PATH", LongOpt.REQUIRED_ARGUMENT, null, 2);
    longopts[3] = new LongOpt("OUTPUT_PATH", LongOpt.REQUIRED_ARGUMENT, null, 3);
    longopts[4] = new LongOpt("MODE", LongOpt.REQUIRED_ARGUMENT, null, 4);

    int input;
    String mode = null;
    String fileInput[] = parseFile(pathToConfigFile);
    Getopt g = new Getopt("Config", fileInput, "", longopts, true);
    while((input = g.getopt()) != -1){
      switch(input){
      case 0:
        password = g.getOptarg();
        break;
      case 1:
        albumName = g.getOptarg();
        break;
      case 2:
        //
        // Notice that MODE WRITER argument needs to come before WRITE_PATH
        //
        if(localPath != null){
          System.out.println("PATH specified multiple times in  "
                             + pathToConfigFile);
          System.exit(-1);
          
        }
        if(mode != null && mode.equals("WRITER_TO_PICSHARE")){
          localPath = g.getOptarg();
        }
        else{
          System.out.println("INPUT_PATH specified but mode is not set to WRITER_TO_PICSHARE in "
                             + pathToConfigFile);
          System.exit(-1);
        }
        break;
      case 3:
        //
        // Notice that MODE READER argument needs to come before READ_PATH
        //
        if(localPath != null){
          System.out.println("PATH specified multiple times in  "
                             + pathToConfigFile);
          System.exit(-1);
          
        }
        if(mode != null && mode.equals("READER_FROM_PICSHARE")){
          localPath = g.getOptarg();
        }
        else{
          System.out.println("OUTPUT_PATH specified but mode is not set to READER_FROM_PICSHARE in "
                             + pathToConfigFile);
          System.exit(-1);
        }
        break;
      case 4:
        mode = g.getOptarg();
        if(mode.equals("READER_FROM_PICSHARE")){
          assert(myId.geq(new NodeId(LOWEST_READER_NODE_ID)));
        }
        else if(mode.equals("WRITER_TO_PICSHARE")){
          assert(myId.lt(new NodeId(LOWEST_READER_NODE_ID)));
        }
        else{
          System.out.println("Unexpected MODE (not READER or WRITER) in " +
                             pathToConfigFile);
        }
        break;
      default:
        assert(false);
        break;
      }
    }
    sanityCheckConfig();
  }
    
  //-------------------------------------------------------------------------
  // sanityCheckConfig()
  //-------------------------------------------------------------------------
  private void sanityCheckConfig(){
    assert(password != null);
    assert(albumName != null);
    assert(localPath != null);
  }

  //-------------------------------------------------------------------------
  // parseFile -- return array of "-key value" strings for getOpt to operate on
  //-------------------------------------------------------------------------
  private static String[] parseFile(String configFile){
    Vector config = new Vector();
    String line = null;
    int lineCounter = 0;
    try{
      BufferedReader in = new BufferedReader(new FileReader(configFile));
      assert in!=null;
      while((line = in.readLine()) != null){
        validateConfigLine(line);
        /* "-" is added infront of each line to construct the format of the input arguments. */
        StringTokenizer st = new StringTokenizer("-" + line);
        while(st.hasMoreTokens()){
          config.add(st.nextToken());
        }
      }
    } catch (Exception e){
      System.out.println("Problem reading config: " + e.toString());
      e.printStackTrace();
      assert(false);
    }
    Object[] objA = config.toArray();
    String[] strA = new String[objA.length];
    for(int i=0; i<objA.length; i++){
      strA[i] = (String) objA[i];
    }
    return strA;
  }
  
 /** 
 *  validateConfigLine - make sure each line is a KEY VALUE pair
 
 **/ 
  private static void validateConfigLine(String line){
    StringTokenizer st = new StringTokenizer(line);
    if((st.countTokens() == 1 && !st.nextToken().equals("***END***"))||
       (st.countTokens() > 2)){ 
      System.err.println("Bad configuration line: " + line);
      System.exit(-1);
    }
  }

 /** 
 *  makeUniqHash -- generate a hash of the password and album name
 
 *  We use a weak message digest because we are just trying to avoid
 
 *  inadvertant collisions...the path names will be sent in cleartext
 
 *  anyhow...
 
 **/ 
  private String makeUniqHash(){
    assert(albumName != null);
    assert(password != null);

    try{
      String s1 = albumName + password;
      byte b[] = s1.getBytes("UTF-16LE");
      MessageDigest d = MessageDigest.getInstance("SHA-256");
      d.update(b);
      byte result[] = d.digest();
      downsample(result);
      return new String(result, "US-ASCII");
    }
    catch(UnsupportedEncodingException uee){
      System.out.println("Unexpected error: " + uee.toString());
      uee.printStackTrace();
      System.exit(-1);
      return null; // Make compiler happy
    }
    catch(java.security.NoSuchAlgorithmException nsae){
      System.out.println("Unexpected error: " + nsae.toString());
      nsae.printStackTrace();
      System.exit(-1);
      return null; // Make compiler happy
    }

  }


 /** 
 *  downsample -- convert hash to bytes corresponding to 0-9, a-z, A-Z
 
 * 
 
 *  We discard bits from the message digest because we want to
 
 *  get a legit path name. This is OK b/c are just trying to avoid
 
 *  inadvertant collisions...the path names will be sent in cleartext
 
 *  anyhow...
 
 **/ 
  private void downsample(byte b[]){
    int ii;
    for(ii = 0; ii < b.length; ii++){
      b[ii] = (byte)(b[ii] % 62);
      if(b[ii] < 0){
        b[ii] += 62;
      }
      assert(b[ii] >= 0);
      if(b[ii] < 10){
        b[ii] = (byte)('0' + b[ii]);
      }
      else{
        b[ii] = (byte)(b[ii] - 10);
        if(b[ii] < 26){
          b[ii] = (byte)('A' + b[ii]);
        }
        else{
          b[ii] = (byte)(b[ii] - 26);
          assert(b[ii] < 26);
          b[ii] = (byte)('a' + b[ii]);
        }
      }
    }
  }


 /** 
 *  Return base PRACTI path to data
 
 **/ 
  public String basePRACTIDataPath(){
    return "/" + uniqHash + "/data/";
  }

 /** 
 *  Return PRACTI objId for obj given path relative to base
 
 **/ 
  public ObjId objIdPRACTIDataPath(String pathRelativeToBase){
    String fullPRACTIpath = basePRACTIDataPath() + "/" + pathRelativeToBase;
    return new ObjId(fullPRACTIpath);
  }

 /** 
 *  Return base PRACTI path to metadata
 
 **/ 
  public String basePRACTIMetadataPath(){
    return "/" + uniqHash + "/metadata/";
  }

 /** 
 *  Return PRACTI objId for obj given path relative to base
 
 **/ 
  public ObjId objIdPRACTIMetadataPath(String pathRelativeToBase){
    String fullPRACTIpath = basePRACTIMetadataPath() + "/" + pathRelativeToBase;
    return new ObjId(fullPRACTIpath);
  }

 /** 
 *  Return base local file system path to data
 
 **/ 
  public File baseLocalDataPath(){
    return new File(localPath);
  }


 /** 
 *  Return base PRACTI path to 
 
 **/ 
  public String basePRACTIStatusPath(){
    return "/" + uniqHash + "/config/status/";
  }

 /** 
 *  Return a List of paths that all readers should 
 
 *  subscribe to
 
 **/ 
  public synchronized SubscriptionSet getReaderSubscriptionTargets(){
    return (SubscriptionSet)readerTargets.clone();
  }
  
 /** 
 *  Return PRACTI id of an object that lists history of all writes
 
 *  by a node
 
 *  H(data_key)/update_list/node[id]
 
 **/ 
  public synchronized ObjId getWriteHistoryId(NodeId id){
    return new ObjId("/" + uniqHash + "/config/writeHistory/node" + id.toString());
  }

 /** 
 *  Return PRACTI id of an object that lists history of all writes
 
 *  by a node
 
 *  H(data_key)/update_list/node[id]
 
 **/ 
  public synchronized ObjId getReaderHistoryId(NodeId reader, NodeId writer){
    assert(reader.gt(writer)); // Guard against backwards args...
    return new ObjId("/" + uniqHash + "/config/readHistory/reader" 
                     + reader.toString() + "/writer" + writer.toString());
  }

 /** 
 *  return sleep time between scans by writer in milliseconds
 
 **/ 
  public synchronized long getWriterScanFrequencyMS(){
    // return 60 * 60 * 1000; // 60 minutes
    System.out.println("DBG PicShareConfig scan frequency of 3 seconds");
    return 3 * 1000;
  }

 /** 
 *  return sleep time between scans by reader in milliseconds
 
 **/ 
  public synchronized long getReaderScanFrequencyMS(){
    //return 60 * 60 * 1000; // 60 minutes
    System.out.println("DBG PicShareConfig scan frequency of 3 seconds");
    return 3 * 1000;
  }

 /** 
 *  Return list of writers
 
 * 
 
 *  Note -- depends on Config already initialized (or it
 
 *  will throw a null pointer exception, which seems better
 
 *  than silently giving back an empty list, right?)
 
 **/ 
  public synchronized List getWriters() {
    Enumeration allNodes = Config.getKnownNodeIds();
    Vector writers = new Vector();
    NodeId minReader = new NodeId(LOWEST_READER_NODE_ID);
    NodeId minWriter = new NodeId(LOWEST_WRITER_NODE_ID);
    while(allNodes.hasMoreElements()){
      NodeId id = (NodeId)allNodes.nextElement();
      Assert.affirm(id.geq(minWriter));
      if(id.lt(minReader)){
	writers.add(id);
      }
    }
    return (List)writers;
  }


 /** 
 *  Generate config file
 
 **/ 
  public static void createConfigFile(String configFilePath,
                                      boolean reader,
                                      String albumName,
                                      String password,
                                      String path){
    try{
      File f = new File(configFilePath);
      FileWriter fw = new FileWriter(f);
      String mode, pt;
      if(reader){
        mode = "READER_FROM_PICSHARE";
        pt = "OUTPUT_PATH";
      }
      else{
        mode = "WRITER_TO_PICSHARE";
        pt = "INPUT_PATH";
      }
      fw.write("MODE " + mode + "\n");
      fw.write("PASSWORD " + password + "\n");
      fw.write("ALBUM_NAME " + albumName + "\n");
      fw.write(pt + " " + path + "\n");
      fw.close();
    }
    catch(IOException ieo){
      ieo.printStackTrace();
      System.exit(-1);
    }
  }
    

}
