 /** 
/* PicShareReader.java
 * 
 * Program that runs at the reader side of PicShare picture sharing
 * application. It both provides an interface for reading data out
 * of PRACTI to the local system and it handles the subscriptions
 * and internal PRACTI requests needed to suck the data from the 
 * writer.
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
 **/ 
import java.io.*;
import java.util.Vector;
import java.util.Enumeration;

public class PicShareReader{

  Policy policy;
  P2Runtime runtime;
  LocalInterface localInterface;
  PicShareConfig picShareConfig;
  NodeId myId;

 /** 
 *  Command line 
 **/ 
  public static void main(String[] args){
    
    if(args.length < 2){
      System.out.println("Usage: PicShareReader "
			 + "<configDirectory> <myNodeId>");
      System.exit(-1);
    }
    String configBase = args[0];
    String PRACTIconfigPath = configBase + File.separatorChar 
      + "practi_picshare_config";
    String p2ConfigPath = configBase + File.separatorChar 
      + "p2_picshare_config";
    String picShareConfigPath = configBase + File.separatorChar 
      + "picshare_picshare_config";
    NodeId myId = new NodeId((new Integer(args[1])).intValue());
    PicShareReader psr = new PicShareReader(PRACTIconfigPath, 
					    p2ConfigPath, 
					    picShareConfigPath,
					    myId, false);
    psr.loop();
  }


 /** 
 *  Constructor 
 **/ 
  public PicShareReader(String PRACTIconfigPath, 
			String p2ConfigPath, 
			String picShareConfigPath,
			NodeId myId, 
			boolean clearDB){
    picShareConfig = new PicShareConfig(picShareConfigPath, myId);
    policy = new PicShareReaderPolicy(picShareConfig);
    this.myId = (NodeId)myId.clone();
    runtime = new P2Runtime(PRACTIconfigPath, p2ConfigPath, 
			    myId, clearDB, policy, false);
    policy.setRuntime(runtime);
    runtime.start();
    localInterface = new LocalInterface(runtime.getController(), 
					runtime.getCore());
  }

 /** 
 *  Shutdown -- release ports; try to kill most threads... 
 **/ 
  public void shutdown(){
    if(runtime != null){
      runtime.shutdown();
    }
    runtime = null;
  }

 /** 
 *  Get local interface 
 **/ 
  public LocalInterface getLocalInterface(){
    return localInterface;
  }


 /** 
 *  This is used in the unit tests to query the policy about 
 *  how many demand reads it saw... 
 **/ 
  protected PicShareReaderPolicy dbgGetPolicy(){
    return (PicShareReaderPolicy)policy;
  }


 /** 
 *  loop() 
 **/ 
  private void loop(){
    while(true){
      int ii;
      assert(PicShareConfig.LOWEST_WRITER_NODE_ID < PicShareConfig.LOWEST_READER_NODE_ID);
      for(ii = PicShareConfig.LOWEST_WRITER_NODE_ID;
          ii < PicShareConfig.LOWEST_READER_NODE_ID;
          ii++){
        getNewFiles(ii);
      }
      try{
        Thread.sleep(picShareConfig.getReaderScanFrequencyMS());
      }
      catch(InterruptedException e){
      }
      
    }
  }

 /** 
 *  getNewFiles -- get any files written by writer ii that we don't 
 *  already have 
 **/ 
  private void getNewFiles(int writer){
    int ii;
    NodeId writerNode = new NodeId(writer);
    ObjId writerHistory = picShareConfig.getWriteHistoryId(writerNode);
    ObjId readerHistory = picShareConfig.getReaderHistoryId(myId, writerNode);
    PicShareWriteHistory writeHistory = new PicShareWriteHistory(localInterface, writerHistory);
    if(writeHistory.isEmpty()){
      return;
    }
    PicShareWriteHistory readHistory = new PicShareWriteHistory(localInterface, readerHistory);
    for(ii = readHistory.getNextSeq();
        ii < writeHistory.getNextSeq();
        ii++){
      PicShareWriteHistoryEntry entry = writeHistory.getWriteEntry(ii);
      readAndWrite(entry);
      readHistory.update(entry.getObjMeta(), entry.getRelativeFileId(), ii);
    }
  }

 /** 
 *  readAndWrite() -- read item from PRACTI, decrypt it, write it to 
 *  local store 
 **/ 
  private void readAndWrite(PicShareWriteHistoryEntry entry){
    System.out.println("STUB: PicShareReader::readAndWrite()");

    String relativeFileId = entry.getRelativeFileId();

    ObjId oid = picShareConfig.objIdPRACTIDataPath(relativeFileId);
    
    byte b[] = readAllPracti(oid);

    try{
      File base = picShareConfig.baseLocalDataPath();
      File target = new File(base, relativeFileId);
      File parent = target.getParentFile();
      if(parent != null){
        parent.mkdirs();
      }
      FileOutputStream fos = new FileOutputStream(target);
      fos.write(b);
      fos.close();
    }
    catch(IOException e){
      if(relativeFileId.endsWith("Thumbs.db")){
        // Ignore failes attempt to update protected/in-use thumbs.db file
      }
      else{
        System.out.println("IO exception. Exiting.");
        e.printStackTrace();
        System.exit(-1);
      }
    }
    
  }

 /** 
 *  readAllPracti -- read entire practi object and return it. Keep retrying 
 *  until success. For now, we assume one read returns the whole shebang. 
 **/ 
  private byte[] readAllPracti(ObjId oid)
  {
    
    try{
      BodyMsg bm = localInterface.read(oid, 0, Integer.MAX_VALUE, true, true);
      byte b[] = bm.getBody().getCopyBytes();
      return b;
    }
    catch(Exception e){
      e.printStackTrace();
      System.exit(-1);
      return null; // Keep compiler happy
    }
  }

}
