/*
 * Unit test that stress tests case where we garbage collect
 * logs.
 */

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;

/**
 * TBD: Update class name
 */
public class PicShareReaderGCUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean verbose = true; // Start/end of test
  protected static boolean vverbose = true; // Test internals
  private Process rmiregistry;
  private Process writer;
  private BarrierServer barrier;
  public static final int BARRIER_PORT = 9300;
  public static final long CACHE_SIZE_BYTES = 500000; // 500KB
  public static final long MAX_READER_LOG_DISK_SIZE_BYTES = 2000000; // 2MB
  public static final long MAX_READER_LOG_MEM_SIZE_BYTES = 1000000; // 1MB
  public static final long MAX_WRITER_LOG_DISK_SIZE_BYTES = 4000000; // 4MB
  public static final long MAX_WRITER_LOG_MEM_SIZE_BYTES = 3000000; // 3MB

  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public PicShareReaderGCUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }

    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    System.out.println("rmiregistry started");
    Thread.sleep(2000);

    //
    // Start barrier server
    //
    barrier = new BarrierServer(BARRIER_PORT, 2, 100);
    barrier.start();

    //
    // Set up config files
    //
    READER_NODE_ID = PicShareConfig.LOWEST_READER_NODE_ID;
    WRITER_NODE_ID = PicShareConfig.LOWEST_WRITER_NODE_ID;
    PicShareConfig.createConfigFile(CONFIG_PICSHARE_PATH + READER_NODE_ID, 
                                    true,
                                    "PicShareReaderGCUnitAlbum",
                                    "847584fakekey",
                                    "IGNORE-THIS-PATH");
    PicShareConfig.createConfigFile(CONFIG_PICSHARE_PATH + WRITER_NODE_ID, 
                                    false,
                                    "PicShareReaderGCUnitAlbum",
                                    "847584fakekey",
                                    "IGNORE-THIS-PATH");
    PicShareConfig dummyPsc;
    dummyPsc = new PicShareConfig(CONFIG_PICSHARE_PATH + WRITER_NODE_ID, 
                                  new NodeId(WRITER_NODE_ID));
    makeDummyConfig(CONFIG_PATH, dummyPsc);
    makeDummyP2Config(CONFIG_P2_PATH);

    //
    // Start helper process
    //
    startHelper("0");

  }

 /** 
 *  Start helper process. Set global writer to refer to helper process.
 
 **/ 
  private void startHelper(String arg){
    //
    // Start the writer PRACTI node
    // "make foo.unit" runs "java foo" with appropriate arguments
    //
    // Use "make" to figure out what arguments needed to
    // set up class paths, etc. Grab the command line
    // rather than just run make b/c we want a handle
    // on the helper process so we can kill it later.
    //
    try{
      if(arg.equals("0")){
        writer = Runtime.getRuntime().exec("make PicShareReaderGCUnitHelper.unit0");
      }
      else{
        // Makefile doesn't know about other arguments
        assert(false);
      }
     
      System.out.println("writer " + arg + " started");
      setUpHelperIO(writer, "arg");
    }
    catch(Exception ex){
      ex.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  Wait for the helper process to create the files
 
 *  I want to read. Spawn "/dev/null" threads to
 
 *  consume itstheir output so it doesn't block.
 
 **/ 
  private void setUpHelperIO(Process helper, String tag){
    InputStream is = helper.getInputStream();
    Thread dn = new DvNull(is, "PicShareReaderGCUnitHelper " + tag + " stdout", 
                           verbose, vverbose); 
    dn.start();
    is = writer.getErrorStream();
    dn = new DvNull(is, "PicShareReaderGCUnitHelper " + tag + " stderr", 
                    verbose, vverbose);
    dn.start();
  }



 /** 
 *  Kill helper process and kill rmiregistry
 
 **/ 
  protected void tearDown() throws Exception{
    if(writer != null){
      writer.destroy();
    }
    System.out.println("writer terminated");
    rmiregistry.destroy();
    System.out.println("rmiregistry terminated");
    super.tearDown();
  }


  /* 
   * The PicShareReaderGCUnitHelper wrote some stuff
   * into PRACTI. Read it back.
   */
  protected static String CONFIG_PATH = "test" + File.separatorChar 
  + "tmp.PicShareReaderGCUnit.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar 
  + "tmp.PicShareReaderGCUnit.p2config";
  protected static String CONFIG_PICSHARE_PATH = "test" + File.separatorChar 
  + "tmp.PicShareReaderGCUnit.picShareConfig";
  protected long READER_NODE_ID;
  protected long WRITER_NODE_ID;
  

 /** 
 *  start helper and wait for writing to complete
 
 *  read files
 
 *  signal helper to die
 
 **/ 
  public void testRead(){

    BarrierClient bc = new BarrierClient("127.0.0.1", BARRIER_PORT, 0);
    if(verbose){
      System.out.println("barrier 1");
    }
    bc.sendBarrierRequest(0, 0); //1 Wait for helper up and done writing
    if(verbose){
      System.out.println("writer done writing");
    }

    NodeId myNodeId = new NodeId(READER_NODE_ID);
    if(vverbose){
      System.out.println("READER ID: " + myNodeId.toString());
    }
    PicShareReader psr = new PicShareReader(CONFIG_PATH, CONFIG_P2_PATH, 
					    CONFIG_PICSHARE_PATH + READER_NODE_ID, 
                                            myNodeId, true);

    if(verbose){
      System.out.println("barrier 2");
    }
    bc.sendBarrierRequest(0, 0); //  2Signal helper to write more

    try{
      PicShareConfig psc = new PicShareConfig(CONFIG_PICSHARE_PATH + READER_NODE_ID, myNodeId);
      LocalInterface li = psr.getLocalInterface();


      readPass(0, psc, li, psr);
      readPass(1, psc, li, psr);

      if(verbose){
        System.out.println("barrier 3");
      }
      bc.sendBarrierRequest(0, 0); //3 Tell helper to die
      
      try{
        Thread.sleep(10000);
      }
      catch(InterruptedException z){
      }
      
    }
    finally{
      //
      // Make sure we release all of the ports so that next
      // unit test can run.
      //
      psr.shutdown();
    }
  }


 /** 
 *  First pass -- expect everything to be demand readed.
 
 *  Second pass -- expect everything to be prefetched.
 
 *  Also on second pass -- verify that stuff outside of interest set
 
 *  is imprecise.
 
 **/ 
  private void readPass(int pass, PicShareConfig psc, LocalInterface li,
                        PicShareReader psr){
    int ii;
    int successCount = 0;
    int jj = 0;
    int dots = 10;

    try{
      Hashtable t = PicShareReaderGCUnitHelper.makeStuffToWrite(psc, pass);
      Enumeration i = t.keys();

        
      while(i.hasMoreElements()){
        ObjId oid = (ObjId)i.nextElement();
        byte b[] = (byte[])t.get(oid);
        BodyMsg bm = null;
        int retries = 0;
        int MAX_RETRIES = 50;
        
        Assert.affirm(b != null);
        if(vverbose){
          System.out.print("Checking " + oid.toString() + "...");
        }
        while(bm == null){
          try{
            bm = li.read(oid, 0, b.length, true, true);
            jj++;
            if(jj % dots == 0){
              System.out.print(".");
            }
            assert(bm.getLength() == b.length);
            ImmutableBytes ib = bm.getBody();
            byte b2[] = ib.getCopyBytes();
            for(ii = 0; ii < b.length; ii++){
              Assert.affirm(b[(int)ii] == b2[(int)ii]);
            }
            successCount++;
            if(vverbose){
              System.out.println("OK (" + successCount + ")");
            }
          }
          catch(ObjNotFoundException onfe){
            if(retries < MAX_RETRIES){
              if(vverbose){
                System.out.println("testRead: " + oid + " not there yet. Retry");
              }
              try{
                Thread.sleep(1000);
              }
              catch(InterruptedException ie){
              }
              retries++;
              // retry again
            }
            else{
              System.out.println("Object not found: OID = " + oid
                                 + " successes so far: " + successCount);
              throw onfe;
            }
          }
        }
      }
    }
    catch(ObjNotFoundException n){
      n.printStackTrace();
      assert(false);
    }
    catch(ReadOfHoleException rhe){
      rhe.printStackTrace();
      assert false;
    }
    catch(IOException e){
      assert(false);
    }
    catch(ReadOfInvalidRangeException roire){
      assert(false);
    }
  }



  /*
   * Just make a 2-node config file on localhost in the 
   * specified path
   */
  private void  makeDummyConfig(String path, PicShareConfig psc){
    Config.createEmptyConfig();
    NodeId id = new NodeId(WRITER_NODE_ID);
    Config.addOneNodeConfig(id, "localhost", 6378, 6379, 6380, 6381, 6382,
			    "test" + File.separatorChar + "tmp.PicShareWriterUnitGCHelper.db",
			    "/*",
			    -1L,
			    "localhost.cs.utexas.edu",
			    6383,
			    6384,
			    -1, 
                            CACHE_SIZE_BYTES,
                            MAX_WRITER_LOG_DISK_SIZE_BYTES,
                            MAX_WRITER_LOG_MEM_SIZE_BYTES);
    id = new NodeId(READER_NODE_ID);
    Config.addOneNodeConfig(id, "localhost", 6478, 6479, 6480, 6481, 6482,
			    "test" + File.separatorChar + "tmp.PicShareWriterGCUnit.db",
			    psc.getReaderSubscriptionTargets().toString() /* "/*" */,
			    -1L,
			    "localhost.cs.utexas.edu",
			    6483,
			    6484,
			    -1, 
                            CACHE_SIZE_BYTES,
                            MAX_READER_LOG_DISK_SIZE_BYTES,
                            MAX_READER_LOG_MEM_SIZE_BYTES);
    Config.writeToFile(path);
  }

  private void makeDummyP2Config(String path){
    P2Config.createEmptyConfig();
    P2Config.addOneNodeConfig(new NodeId(WRITER_NODE_ID), 10, 10, 
                              5000, 5000, 3, 3, 3, 3, 3);
    P2Config.addOneNodeConfig(new NodeId(READER_NODE_ID), 10, 10, 
                              5000, 5000, 3, 3, 3, 3, 3);
    P2Config.writeToFile(path);
  }

  
  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(PicShareReaderGCUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   PicShareReaderGCUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "PicShareReaderGCUnit";
    System.err.print(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
          verbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new PicShareReaderGCUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);
    
    System.exit(0);
  }

}




