 /** 
/* PicShareReaderGCUnitHelper.java.java
 * 
 * Helper for PicShareReaderGCUnit -- when invoked by that code, start
 * a PicShareWriterProcess that writes a bunch of known files.
 * The trick here is that the memory and disk logs are short enough
 * to force garbage collection and checkpoint transfer.
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.util.*;
import java.io.*;

public class PicShareReaderGCUnitHelper {

  PicShareWriter psw;
  LocalInterface li;
  private static final boolean verbose = true;
  private static final boolean vverbose = false;
  PicShareConfig psc;

  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public PicShareReaderGCUnitHelper(boolean cleanDisk) throws Exception{
    // Assume first writer ID is me in this test
    NodeId myId = new NodeId(PicShareConfig.LOWEST_WRITER_NODE_ID);
    //PicShareConfig depends on Config being initialized
    String pscConfigPath = PicShareReaderGCUnit.CONFIG_PICSHARE_PATH
      + PicShareConfig.LOWEST_WRITER_NODE_ID;
    Config.readConfig(PicShareReaderGCUnit.CONFIG_PATH);
    psc = new PicShareConfig(pscConfigPath, myId);
    if(verbose){
      System.out.println("HELPER ID: " + myId.toString());
    }
    try{
      psw = new PicShareWriter(PicShareReaderGCUnit.CONFIG_PATH, 
                               PicShareReaderGCUnit.CONFIG_P2_PATH, 
                               pscConfigPath,
                               myId, 
                               cleanDisk);
      li = psw.getLocalInterface();
    }
    catch(Exception e){
      e.printStackTrace();
      System.out.println("*** Did you kill the previous PicShareReaderGCUnitHelper? ***");
      throw e;
    }
  }



 /** 
 *  Write specified list of objId->data tuples to PRACTI
 
 **/ 
  private void writeStuff(int pass){
    Hashtable stuffToWrite = makeStuffToWrite(psc, pass);
    Enumeration i = stuffToWrite.keys();
    int dots = 100;
    int ii = 0;
    while(i.hasMoreElements()){
      ii++;
      if(dots % ii == 0){
        System.out.print(".");
      }
      ObjId oid = (ObjId)i.nextElement();
      byte b[] = (byte[])stuffToWrite.get(oid);
      Assert.affirm(b != null);
      try{
        if(verbose){
          System.out.println("Writing " + oid);
        }
        long got = li.write(oid, 0, b.length, b, false);
        Assert.affirm(got == b.length);
      }
      catch(IOException e){
        System.out.println("IO Exception in PicShareReaderGCUnitHelper ("
                           + oid.toString()
                           + ") -- "
                           + e.toString());
      }
    }
    //
    // Since writes are unbound, need to ensure they are in checkpoint before
    // we die. The bodies will not be in the log (lets us test demand read,
    // prefetching.)
    //
    System.out.println("Going to sync stat to disk");
    li.syncStateToDisk();
    System.out.println("disk synced");
  }

 /** 
 *  Return a list of stuff we are supposed to write
 
 *  Hashtable is path->data pairs
 
 *  Used by PicShareReaderGCUnit to find out what it is supposed
 
 *  to read.
 
 **/ 
  public static Hashtable makeStuffToWrite(PicShareConfig psc, int pass){
    long ii;
    int jj;
    long logSize = PicShareReaderGCUnit.MAX_WRITER_LOG_DISK_SIZE_BYTES;
    int perWriteLength = 10000;
    long writesToFillLog = logSize/perWriteLength + 1;
    long overage = 2;
    long nWrites = overage * writesToFillLog;
    assert(nWrites * perWriteLength >= logSize * overage);

    Hashtable t = new Hashtable();
    for(ii = 0; ii < nWrites; ii++){
      byte b[] = new byte[perWriteLength];
      for(jj = 0; jj < perWriteLength; jj++){
	b[jj] = (byte)(ii + jj);
      }
      String suffix = (new Long(ii)).toString()
        + "-"
        + (new Integer(pass).toString());
      ObjId o = new ObjId(psc.basePRACTIDataPath() + suffix);
      t.put(o, b);
    }
    Assert.affirm(t.size() == nWrites);
    return t;
  }


 /** 
 *  Check to make sure the stuff we wrote exists.
 
 **/ 
  public void checkRead(Hashtable t)
    throws ObjNotFoundException, IOException, ReadOfInvalidRangeException, 
	   ReadOfHoleException{
    int ii;
    Enumeration i = t.keys();
    int successCount = 0;
    while(i.hasMoreElements()){
      ObjId oid = (ObjId)i.nextElement();
      byte b[] = (byte[])t.get(oid);
      BodyMsg bm = null;
      Assert.affirm(b != null);
      if(vverbose){
        System.out.println("Checking " + oid.toString() + "...");
      }
      try{
        bm = li.read(oid, 0, b.length, false, true);
        assert(bm.getLength() == b.length);
        ImmutableBytes ib = bm.getBody();
        byte b2[] = ib.getCopyBytes();
        for(ii = 0; ii < b.length; ii++){
          Assert.affirm(b[(int)ii] == b2[(int)ii]);
        }
        successCount++;
      }
      catch(ObjNotFoundException onfe){
        System.out.println("Object not found: OID = " + oid
                           + " successes so far: " + successCount);
        throw onfe;
      }
      catch(IOException ie){
        System.out.println("IO Exception: OID = " + oid
                           + " successes so far: " + successCount);
        throw ie;
      }
      catch(ReadOfInvalidRangeException roire){
        System.out.println("Read invalid range exception: OID = " + oid
                           + " successes so far: " + successCount);
        throw roire;
      }
      catch(ReadOfHoleException rhe){
	System.out.println("Read hole: OID = " + oid
                           + " successes so far: " + successCount);
	throw rhe;
      }

    }
  }

  public void shutdown(){
    if(psw != null){
      psw.shutdown();
      psw = null;
    }
  }

 /** 
 *  Write the specified PRACTI files. Then wait until reading done. Then write
 
 *  more stuff. Then die. When restarted, read the old stuff and write some 
 
 *  different stuff.
 
 **/ 
  public static void main(String s[]) {
    if(s[0].equals("0")){
      doFirst();
    }
    else{
      assert(false);
    }
  }

 /** 
 *  First time around -- write a bunch of stuff.
 
 **/ 
  private static void doFirst(){
    BarrierClient c = new BarrierClient("127.0.0.1", PicShareReaderGCUnit.BARRIER_PORT, 1);
    PicShareReaderGCUnitHelper p = null;
    if(verbose){
      System.out.println("HELLO from PicShareReaderGCUnitHelper 0");
    }
    try{
      p = new PicShareReaderGCUnitHelper(true);
      if(verbose){
        System.out.println("PicShareReaderGCUnitHelper constructor done");
      }
      p.writeStuff(0);
    }
    catch(Exception e){
      e.printStackTrace();
    }

    try{
      if(verbose){
	System.out.println("barrier 1");
      }
      c.sendBarrierRequest(0, 1); // Tell ReaderUnit that we are up and running
      
      //
      // Write second batch of stuff (concurrent with remote reading to make life fun)
      //
      if(p != null){
	p.writeStuff(1);
      }
      
      if(verbose){
	System.out.println("barrier 2");
      }
      c.sendBarrierRequest(0, 1); // Wait for ReaderUnit to be running, set up subscriptions, etc.
      
      
      if(verbose){
	System.out.println("barrier 3");
      }
      c.sendBarrierRequest(0, 1); // Die.
      
      
      //if(verbose){
	System.out.println("PicShareReaderGCUnitHelper exiting");
	//}
    } finally{
      //
      // Make sure we release all of the ports so that next
      // unit test can run.
      //
      p.shutdown();
      System.exit(0);
    }
  }

}
