 /** 
/* PicShareReaderPolicy.java
 * 
 *  (1) Subscribe to all writers for 
 *        PicShare.BaseObjId()/data/*
 *        PicShare.BaseObjId()/config/status
 *
 *    resubscribe if failure (assumption is that there is a reasonable
 *        number of worker threads to cycle through the possible options
 *        at some reasonable rate)
 * 
 *  (2) send demand reads to original writer; 
 *    TBD: if that
 *        fails, try sending to all writers;
 *        if that fails, give up (local interface
 *        needs a timeout and retry strategy)
 *
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.io.*;
import java.util.List;
import java.util.Iterator;
import java.util.Hashtable;

public class PicShareReaderPolicy extends Policy{

  //
  // Immutable local state
  //
  private P2Runtime runtime;
  private PicShareConfig psconfig;
  private static final boolean dbg = false;

  //
  // Hashtable has its own lock
  //
  private Hashtable pendingReads;
  
  
  public
  PicShareReaderPolicy(PicShareConfig psc){
      psconfig = psc;
      pendingReads = new Hashtable();
      demandReads = new DbgCounter();
  }

  public void setRuntime(P2Runtime runtime_){    
    this.runtime = runtime_;
    makeSubscriptions();
  }

  private void makeSubscriptions() {
    SubscriptionSet ss = psconfig.getReaderSubscriptionTargets();
    List w = psconfig.getWriters();
    Iterator iw = w.iterator();
    while(iw.hasNext()){
      NodeId id = (NodeId)iw.next();
      Env.dprintln(dbg, "PicShareReaderPolicy::makeSubscriptions for " + ss.toString());
      runtime.addInvalSubscription(id, ss, false);
      runtime.addBodySubscription(id, ss);
    }
  }

 /** 
 *  starts the policy layer 
 **/ 
  public void start(){
  }

 /** 
 *  shuts down the policy layer 
 **/ 
  public void shutdown(){
  }

 /** 
 *  interface to interact with safety module 
 **/ 

  // sendMsg: called by safety to send a msg to liveness policy
  public void receiveMsg(String[] msg){
  }

 /** 
 *  information of practi events 
 **/ 

  public void informLocalReadInvalid(ObjId objId, long offset, long length, 
				     AcceptStamp inval){
    Env.dprintln(dbg, "informLocalReadInvalid:"+objId+":"+offset+":"+length);
    Object prev;

    // first send the request to the original writer
    ObjInvalTarget t = new ObjInvalTarget(objId, offset, length);
    prev = pendingReads.put(t, inval.getNodeId());
    //
    // If there was no outstanding read request for this object,
    // then generate one. Note that DataStore may generate
    // multiple events per miss, so this lets us filter
    // out duplicates while stile ensuring that if the demand read
    // reply that we eventually get is not good enough, we will
    // eventually re-issue the demand read.
    //
    if(prev == null){
      demandReads.incrementCount();
      Env.dprintln(dbg, "informLocalReadInvalid...going to demandRead " 
                   + objId+":"+offset+":"+length);
      runtime.demandRead(inval.getNodeId(), objId, offset, length, new AcceptStamp(-1, new NodeId(0)));
    }
    else{
      Env.dprintln(dbg, "informLocalReadInvalid...ignore duplicate to outstanding demandRead of" 
                   + objId+":"+offset+":"+length);
    }
  }

  public void informLocalReadImprecise(ObjId objId, long offset,
				       long length){
    // I originally thought this couldn't happen, but in new
    // connection architecture, the connection start event
    // is essentially an imprecise invalidation on *
    // Assert.affirm(false); -- THIS CAN HAPPEN!
    //

    Env.dprintln(dbg, "InformLocalReadImprecise"+objId+":"+offset+":"+length);
  }

  public void informLocalWrite(ObjId objId, long offset, long length, 
			       AcceptStamp as,
			       boolean isBound, boolean isEmbargoed){
    Env.dprintln(dbg, "informLocalWrite"+objId+":"+offset+":"+length);
  }

  public void informLocalDelete(ObjId objId){
    Env.dprintln(dbg, "informLocalDelete"+objId);
  }

  public void informReceiveInval(NodeId senderNodeId, ObjId objId, 
				 long offset, long length,
				 AcceptStamp as, boolean isBound, 
				 boolean isEmbargoed){
    Env.dprintln(dbg, "informReceiveInval"+objId+":"+offset+":"+length
                 +"@"+as.toString());
  }


  public void informDemandReadSuccess(NodeId senderNodeId, 
				      ObjId objId, long offset,
				      long length,
				      AcceptStamp as) {
      Env.dprintln(dbg, "informDemandReadSuccess"+senderNodeId+":"+objId+":"+offset+":"+length 
		   + ":" + as.toString());
 
    ObjInvalTarget t = new ObjInvalTarget(objId, offset, length);
    pendingReads.remove(t);
  }
    

  public void informDemandReadFailedMiss(NodeId senderNodeId, 
					 ObjId objId, 
					 long offset, 
					 long length,
					 AcceptStamp as) {
      Env.dprintln(dbg, "informDemandReadFailedMiss"+senderNodeId+":"+objId
		   +":"+offset+":"+length + ":" + as.toString());
    handleReadMiss(senderNodeId, objId, offset, length);
  }
 
  public void informDemandReadFailedMaxRetries(NodeId senderNodeId,
					       ObjId objId, 
					       long offset,
					       long length,
					       AcceptStamp as) {
    Env.dprintln(dbg, "informDemandReadFailedMaxRetries"
		 +senderNodeId+":"+objId+":"+offset+":"+length + ":" + as.toString());
    handleReadMiss(senderNodeId, objId, offset, length);
  }


  //
  // If this read was to original writer, retry to the other
  // writers (maybe they have the data?). If this was to a different
  // writer, then this was a retry -- give up. (LocalInterface
  // should timeout and retry.)
  //
  private void handleReadMiss(NodeId senderNodeId, ObjId objId, 
			      long offset, long length) {
    ObjInvalTarget t = new ObjInvalTarget(objId, offset, length);
    NodeId writer = (NodeId)pendingReads.remove(t); // Only retry once
    if(writer != null && senderNodeId.equals(writer)){
      List w = psconfig.getWriters();
      Iterator iw = w.iterator();
      while(iw.hasNext()){
        NodeId id = (NodeId)iw.next();
	if(!id.equals(writer)){
	    runtime.demandRead(id, objId, offset, length, new AcceptStamp(-1, new NodeId(0)));	 
	}
      }
    }
  }

  public void recvSyncReply(NodeId senderNodeId, AcceptStamp acceptStamp) {
    Env.dprintln(dbg, "recvSyncReply "+senderNodeId+":"+acceptStamp);
  }

 /** 
 *  Subscription add events 
 **/ 

  public void informAttachedInvalSubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informAttachedInvalSubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

  public void informAddedInvalSubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "InformAddedInvalSubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

  public void informAddedBodySubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informAddedBodySubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

  public void informAddedOutgoingInvalSubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informAddedOutgoingInvalSubscription "+senderNodeId+":"+receiverNodeId+":"+ss);
  }

  public void informAddedOutgoingBodySubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "InformAddedOutgoingBodySubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

 /** 
 *  Subscription removed events 
 **/ 

  public void informRemovedInvalSubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informRemovedInvalSubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
    runtime.addInvalSubscription(senderNodeId, ss, false);
  }

  public void informRemovedBodySubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informRemovedBodySbuscription"+senderNodeId+":"+receiverNodeId+":"+ss);
    assert(ss != null);
    if(ss.isEmpty()){
      //
      // Currently we always get null -- this is a TBD
      // on RuntimeController. Workaround is to
      // resubscribe to everything...
      //
      ss = psconfig.getReaderSubscriptionTargets();
    }
    runtime.addBodySubscription(senderNodeId, ss);
  }
  
  public void informRemovedOutgoingInvalSubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informRemovedOutgoingInvalSubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

  public void informRemovedOutgoingBodySubscription(NodeId senderNodeId, 
                                            NodeId receiverNodeId, 
                                            SubscriptionSet ss){
    Env.dprintln(dbg, "informRemovedOutgiongBodySubscription"+senderNodeId+":"+receiverNodeId+":"+ss);
  }

 /** 
 *  Subscription failed   
 *   -- the following methods implement the same logic as above 
 **/ 
 
  public void informAddInvalSubscriptionFailed(NodeId senderNodeId, 
                                               NodeId receiverNodeId, 
                                               SubscriptionSet ss){
    Env.dprintln(dbg, "informAddInvalSubscriptionFailed"+senderNodeId+":"+receiverNodeId+":"+ss);
    runtime.addInvalSubscription(senderNodeId, ss, false);
  }

  public void informAddBodySubscriptionFailed(NodeId senderNodeId, 
                                              NodeId receiverNodeId, 
                                              SubscriptionSet ss){
   Env.dprintln(dbg, "informAddBodySubscriptionFailed"+senderNodeId+":"+receiverNodeId+":"+ss);
    assert(ss != null);
    if(ss.isEmpty()){
      //
      // Currently we always get null -- this is a TBD
      // on RuntimeController. Workaround is to
      // resubscribe to everything...
      //
      ss = psconfig.getReaderSubscriptionTargets();
    }
    runtime.addBodySubscription(senderNodeId, ss);
  }
  



 /** 
 *  To support unit tests, include a count of demand reads. 
 **/ 
  //
  // Counter has its own lock 
  //
  private DbgCounter demandReads;
  public long getDemandReadCount(){
    return demandReads.getCount();
  }
  
  public void resetDemandReadCount(){
    demandReads.resetCount();
  }

  private class DbgCounter{
    private long count;
    DbgCounter(){
      count = 0;
    }
    synchronized long getCount(){
      return count;
    }
    synchronized void incrementCount(){
      count++;
    }

    synchronized void resetCount(){
      count = 0;
    }
  }
  
}
 /** 
/* $Log: PicShareReaderPolicy.java,v $
/* Revision 1.15  2007/11/28 08:11:34  nalini
/* safety policy module and example checked in
/*
/* Revision 1.14  2007/06/07 20:48:59  dahlin
/* Writer now scans and writes...
/*
/* Revision 1.13  2007/06/07 17:55:29  dahlin
/* Writer now scans and writes...
/*
/* Revision 1.12  2007/06/05 20:49:24  nalini
/* exposed LOG|CP option for invalSubscriptions to P2Runtime and Overlog layer
/*
/* Revision 1.11  2007/05/29 14:21:09  nalini
/* updated unit test to reflect new body subscription semantics
/*
/* Revision 1.10  2007/04/05 21:49:26  nalini
/* trying to get subscribeBWUnit to work
/*
/* Revision 1.9  2007/04/02 22:10:05  nalini
/* junit fixes
/*
/* Revision 1.8  2007/03/09 21:46:09  nalini
/* added 2 events: informAddInvalSubscriptionFailed & informAddBodySubscriptionFailed
/*
*/
 **/ 
