//package utils.junit;

import junit.textui.TestRunner;
import junit.framework.*;
import java.util.*;
import java.io.*;

/**
 * TBD: Update class name
 */
public class PicShareReaderUnit extends TestCase {
  public static final String TEST_ALL_TEST_TYPE = "UNIT";
  protected static boolean verbose = false; // Start/end of test
  protected static boolean vverbose = false; // Test internals
  private Process rmiregistry;
  private Process writer;
  private BarrierServer barrier;
  public static final int BARRIER_PORT = 9000;

  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public PicShareReaderUnit (final String s) {
    super (s);
  }

  /*
   * Fixtures are run before and after each test case
   * to set up environment in which tests must run.
   */
  protected void setUp() throws Exception{
    super.setUp();
    /* 
     * TBD: Insert other fixture code here
     * e.g., this.objectUnderTest = new ClassToBeTested(...);
     */
    try{
      Process p = Runtime.getRuntime().exec("./killRMIRegistry.sh");
      p.waitFor();
    }
    catch(Exception e){
      // Non-fatal exception; we just killed it to 
      // ensure we could start it. Now try starting it.
    }

    //
    // Start the registry
    //
    rmiregistry = Runtime.getRuntime().exec("rmiregistry");
    System.out.println("rmiregistry started");
    Thread.sleep(2000);

    //
    // Start barrier server
    //
    barrier = new BarrierServer(BARRIER_PORT, 2, 100);
    barrier.start();

    //
    // Set up config files
    //
    READER_NODE_ID = PicShareConfig.LOWEST_READER_NODE_ID;
    WRITER_NODE_ID = PicShareConfig.LOWEST_WRITER_NODE_ID;
    String pscConfigPath = CONFIG_PICSHARE_PATH + WRITER_NODE_ID;
    makeDummyPicShareConfig(pscConfigPath, false, "IGNORE-THIS-PATH");
    pscConfigPath = CONFIG_PICSHARE_PATH + READER_NODE_ID;
    makeDummyPicShareConfig(pscConfigPath, true, "IGNORE-THIS-PATH");
    PicShareConfig psc = new PicShareConfig(pscConfigPath, new NodeId(READER_NODE_ID));
    makeDummyConfig(CONFIG_PATH, psc);
    makeDummyP2Config(CONFIG_P2_PATH);

    //
    // Start helper process
    //
    startHelper("0");

  }

 /** 
 *  Start helper process. Set global writer to refer to helper process.
 
 **/ 
  private void startHelper(String arg){
    //
    // Start the writer PRACTI node
    // "make foo.unit" runs "java foo" with appropriate arguments
    //
    // Use "make" to figure out what arguments needed to
    // set up class paths, etc. Grab the command line
    // rather than just run make b/c we want a handle
    // on the helper process so we can kill it later.
    //
    try{
      if(arg.equals("0")){
        writer = Runtime.getRuntime().exec("make PicShareReaderUnitHelper.unit0");
      }
      else if(arg.equals("1")){
        writer = Runtime.getRuntime().exec("make PicShareReaderUnitHelper.unit1");
      }
      else{
        // Makefile doesn't know about other arguments
        assert(false);
      }
     
      System.out.println("writer " + arg + " started");
      setUpHelperIO(writer, arg);
    }
    catch(Exception ex){
      ex.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  Wait for the helper process to create the files
 
 *  I want to read. Spawn "/dev/null" threads to
 
 *  consume itstheir output so it doesn't block.
 
 **/ 
  private void setUpHelperIO(Process helper, String tag){
    InputStream is = helper.getInputStream();
    Thread dn = new DvNull(is, "PicShareReaderUnitHelper " + tag + " stdout", 
                           verbose, vverbose);
    dn.start();
    is = writer.getErrorStream();
    dn = new DvNull(is, "PicShareReaderUnitHelper " + tag + " stderr",
                    verbose, vverbose);
    dn.start();
  }



 /** 
 *  Kill helper process and kill rmiregistry
 
 **/ 
  protected void tearDown() throws Exception{
    if(writer != null){
      writer.destroy();
    }
    System.out.println("writer terminated");
    rmiregistry.destroy();
    System.out.println("rmiregistry terminated");
    super.tearDown();
  }


 /** 
 *  Wait for the objects listed in t to be prefetched.
 
 **/ 
  protected void waitForPrefetching(int pass, PicShareConfig psc,
                                    LocalInterface li){
    Hashtable t = PicShareReaderUnitHelper.makeStuffToWrite(psc, pass);
    Enumeration i = t.keys();
    try{
      while(i.hasMoreElements()){
        ObjId oid = (ObjId)i.nextElement();
        int retries = 0;
        int MAX_RETRIES = 20;
        boolean valid = false;
        
        if(vverbose){
          System.out.print("Wait for prefetch " + oid.toString() + "...");
        }
        while(!valid){
          try{
            valid = li.isValid(oid, 0, 1);
          }
          catch(ObjNotFoundException onfe){
            if(retries < MAX_RETRIES){
              if(vverbose){
                System.out.println("waitForPrefetch: " + oid + " not there yet. Retry");
              }
              try{
                Thread.sleep(1000);
              }
              catch(InterruptedException ie){
              }
              retries++;
              // retry again
            }
            else{
              System.out.println("Object not found: OID = " + oid);
              throw onfe;
            }
          }
        }
      }
    }
    catch(ObjNotFoundException n){
      n.printStackTrace();
      assert(false);
    }
    catch(ReadOfHoleException rhe){
      rhe.printStackTrace();
      assert false;
    }
    catch(IOException e){
      assert(false);
    }
  }


  /* 
   * The PicShareReaderUnitHelper wrote some stuff
   * into PRACTI. Read it back.
   */
  protected static String CONFIG_PATH = "test" + File.separatorChar 
  + "tmp.PicShareReaderUnit.config";
  protected static String CONFIG_P2_PATH = "test" + File.separatorChar 
  + "tmp.PicShareReaderUnit.p2config";
  protected static String CONFIG_PICSHARE_PATH = "test" + File.separatorChar 
  + "tmp.PicShareReaderUnit.picShareConfig";
  protected long READER_NODE_ID;
  protected long WRITER_NODE_ID;
  

 /** 
 *  start helper and wait for writing to complete
 
 *  read first buch of files
 
 *  signal helper to write next batch
 
 *  prefetch this batch
 
 *  read second bunch of files
 
 **/ 
  public void testRead(){
    int pass;


    BarrierClient bc = new BarrierClient("127.0.0.1", BARRIER_PORT, 0);
    if(verbose){
      System.out.println("barrier 1");
    }
    bc.sendBarrierRequest(0, 0); // Wait for helper up and running
    if(verbose){
      System.out.println("writer done writing");
    }

    NodeId myNodeId = new NodeId(READER_NODE_ID);
    if(vverbose){
      System.out.println("READER ID: " + myNodeId.toString());
    }
    assert(myNodeId.equals(new NodeId(READER_NODE_ID)));
    PicShareReader psr = new PicShareReader(CONFIG_PATH, CONFIG_P2_PATH, 
					    CONFIG_PICSHARE_PATH + myNodeId, 
                                            myNodeId, true);
    try{
      PicShareConfig psc = new PicShareConfig(CONFIG_PICSHARE_PATH + READER_NODE_ID, myNodeId);
      LocalInterface li = psr.getLocalInterface();


      readPass(0, psc, li, psr);

      if(verbose){
        System.out.println("barrier 2");
      }
      bc.sendBarrierRequest(0, 0); // Signal helper to write more

      waitForPrefetching(1, psc, li);

      readPass(1, psc, li, psr);

      if(verbose){
        System.out.println("barrier 3");
      }
      bc.sendBarrierRequest(0, 0); // Tell helper to die
      try{
        Thread.sleep(5000);
      }
      catch(InterruptedException z){
      }

      startHelper("1");
      bc.sendBarrierRequest(0, 0); // Wait for helper to be back up
      readPass(3, psc, li, psr);
      bc.sendBarrierRequest(0, 0); // Tell helper to die
      if(verbose){
	System.err.println("\n---------- pass all read check ------\n");
      }
      try{
        Thread.sleep(2000);
      }
      catch(InterruptedException zz){
      }
    }
    finally{
      //
      // Make sure we release all of the ports so that next
      // unit test can run.
      //
      psr.shutdown();
    }
  }


 /** 
 *  First pass -- expect everything to be sent across if you 
 
 *                wait long enough.  (But there may be some demand
 
 *                reads if body subscription lags)
 
 *  Second pass -- expect everything to be prefetched.
 
 *  Also on second pass -- verify that stuff outside of interest set
 
 *  is imprecise.
 
 **/ 
  private void readPass(int pass, PicShareConfig psc, LocalInterface li,
                        PicShareReader psr){
    int ii;
    int successCount = 0;

    try{
      Hashtable t = PicShareReaderUnitHelper.makeStuffToWrite(psc, pass);
      Enumeration i = t.keys();
      
      if(pass == 0) {
	try{
	  // hopefully 5 sec should be enough for body subscriptions to 
	  // send all previous bodies across
	  Thread.sleep(5000); 
	}catch(InterruptedException e){
	}
      }
        
      while(i.hasMoreElements()){
        ObjId oid = (ObjId)i.nextElement();
        byte b[] = (byte[])t.get(oid);
        BodyMsg bm = null;
        int retries = 0;
        int MAX_RETRIES = 20;
        
        Assert.affirm(b != null);
        if(vverbose){
          System.out.print("Checking " + oid.toString() + "...");
        }
        while(bm == null){
          try{
            bm = li.read(oid, 0, b.length, true, true);
            assert(bm.getLength() == b.length);
            ImmutableBytes ib = bm.getBody();
            byte b2[] = ib.getCopyBytes();
            for(ii = 0; ii < b.length; ii++){
              Assert.affirm(b[(int)ii] == b2[(int)ii]);
            }
            successCount++;
            if(vverbose){
              System.out.println("OK (" + successCount + ")");
            }
          }
          catch(ObjNotFoundException onfe){
            if(retries < MAX_RETRIES){
              if(vverbose){
                System.out.println("testRead: " + oid + " not there yet. Retry");
              }
              try{
                Thread.sleep(1000);
              }
              catch(InterruptedException ie){
              }
              retries++;
              // retry again
            }
            else{
              System.out.println("Object not found: OID = " + oid
                                 + " successes so far: " + successCount);
              throw onfe;
            }
	  }
        }
      }
      //
      // Expect zero demand reads on first pass (but depends on timing)
      // and zero on second  pass. 
      //
      if(pass == 0){
        if(verbose){
          System.out.println("First pass done -- " + successCount
                             + " reads (" 
                             + psr.dbgGetPolicy().getDemandReadCount()
                             + " demand)");
        }

        assert(psr.dbgGetPolicy().getDemandReadCount() == 0):"Not enough time to transfe all bodies";
        //assert(psr.dbgGetPolicy().getDemandReadCount() < t.size());
	psr.dbgGetPolicy().resetDemandReadCount();
      }
      if(pass == 1){ // No change expected.
        if(verbose){
          System.out.println("Second pass done -- " + successCount
                             + " reads (" 
                             + psr.dbgGetPolicy().getDemandReadCount()
                             + " demand)");
        }
        assert(psr.dbgGetPolicy().getDemandReadCount() == 0);
        tryCheckReadImprecise(li);
      }
    }
    catch(ObjNotFoundException n){
      n.printStackTrace();
      assert(false);
    }
    catch(IOException e){
      assert(false);
    }
    catch(ReadOfInvalidRangeException roire){
      assert(false);
      
    }catch(ReadOfHoleException rhe){
      assert false;
    }
  }


  /*
   * tryCheckReadImprecise() -- read an object that is expected to
   * be imprecise (written by sender but not subscribed).
   */
  protected static String impreciseTestPath = "/test-imprecise-read-path-sd023j234oslfwu230dlc";
  protected static String impreciseTestWrittenObj = "foo/bar/baz";
  protected static String impreciseTestAnotherObj = "foo/bar/ble";
  protected static String impreciseTestStillAnotherObj = "foo/bar";
  protected static String impreciseTestObj4 = "foo/biz/bing/bang/bongo";
  protected static String impreciseTestPath2 = "/test-imprecise-read-path-2-sd023j234oslfwu230dlc";
  private void tryCheckReadImprecise(LocalInterface li){

    if(verbose){
      System.out.print("test tryCheckReadImprecise...");
    }
    ObjId oid = new ObjId(impreciseTestPath + impreciseTestWrittenObj);
    assert(!li.isPrecise(oid));
    oid = new ObjId(impreciseTestPath + impreciseTestAnotherObj);
    assert(!li.isPrecise(oid));
    oid = new ObjId(impreciseTestPath + impreciseTestStillAnotherObj);
    assert(!li.isPrecise(oid));
    oid = new ObjId(impreciseTestPath + impreciseTestObj4);
    assert(!li.isPrecise(oid));
    oid = new ObjId(impreciseTestPath2);
    assert(!li.isPrecise(oid));
    if(verbose){
      System.out.println("...tryCheckReadImprecise SUCCEEDS");
    }
  }


  /*
   * Just make a 2-node config file on localhost in the 
   * specified path
   */
  protected void  makeDummyConfig(String path, PicShareConfig psc){
    Config.createEmptyConfig();
    NodeId id = new NodeId(WRITER_NODE_ID);
    Config.addOneNodeConfig(id, "localhost", 9378, 9379, 9380, 9381, 9382,
			    "test" + File.separatorChar + "tmp.PicShareWriterUnitHelper.db",
			    "/*",
			    -1L,
			    "localhost.cs.utexas.edu",
			    9383,
			    9384,
			    -1, 
			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    id = new NodeId(READER_NODE_ID);
    Config.addOneNodeConfig(id, "localhost", 9478, 9479, 9480, 9481, 9482,
			    "test" + File.separatorChar + "tmp.PicShareWriterUnit.db",
			    psc.getReaderSubscriptionTargets().toString() /* "/*" */,
			    -1L,
			    "localhost.cs.utexas.edu",
			    9483,
			    9484,
			    -1, 
			    Config.CACHE_SIZE_BYTES_DEFAULT,
			    Config.MAX_LOG_DISK_SIZE_BYTES,
			    Config.MAX_LOG_MEM_SIZE_BYTES);
    Config.writeToFile(path);
  }

  private void makeDummyP2Config(String path){
    P2Config.createEmptyConfig();
    P2Config.addOneNodeConfig(new NodeId(WRITER_NODE_ID), 10, 10, 
                              5000, 5000, 3, 3, 3, 3, 3);
    P2Config.addOneNodeConfig(new NodeId(READER_NODE_ID), 10, 10, 
                              5000, 5000, 3, 3, 3, 3, 3);
    P2Config.writeToFile(path);
  }

  private void makeDummyPicShareConfig(String configFilePath, 
                                       boolean reader, 
                                       String dataPath){
    PicShareConfig.createConfigFile(configFilePath, 
                                    reader,
                                    "PicShareReaderUnitAlbum",
				    "847584fakekey",
                                    dataPath);
  }
  
  /*
   * "new TestSuite(Class c)" constructs a test suite
   * containg every method whose name begins with "test"
   * 
   * TBD: update class name
   */
  public static Test suite(){
    TestSuite suite = new TestSuite(PicShareReaderUnit.class);
    return suite;
  }


  /*
   * main() lets us run just this set of unit tests
   * from the comand line (you can also invoke 
   * the testrunner on this class and it will find
   * the suite())
   *
   * usage: java <classname> [-verbose] [-vverbose] [testName]*
   * 
   *   If verbose or vverbose are included, print info to screen
   *
   *   If [testName]* are included, then run test called "test[testName]"
   *   for each such [testName]. E.g., "java TestEmtpy foo" runs
   *   PicShareReaderUnit.testfoo() as a TestCase.
   *
   * TBD: update class name
   */
  public static void main(String s[]) {
    String name = "PicShareReaderUnit";
    System.err.print(name + " self test begins...");
    //
    // Default: run all tests
    //
    TestSuite ste = new TestSuite();
    Test test;
    boolean doAllTests = true;

    if(s.length > 0){
      int ii;
      for(ii = 0; ii < s.length; ii++){
        if(s[ii].equals("-verbose")){
          verbose = true;
        }
        else if(s[ii].equals("-vverbose")){
          verbose = true;
        }
        else{
          doAllTests = false;
          ste.addTest(new PicShareReaderUnit("test" + s[ii]));
        }
        
      }
    }
    if(doAllTests){
      test = suite();
    }
    else{
      test = ste;
    }
    TestRunner tr = new TestRunner();
    tr.doRun(test);

    System.exit(0);
  }

}



