 /** 
/* PicShareReaderUnitHelper.java
 * 
 * Helper for PicShareReaderUnit -- when invoked by that code, start
 * a PicShareWriterProcess that writes a bunch of known files.
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
 **/ 

import java.util.*;
import java.io.*;

public class PicShareReaderUnitHelper {

  PicShareWriter psw;
  LocalInterface li;
  private static final boolean verbose = false;
  private static final boolean vverbose = false;
  PicShareConfig psc;

  /**
   * Basic constructor - called by the test runners.
   * TBD: Update constructor name to match class
   */
  public PicShareReaderUnitHelper(boolean cleanDisk) throws Exception{
    //PicShareConfig depends on Config being initialized
    Config.readConfig(PicShareReaderUnit.CONFIG_PATH);
    NodeId myId = new NodeId(PicShareConfig.LOWEST_WRITER_NODE_ID);
    psc = new PicShareConfig(PicShareReaderUnit.CONFIG_PICSHARE_PATH 
                             + PicShareConfig.LOWEST_WRITER_NODE_ID,
                             myId);

    if(verbose){
      System.out.println("HELPER ID: " + myId.toString());
    }
    try{
      psw = new PicShareWriter(PicShareReaderUnit.CONFIG_PATH, 
                               PicShareReaderUnit.CONFIG_P2_PATH, 
                               PicShareReaderUnit.CONFIG_PICSHARE_PATH
                               + PicShareConfig.LOWEST_WRITER_NODE_ID, 
                               myId, 
                               cleanDisk);
      li = psw.getLocalInterface();
    }
    catch(Exception e){
      e.printStackTrace();
      System.out.println("*** Did you kill the previous PicShareReaderUnitHelper? ***");
      throw e;
    }
  }

 /** 
 *  Write some stuff to IDs that reader will not subscribe to.
 
 *  Will let the reader test imprecise reads.
 
 **/ 
  private void writeOtherStuff(){
    ObjId oid = new ObjId(PicShareReaderUnit.impreciseTestPath 
                          + PicShareReaderUnit.impreciseTestWrittenObj);
    ObjId oid2 = new ObjId(PicShareReaderUnit.impreciseTestPath 
                          + PicShareReaderUnit.impreciseTestAnotherObj);
    byte b[] = new byte[1];
    b[0] = 42;
    try{
      if(verbose){
        System.out.println("Writing " + oid);
      }
      long got = li.write(oid, 0, b.length, b, false);
      Assert.affirm(got == b.length);
      got = li.write(oid2, 0, b.length, b, false);
      Assert.affirm(got == b.length);
    }
    catch(IOException e){
      System.out.println("IO Exception in PicShareReaderUnitHelper writeOtherStuff("
                         + oid.toString()
                         + ") -- "
                         + e.toString());
    }
    //
    // Since writes are unbound, need to ensure they are in checkpoint before
    // we die. The bodies will not be in the log (lets us test demand read,
    // prefetching.)
    //
    li.syncStateToDisk(); 
  }


 /** 
 *  Write specified list of objId->data tuples to PRACTI
 
 **/ 
  private void writeStuff(int pass){
    Hashtable stuffToWrite = makeStuffToWrite(psc, pass);
    Enumeration i = stuffToWrite.keys();
    while(i.hasMoreElements()){
      ObjId oid = (ObjId)i.nextElement();
      byte b[] = (byte[])stuffToWrite.get(oid);
      Assert.affirm(b != null);
      try{
        if(verbose){
          System.out.println("Writing " + oid);
        }
        long got = li.write(oid, 0, b.length, b, false);
        Assert.affirm(got == b.length);
      }
      catch(IOException e){
        System.out.println("IO Exception in PicShareReaderUnitHelper ("
                           + oid.toString()
                           + ") -- "
                           + e.toString());
      }
    }
    //
    // Since writes are unbound, need to ensure they are in checkpoint before
    // we die. The bodies will not be in the log (lets us test demand read,
    // prefetching.)
    //
    li.syncStateToDisk();
  }

 /** 
 *  Return a list of stuff we are supposed to write
 
 *  Hashtable is path->data pairs
 
 *  Used by PicShareReaderUnit to find out what it is supposed
 
 *  to read.
 
 **/ 
  public static Hashtable makeStuffToWrite(PicShareConfig psc, int pass){
    int max = 20;
    int ii, jj;
    int length = 50;

    Hashtable t = new Hashtable();
    for(ii = 0; ii < max; ii++){
      byte b[] = new byte[length];
      for(jj = 0; jj < length; jj++){
	b[jj] = (byte)(ii + jj);
      }
      String suffix = (new Integer(ii)).toString()
        + "-"
        + (new Integer(pass).toString());
      ObjId o = new ObjId(psc.basePRACTIDataPath() + suffix);
      t.put(o, b);
      o = new ObjId(psc.basePRACTIMetadataPath() + suffix);
      t.put(o, b);
      b[0] = (byte)255;
      o = new ObjId(psc.basePRACTIStatusPath() + suffix);
      t.put(o, b);
    }
    Assert.affirm(t.size() == 3 * max);
    return t;
  }

 /** 
 *  During our second lifetime, check to make sure the stuff we wrote
 
 *  duringo our first lifetime still exists. Throw obj not found exception
 
 *  if there is a problem.
 
 **/ 
  public void checkReadAfterReincarnation()
    throws ObjNotFoundException, IOException, 
	   ReadOfInvalidRangeException, ReadOfHoleException{
    Hashtable h = makeStuffToWrite(psc, 0);
    Env.dprintln(verbose, "checkReadAfterReincarnation 1");
    checkRead(h);
    h = makeStuffToWrite(psc, 1);
    Env.dprintln(verbose, "checkReadAfterReincarnation 2");
    checkRead(h);
    Env.dprintln(verbose, "checkReadAfterReincarnation 3");
  }

 /** 
 *  Check to make sure the stuff we wrote exists.
 
 **/ 
  public void checkRead(Hashtable t)
    throws ObjNotFoundException, IOException, 
	   ReadOfInvalidRangeException, ReadOfHoleException{
    int ii;
    Enumeration i = t.keys();
    int successCount = 0;
    while(i.hasMoreElements()){
      ObjId oid = (ObjId)i.nextElement();
      byte b[] = (byte[])t.get(oid);
      BodyMsg bm = null;
      Assert.affirm(b != null);
      if(vverbose){
        System.out.println("Checking " + oid.toString() + "...");
      }
      try{
        bm = li.read(oid, 0, b.length, false, true);
        assert(bm.getLength() == b.length);
        ImmutableBytes ib = bm.getBody();
        byte b2[] = ib.getCopyBytes();
        for(ii = 0; ii < b.length; ii++){
          Assert.affirm(b[(int)ii] == b2[(int)ii]);
        }
        successCount++;
      }
      catch(ObjNotFoundException onfe){
        System.out.println("Object not found: OID = " + oid
                           + " successes so far: " + successCount);
        throw onfe;
      }
      catch(IOException ie){
        System.out.println("IO Exception: OID = " + oid
                           + " successes so far: " + successCount);
        throw ie;
      }
      catch(ReadOfInvalidRangeException roire){
        System.out.println("Read invalid range exception: OID = " + oid
                           + " successes so far: " + successCount);
        throw roire;
      }
      catch(ReadOfHoleException rhe){
	System.out.println("Read of Hole: OID = " + oid
                           + " successes so far: " + successCount);
        throw rhe;
      }
    }
  }


 /** 
 *  Write the specified PRACTI files. Then wait until reading done. Then write
 
 *  more stuff. Then die. When restarted, read the old stuff and write some 
 
 *  different stuff.
 
 **/ 
  public static void main(String s[]) {
    if(s[0].equals("0")){
      doFirst();
    }
    else{
     doSecond();
    }
  }

 /** 
 *  First time around -- write a bunch of stuff.
 
 **/ 
  private static void doFirst(){
    BarrierClient c = new BarrierClient("127.0.0.1", PicShareReaderUnit.BARRIER_PORT, 1);
    PicShareReaderUnitHelper p = null;
    if(verbose){
      System.out.println("HELLO from PicShareReaderUnitHelper 0");
    }
    try{
      p = new PicShareReaderUnitHelper(true);
      if(verbose){
        System.out.println("PicShareReaderUnitHelper constructor done");
      }
      p.writeStuff(0);
    }
    catch(Exception e){
      e.printStackTrace();
    }

    if(verbose){
      System.out.println("barrier 1");
    }
    c.sendBarrierRequest(0, 1); // Tell ReaderUnit that we are up and running
    if(verbose){
      System.out.println("barrier 2");
    }
    c.sendBarrierRequest(0, 1); // Wait for ReaderUnit to be running, set up subscriptions, etc.

    //
    // Second pass -- write new stuff
    //
    try{
      p.writeOtherStuff();
      p.writeStuff(1);
    }
    catch(Exception e){
      e.printStackTrace();
    }


    if(verbose){
      System.out.println("barrier 3");
    }
    c.sendBarrierRequest(0, 1); // Wait for unit to tell me to exit
    if(verbose){
      System.out.println("PicShareReaderUnitHelper exiting");
    }

    //
    // This is expected to work after we recover... Double check before...xb
    //
    try{
      p.checkReadAfterReincarnation();
      if(verbose){
        System.out.println("checkread works before I die");
      }
    }
    catch(Exception ezdfr){
      System.out.println("Error checking before dying: " + ezdfr.toString());
      ezdfr.printStackTrace();
      System.exit(-1);
    }

    System.exit(0);
  }


 /** 
 *  Second time around -- check that the stuff we wrote the first time
 
 *  is still there and also write some new stuff.
 
 **/ 
  private static void doSecond(){
    BarrierClient c = new BarrierClient("127.0.0.1", 
                                        PicShareReaderUnit.BARRIER_PORT, 1);
    if(verbose){
      System.out.println("I'M BACK PicShareReaderUnitHelper-2");
    }
    try{
      PicShareReaderUnitHelper p = new PicShareReaderUnitHelper(false);
      Env.dprintln(verbose, "PicShareReaderUnitHelper-2 constructor done");
      p.checkReadAfterReincarnation();
      Env.dprintln(verbose, "Read Check Complete");
      p.writeStuff(3);
      Env.dprintln(verbose, "Writes Complete");
    }
    catch(Exception e){
      System.out.println("PicShareReaderUnitHelper exception: " 
                         + e.toString());
      e.printStackTrace();
    }
    Env.dprintln(verbose, "Before 2/first barrier");
    c.sendBarrierRequest(0, 1); // Tell ReaderUnit that we are up and running
    Env.dprintln(verbose, "After 2/first barrier");
    Env.dprintln(verbose, "Before 2/second barrier");
    c.sendBarrierRequest(0, 1); // Wait for ReaderUnit to be done
    Env.dprintln(verbose, "After 2/second barrier");

    if(verbose){
      System.out.println("PicShareReaderUnitHelper-2 exiting");
    }

    System.exit(0);
  }
}
