 /** 
/* PicShareWriteHistory.java
 * 
 * Since in v0.01 of picshare we're not using a real file system, we don't
 * have directories. So, we keep a list of files in PicShareWriteHistory.
 * This is used by the writer to list what is has written; readers
 * also use it to keep track fo stuff they've extracted from practi
 * and written to the local file system.
 *
 * Each entry is
 *   seq num
 *   relative file ID
 *   per-obj-meta
 * 
 * Serialize to disk as an array of such entries sorted by seqNum
 * (so that we can incrementally append new entries).
 * 
 * Stored in memory as a list indexed by seq num
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
 **/ 
import java.util.*;
import java.io.*;

public class PicShareWriteHistory{
  private ObjId persistent;
  private int nextSeqNum;
  private long offset;
  private ArrayList list;
  private boolean verbose = false;
  private LocalInterface li;
  private HashMap objIdIndex;

  private boolean dbg = false;

 /** 
 *  Constructor reads state from PRACTI
 
 **/ 
  PicShareWriteHistory(LocalInterface li, ObjId persistent){
    this.persistent = persistent;
    this.li = li;
    nextSeqNum = 0;
    list = new ArrayList();
    objIdIndex = new HashMap();
    
    byte b[] = readFile(persistent);
    if(b == null){
      return;
    }
    
    initializeTable(new ByteArrayInputStream(b));
  }

 /** 
 *  readBytes -- read entire object into a byte array
 
 **/ 
  private byte[] readFile(ObjId oid){
    Vector v = new Vector();
    BodyMsg got;
    offset = 0;
    int ii;

    try{
      while(true){
        got = li.read(oid, offset, Integer.MAX_VALUE, true, true);
        ImmutableBytes ib = got.getBody();
        byte bb[] = ib.dangerousGetReferenceToInternalByteArray();
        assert(bb.length > 0);
        Env.dprintln(dbg, "PSWH: read " + bb.length + " at " + offset);
        for(ii = 0; ii < bb.length; ii++){
          v.addElement(new Byte(bb[ii]));
        }
        offset += bb.length;
      }
    }
    catch(EOFException eof){
      // Exit while loop in expected way
    }
    catch(ObjNotFoundException onf){
      assert(offset == 0);
      return null;
    }
    catch(IOException onf){
      assert(offset == 0);
      return null;
    }
    catch(ReadOfInvalidRangeException roef){
      assert(false);
      return null;
    }
    catch(ReadOfHoleException roeh){
      assert(false);
      return null;
    }
    byte b[] = new byte[v.size()];
    for(ii = 0; ii < b.length; ii++){
      b[ii] = ((Byte)v.get(ii)).byteValue();
    }
    return b;
  }

 /** 
 *  InitializeTable -- read all elements out of byte array of elements
 
 *  and put them into in-memory table
 
 **/ 
  private void initializeTable(ByteArrayInputStream bais){
    int count = 0;
    try{
      while(true){
        //
        // Notice that we creat a new OOS each time around. This is
        // because each time we inserted an object, we did 
        // it with a new object output stream and we need to
        // be symmetrical
        //
        ObjectInputStream ois = new ObjectInputStream(bais);
        Object o = ois.readObject();
        PicShareWriteHistoryEntry pswhe = (PicShareWriteHistoryEntry)o;
        String id = pswhe.getRelativeFileId();
        list.add(pswhe.getSeqNum(), pswhe);
        objIdIndex.put(pswhe.getRelativeFileId(), pswhe);
        count++;
        if(pswhe.getSeqNum() >= nextSeqNum){
          nextSeqNum = pswhe.getSeqNum() + 1;
        }
      }
    }
    catch(IOException e){
      // This is EOF of byte array -- normal termination. Continue.
    }
    catch(ClassNotFoundException cnf){
      assert(false);
    }
    Env.dprintln(verbose, "PicShareWriteHistory got " + count + " items from storage");
  }

 /** 
 *  update -- append per-object meta to the practi state and add it
 
 *            to in-memory list
 
 **/ 
  public synchronized void update(PicSharePerObjMeta meta, String relativeId){
    PicShareWriteHistoryEntry pswhe = new PicShareWriteHistoryEntry(nextSeqNum,
                                                                    relativeId,
                                                                    meta);
    list.add(nextSeqNum, pswhe);
    objIdIndex.put(relativeId, pswhe);
    nextSeqNum++;
    assert(nextSeqNum < Integer.MAX_VALUE); // Guard against wrap
    try{
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      ObjectOutputStream oos = new ObjectOutputStream(baos);
      oos.writeObject(pswhe);
      oos.close();
      byte b[] = baos.toByteArray();
      li.write(persistent, offset, b.length, new ImmutableBytes(b),
               true);
      Env.dprintln(dbg, "PSWH: write " + b.length + " at " + offset);
      offset += b.length;
    }
    catch(IOException e){
      e.printStackTrace();
      assert(false);
    }
  }

 /** 
 *  update -- append per-object meta to the practi state and add it
 
 *            to in-memory list. Called with an expected sequence
 
 *            number -- we expect writer and readers' histories to 
 
 *            be identical
 
 **/ 
  public synchronized void update(PicSharePerObjMeta meta, String relativeId,
                                  int expectedSeqNum){
    assert(nextSeqNum == expectedSeqNum);
    update(meta, relativeId);
    assert(nextSeqNum == expectedSeqNum + 1);
  }

 /** 
 *  equals -- used for testing
 
 *  NOT SYNCHRONIZED (since we would need to lock both this and wh)
 
 **/ 
  public boolean equals(Object o){
    if(!(o instanceof PicShareWriteHistory)){
      return false;
    }
    if(o == this){
      return true;
    }

    PicShareWriteHistory wh = (PicShareWriteHistory)o;
    if(this.nextSeqNum != wh.nextSeqNum){
      return false;
    }
    if(this.offset != wh.offset){
      return false;
    }
    if(this.list.size() != wh.list.size()){
      return false;
    }
    int ii;
    for(ii = 0; ii < list.size(); ii++){
      PicShareWriteHistoryEntry e1 = (PicShareWriteHistoryEntry)this.list.get(ii);
      assert(e1 != null);
      PicShareWriteHistoryEntry e2 = (PicShareWriteHistoryEntry)wh.list.get(ii);
      if(e2 == null){
        return false;
      }
      if(!e1.equals(e2)){
        return false;
      }
    }
    return true;
  }

 /** 
 *  isEmpty -- return true if no writes in this history
 
 **/ 
  public boolean isEmpty(){
    if(list.size() == 0){
      assert(nextSeqNum == 0);
      return true;
    }
    else{
      assert(nextSeqNum > 0);
      return false;
    }
  }


 /** 
 *  nextSeq
 
 **/ 
  public int getNextSeq(){
    return nextSeqNum;
  }

 /** 
 *  getWriteMeta()
 
 **/ 
  public PicShareWriteHistoryEntry getWriteEntry(int seqNum){
    assert(seqNum >= 0);
    return (PicShareWriteHistoryEntry)list.get(seqNum);
  }

 /** 
 *  isPresent() -- return true if exact match found
 
 **/ 
  public boolean isPresent(String relativeFileId, long time, long length,
                           ImmutableBytes hash)
  {
    PicShareWriteHistoryEntry e = (PicShareWriteHistoryEntry)objIdIndex.get(relativeFileId);
    if(e == null){
      return false;
    }

    PicSharePerObjMeta meta = e.getObjMeta();
    
    if(time != meta.getTimeAddedMS()){
      return false;
    }

    if(length != meta.getBodyLength()){
      return false;
    }

    if(!hash.equals(meta.getHash())){
      return false;
    }

    return true;
  }


}










