 /** 
/* PicShareWriter.java
 * 
 * Program that runs at the writer side of PicShare picture sharing
 * application. The writer side mainly consists of a localinterface
 * to which writes are sent. (Readers handle setting up subscriptions, etc.)
 * 
 * (C) Copyright 2006 -- See the file COPYRIGHT for additional details
 */
 **/ 
import java.io.*;

public class PicShareWriter{

  private Policy policy;
  private P2Runtime runtime;
  private PicShareConfig picShareConfig;
  private LocalInterface localInterface;
  private PicShareWriteHistory writeHistory;
  private static boolean verbose = true;

  //
  // How many files in a row should fail for us to abandon
  // the scan (idea is to distinguish per-file from global
  // problems...
  //
  private int ioExceptionCount = 0;
  private int maxIOExceptions = 3; 

  private File baseFile;
  
 /** 
 *  Command line
 
 **/ 
  public static void main(String[] args){
    
    if(args.length < 2){
      System.out.println("Usage: PicShareWriter "
			 + "<configDirectory> <myNodeId>");
      System.exit(-1);
    }
    String configBase = args[0];
    String PRACTIconfigPath = configBase + File.separatorChar 
      + "practi_picshare_config";
    String p2ConfigPath = configBase + File.separatorChar 
      + "p2_picshare_config";
    String picShareConfigPath = configBase + File.separatorChar 
      + "picshare_picshare_config";
    NodeId myId = new NodeId((new Integer(args[1])).intValue());
    PicShareWriter psw = new PicShareWriter(PRACTIconfigPath, 
					    p2ConfigPath, 
                                            picShareConfigPath,
                                            myId, false);

    psw.loop();
  }


 /** 
 *  Constructor
 
 **/ 
  public PicShareWriter(String PRACTIconfigPath, 
			String p2ConfigPath, 
			String picShareConfigPath,
			NodeId myId, 
			boolean clearDB){
    picShareConfig = new PicShareConfig(picShareConfigPath, myId);
    policy = new NullPolicy();
    runtime = new P2Runtime(PRACTIconfigPath, p2ConfigPath, 
			    myId, clearDB, policy, false);
    policy.setRuntime(runtime);
    runtime.start();
    localInterface = new LocalInterface(runtime.getController(), 
					runtime.getCore());
    writeHistory = new PicShareWriteHistory(localInterface,
                                            picShareConfig.getWriteHistoryId(myId));
    baseFile = picShareConfig.baseLocalDataPath();
    

  }

 /** 
 *  Shutdown -- release ports; try to kill most threads...
 
 **/ 
  public void shutdown(){
    if(runtime != null){
      runtime.shutdown();
    }
    runtime = null;
  }


 /** 
 *  getLocalInterface
 
 **/ 
  public LocalInterface getLocalInterface(){
    return localInterface;
  }

 /** 
 *  loop -- scan all files and send any new ones. Repeat periodically.
 
 **/ 
  private void loop(){
    long lengthToRead;
    if(picShareConfig.verifyBody == true){
      lengthToRead = Long.MAX_VALUE;
    }
    else{
      lengthToRead = 1;
    }
    while(true){
      try{
        ioExceptionCount = 0;
        scan("", lengthToRead);
      }
      catch(IOException e){
        ; // Try again later...
      }
      try{
        Thread.sleep(picShareConfig.getWriterScanFrequencyMS());
      }
      catch(InterruptedException e){
      }
    }
  }


 /** 
 *  scan() -- recursively scan files from fileId and send any new ones
 
 **/ 
  private void scan(String fileId, // Relative path from config.baseLocalDataPath()
                    long lengthToRead)
  throws IOException{ // How much data to scan and compare v. practi
    int ii;
    if(verbose){
      System.out.println("PicShareWriter: Writer is starting scan of "
                         + picShareConfig.baseLocalDataPath() 
                         + "/" + fileId);
    }
    File f = new File(baseFile, fileId);
    if(verbose){
      System.out.println("PicShareWriter: file abs name is " 
                         + f.getAbsoluteFile());
    }
    assert(f.exists());
    if(f.isDirectory()){
      if(verbose){
        System.out.println("PicShareWriter: current file is a directory");
      }
      File children[] = f.listFiles();
      String baseFileString = baseFile.getPath();
      for(ii = 0; ii < children.length; ii++){
        String fullPath = children[ii].getPath();
        if(verbose){
          System.out.println("PicShareWriter: child full path:" + fullPath);
        }
        assert(fullPath.startsWith(baseFileString));
        String relFileId = fullPath.substring(baseFileString.length());
        assert(relFileId.startsWith(fileId));
        scan(relFileId, lengthToRead);
      }
    }
    else{
      assert(f.isFile());
      try{
        if(verbose){
          System.out.println("PicShareWriter: current file is a regular file");
        }
        doFile(fileId, lengthToRead);
        if(ioExceptionCount > 0){
          //
          // We made progress. Probably a per-file issue?
          //
          ioExceptionCount--;
        }
      }
      catch(IOException e){
        //
        // Skip this file (since it did not get logged as
        // done, we will retry on next pass. If we get 
        // a bunch of IOExceptions in a row, then abandon
        // ship.
        //
        ioExceptionCount++;
        Env.warn("IO Error writing " + fileId + " to PRACTI. Will retry next pass."
                 + e.toString());
        if(ioExceptionCount > maxIOExceptions){ // Too many in a row. Give up.
          Env.warn("Too many IO Errors. Giving up. Better luck next pass.");
          throw e;
        }
      }
    }
  }


 /** 
 *  doFile -- check to see if this file is already safely in PRACTI. If not
 
 *  read it and insert it. FileId is relative to base path.
 
 **/ 
  private void doFile(String fileId, long lengthToRead)
  throws IOException{
    Env.dprintln(verbose, "PicShareWriter doFile(" + fileId + ")");

    byte encryptedBody[] = getEncryptedWriteNeeded(fileId, lengthToRead);
    if(encryptedBody == null){
      if(verbose){
        System.out.println("PicShareWriter: file " + fileId
                           + " is current. No update.");
      }
      return;
    }

    if(verbose){
      System.out.println("PicShareWriter: file " + fileId
                         + " needs update.");
    }

    //
    // First, update (encrypted) data. If there is a later error,
    // this write will be repeated next pass, no harm done.
    //
    ObjId practiObjId = picShareConfig.objIdPRACTIDataPath(fileId);
    long got = localInterface.write(practiObjId, 0, encryptedBody.length, 
                                    encryptedBody, false);
    assert(got == encryptedBody.length);

    //
    // Next write the metadata. If there is a later error,
    // this write will be repeated next pass, no harm done.
    //
    ObjId practiMetaObjId = picShareConfig.objIdPRACTIMetadataPath(fileId);
    PicSharePerObjMeta perObjMeta = new PicSharePerObjMeta(fileId, 
                                                           encryptedBody);
    byte[] perObjMetaBytes = perObjMeta.toBytes();
    got = localInterface.write(practiMetaObjId, 0, perObjMetaBytes.length,
                               perObjMetaBytes, false);
    assert(got == perObjMetaBytes.length);

    //
    // Finally write the history. This is the "commit"
    // for these three writes. 
    //
    writeHistory.update(perObjMeta, fileId);
    
  }

 /** 
 *  Decide if we need to do a write of this file. Since some tests
 
 *  touch the bytes of the input file, take this chance to encrypt
 
 *  it and return it to caller. Return null if no write needed.
 
 *  Throw IOException if input file is not readable or changes while 
 
 *  we're looking at it.
 
 * 
 
 *  FileId is relative to base path.  
 
 **/ 
  private byte[] getEncryptedWriteNeeded(String fileId, long lengthToRead)
  throws IOException{

    byte fileBytes[] = getWriteNeeded(fileId, lengthToRead);

    if(fileBytes == null){
      return null;
    }

    byte encryptedBytes[] = encrypt(fileBytes);
    return encryptedBytes;
  }


 /** 
 *  Decide if we need to do a write of this file. Since some tests
 
 *  touch the bytes of the input file, return the plaintext bytes
 
 *  to be written. Return null if no write needed. Throw IOException
 
 *  if input file is not readable or changes while we're looking at it.
 
 * 
 
 *  FileId is relative to base path.  
 
 **/ 
  private byte[] getWriteNeeded(String fileId, long lengthToTest)
  throws IOException{
    boolean writeIsNeeded = false;


    File f = new File(baseFile, fileId);

    if(!f.exists()){
      if(verbose){
        System.out.println("PicShareWriter::getWriteNeeded() -- file "
                           + fileId + " not present. This should not happen.");
      }
      throw new IOException("Missing file " + fileId);
    }

    PicSharePerObjMeta meta = getPractiMeta(fileId);


    if((meta == null && f.length() > 0) // Only create new file if non-empty
       ||
       meta.getTimeAddedMS() < f.lastModified()
       ||
       meta.getBodyLength() != f.length()
       // Check for case where writeHistory on disk is corrupted
       // (or we wrote the data but did not succeed in writing the
       // update list)  and need to 
       // "resend" file so that all readers know to read it
       || 
       !writeHistory.isPresent(fileId, meta.getTimeAddedMS(), meta.getBodyLength(),
                               meta.getHash())){
      writeIsNeeded = true;
    }
       
    



    long length = f.length();
    assert(length <= Integer.MAX_VALUE); // We only support files up to 2**31 bytes
    byte bfile[] = new byte[(int)length];
    
    lengthToTest = (lengthToTest < length ? lengthToTest : length); // min

    //
    // Read the first lengthToTest of the file
    //
    FileInputStream fis = new FileInputStream(f);
    fis.read(bfile, 0, (int)lengthToTest);

    if(!writeIsNeeded){
      //
      // Try to read obj from practi and compare.
      //
      try{
        byte bpracti[] = practiReadFull(fileId, (int)lengthToTest);
        int ii;
        assert(bpracti.length == lengthToTest);
        for(ii = 0; ii < bpracti.length; ii++){
          if(bpracti[ii] != bfile[ii]){
            writeIsNeeded = true;
            break;
          }
        }
      }
      catch(Exception e){
        writeIsNeeded = true;
      }
    }

    if(!writeIsNeeded){
      return null;
    }

    //
    // Read the rest of file if needed
    //
    if(lengthToTest < length){
      fis.read(bfile, (int)lengthToTest, (int)(length - lengthToTest));
    }

    return bfile;
    
    
  }

 /** 
 *  getPractiMeta()
 
 **/ 
  private PicSharePerObjMeta getPractiMeta(String fileId)
  {
    try{
      ObjId practiMetaObjId = picShareConfig.objIdPRACTIMetadataPath(fileId);
      //
      // Assumes single read is enough since meta always written in one write
      //
      BodyMsg pm = localInterface.read(practiMetaObjId, 0, Integer.MAX_VALUE,
                                       false, false);
      ImmutableBytes ib = pm.getBody();
      byte b[] = ib.dangerousGetReferenceToInternalByteArray();
      ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(b));
      PicSharePerObjMeta meta = (PicSharePerObjMeta)ois.readObject();
      return meta;
    }
    catch(Exception e){
      return null;
    }
  }


 /** 
 *  practiReadFull -- attempt to read the full length specified. Throw
 
 *  exception if we get less than that for any reason
 
 **/ 
  private byte [] practiReadFull(String fileId, int lengthToTest)
    throws ObjNotFoundException, IOException,
    EOFException, ReadOfInvalidRangeException, ReadOfHoleException{
    
    ObjId practiObjId = picShareConfig.objIdPRACTIDataPath(fileId);
    
    byte b[] = new byte[lengthToTest];
    
    int gotSoFar = 0;
    while(gotSoFar < lengthToTest){
      int got = practiReadMore(b, practiObjId, gotSoFar, lengthToTest);
      gotSoFar += got;
    }
    return b;
  }


 /** 
 *  practiReadMore -- attempt to read the practi obj from offset specified. 
 
 *  Throw exception if we fail to get more bytes. Return number of new bytes.
 
 **/ 
  private int practiReadMore(byte b[], ObjId oid, int offset, int lengthToTest)
    throws ObjNotFoundException, IOException,
    EOFException, ReadOfInvalidRangeException, ReadOfHoleException{
    
    BodyMsg body = localInterface.read(oid, offset, lengthToTest - offset,
                                       false, false);
    long got = body.getLength();
    assert(got <= Integer.MAX_VALUE);
    ImmutableBytes data = body.getBody();
    byte newBytes[] = data.dangerousGetReferenceToInternalByteArray();
    int ii;
    for(ii = 0; ii < got; ii++){
      b[offset + ii] = newBytes[ii];
    }
    return (int)got;
  }
  

 /** 
 *  encrypt -- STUB
 
 **/ 
  private byte[] encrypt(byte b[])
  {
    System.out.println("STUB: PicShareWriter encrypt");
    return b;
  }


}
